/*
 * Copyright (C) 2016-2019 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "SVGParserUtilities.h"
#include "SVGPrimitiveList.h"
#include <wtf/text/StringBuilder.h>

namespace WebCore {

class SVGStringList final : public SVGPrimitiveList<String> {
    using Base = SVGPrimitiveList<String>;
    using Base::Base;
    using Base::m_items;

public:
    static Ref<SVGStringList> create(SVGPropertyOwner* owner)
    {
        return adoptRef(*new SVGStringList(owner));
    }

    void reset(const String& string)
    {
        parse(string, ' ');

        // Add empty string, if list is empty.
        if (m_items.isEmpty())
            m_items.append(emptyString());
    }

    bool parse(const String& data, UChar delimiter)
    {
        clearItems();

        auto upconvertedCharacters = StringView(data).upconvertedCharacters();
        const UChar* ptr = upconvertedCharacters;
        const UChar* end = ptr + data.length();
        while (ptr < end) {
            const UChar* start = ptr;
            while (ptr < end && *ptr != delimiter && !isSVGSpace(*ptr))
                ptr++;
            if (ptr == start)
                break;
            m_items.append(String(start, ptr - start));
            skipOptionalSVGSpacesOrDelimiter(ptr, end, delimiter);
        }

        return ptr == end;
    }

    String valueAsString() const override
    {
        StringBuilder builder;

        for (const auto& string : m_items) {
            if (builder.length())
                builder.append(' ');

            builder.append(string);
        }

        return builder.toString();
    }
};

} // namespace WebCore
