#!/bin/bash
#
# Copyright (c) 2012, 2021, Oracle and/or its affiliates. All rights reserved.
# DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
#
# This code is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 2 only, as
# published by the Free Software Foundation.
#
# This code is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# version 2 for more details (a copy is included in the LICENSE file that
# accompanied this code).
#
# You should have received a copy of the GNU General Public License version
# 2 along with this work; if not, write to the Free Software Foundation,
# Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
# or visit www.oracle.com if you need additional information or have any
# questions.
#

# This script is processed by configure before it's usable. It is run from
# the root of the build directory.


################################################################################

# Check that we are run via the wrapper generated by configure
if [ -z "$TOPDIR" ]; then
    echo "Error: You must run this script using build/[conf]/compare.sh"
    exit 1
fi

# Make sure all shell commands are executed with the C locale
export LC_ALL=C

if [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
    FULLDUMP_CMD="$OTOOL -v -V -h -X -d"
    LDD_CMD="$OTOOL -L"
    DIS_CMD="$OTOOL -v -V -t"
    STAT_PRINT_SIZE="-f %z"
elif [ "$OPENJDK_TARGET_OS" = "windows" ]; then
    FULLDUMP_CMD="$DUMPBIN -all"
    LDD_CMD="$DUMPBIN -dependents"
    DIS_CMD="$DUMPBIN -disasm:nobytes"
    STAT_PRINT_SIZE="-c %s"
elif [ "$OPENJDK_TARGET_OS" = "aix" ]; then
    FULLDUMP_CMD="dump -h -r -t -n -X64"
    LDD_CMD="$LDD"
    DIS_CMD="$OBJDUMP -d"
    STAT_PRINT_SIZE="-c %s"
else
    FULLDUMP_CMD="$READELF -a"
    LDD_CMD="$LDD"
    DIS_CMD="$OBJDUMP -d"
    STAT_PRINT_SIZE="-c %s"
fi

COMPARE_EXCEPTIONS_INCLUDE="$TOPDIR/make/scripts/compare_exceptions.sh.incl"
if [ ! -e "$COMPARE_EXCEPTIONS_INCLUDE" ]; then
    echo "Error: Cannot locate the exceptions file, it should have been here: $COMPARE_EXCEPTIONS_INCLUDE"
    exit 1
fi
# Include exception definitions
. "$COMPARE_EXCEPTIONS_INCLUDE"

################################################################################
#
# Disassembly diff filters. These filters try to filter out ephemeral parts of the
# disassembly, such as hard-coded addresses, to be able to catch "actual" differences.

if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
  if [ "$OPENJDK_TARGET_CPU" = "x86" ]; then
    DIS_DIFF_FILTER="$SED -r \
        -e 's/^  [0-9A-F]{16}: //' \
        -e 's/^  [0-9A-F]{8}: /  <ADDR>: /' \
        -e 's/(offset \?\?)_C@_.*/\1<SYM>/' \
        -e 's/[@?][A-Za-z0-9_]{1,25}/<SYM>/' \
        -e 's/([-,+])[0-9A-F]{2,16}/\1<HEXSTR>/g' \
        -e 's/\[[0-9A-F]{4,16}h\]/[<HEXSTR>]/' \
        -e 's/: ([a-z]{2}[a-z ]{2})        [0-9A-F]{2,16}h?$/: \1        <HEXSTR>/' \
        -e 's/_20[0-9]{2}_[0-1][0-9]_[0-9]{2}/_<DATE>/' \
        "
  elif [ "$OPENJDK_TARGET_CPU" = "x86_64" ]; then
    DIS_DIFF_FILTER="$SED -r \
        -e 's/^  [0-9A-F]{16}: //' \
        -e 's/\[[0-9A-F]{4,16}h\]/[<HEXSTR>]/' \
        -e 's/([,+])[0-9A-F]{2,16}h/\1<HEXSTR>/' \
        -e 's/([a-z]{2}[a-z ]{2})        [0-9A-F]{4,16}$/\1        <HEXSTR>/' \
        -e 's/\[\?\?_C@_.*/[<SYM>]/' \
        "
  fi
elif [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
  DIS_DIFF_FILTER="$SED \
      -e 's/0x[0-9a-f]\{3,16\}/<HEXSTR>/g' -e 's/^[0-9a-f]\{12,20\}/<ADDR>/' \
      -e 's/-20[0-9][0-9]-[0-1][0-9]-[0-3][0-9]-[0-2][0-9]\{5\}/<DATE>/g' \
      -e 's/), built on .*/), <DATE>/' \
      "
fi

################################################################################
# Compare text files and ignore specific differences:
#
#  * Timestamps in Java sources generated by idl2java
#  * Sorting order and cleanup style in .properties files

diff_text() {
    OTHER_FILE=$1
    THIS_FILE=$2

    SUFFIX="${THIS_FILE##*.}"
    NAME="${THIS_FILE##*/}"

    TMP=$($DIFF $THIS_FILE $OTHER_FILE)

    if test "x$SUFFIX" = "xclass"; then
        if [ "$NAME" = "SystemModules\$all.class" ] \
           || [ "$NAME" = "SystemModules\$default.class" ]; then
            # The SystemModules\$*.classes are not comparable as they contain the
            # module hashes which would require a whole other level of
            # reproducible builds to get reproducible. There is also random
            # order of map initialization.
            TMP=""
        elif [ "$NAME" = "module-info.class" ]; then
            # The module-info.class have several issues with random ordering of
            # elements in HashSets.
            MODULES_CLASS_FILTER="$SED \
                -e 's/,$//' \
                -e 's/;$//' \
                -e 's/^ *[0-9]*://' \
                -e 's/#[0-9]* */#/' \
                -e 's/ *\/\// \/\//' \
                -e 's/aload *[0-9]*/aload X/' \
                -e 's/ldc_w/ldc  /' \
                | $SORT \
                "
            $JAVAP -c -constants -l -p "${OTHER_FILE}" \
                | eval "$MODULES_CLASS_FILTER" >  ${OTHER_FILE}.javap &
            $JAVAP -c -constants -l -p "${THIS_FILE}" \
                | eval "$MODULES_CLASS_FILTER" > ${THIS_FILE}.javap &
            wait
            TMP=$($DIFF ${OTHER_FILE}.javap ${THIS_FILE}.javap)
        fi
    fi

    if test -n "$TMP"; then
        echo Files $OTHER_FILE and $THIS_FILE differ
        return 1
    fi

    return 0
}

################################################################################
# Compare directory structure

compare_dirs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    mkdir -p $WORK_DIR

    (cd $OTHER_DIR && $FIND . -type d | $SORT > $WORK_DIR/dirs_other)
    (cd $THIS_DIR && $FIND . -type d | $SORT > $WORK_DIR/dirs_this)

    $DIFF $WORK_DIR/dirs_other $WORK_DIR/dirs_this > $WORK_DIR/dirs_diff

    echo -n Directory structure...
    if [ -s $WORK_DIR/dirs_diff ]; then
        echo Differences found.
        REGRESSIONS=true
        # Differences in directories found.
        ONLY_OTHER=$($GREP '<' $WORK_DIR/dirs_diff)
        if [ "$ONLY_OTHER" ]; then
            echo Only in $OTHER
            $GREP '<' $WORK_DIR/dirs_diff | $SED 's|< ./|    |g'
        fi
        ONLY_THIS=$($GREP '>' $WORK_DIR/dirs_diff)
        if [ "$ONLY_THIS" ]; then
            echo Only in $THIS
            $GREP '>' $WORK_DIR/dirs_diff | $SED 's|> ./|    |g'
        fi
    else
        echo Identical!
    fi
}


################################################################################
# Compare file structure

compare_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    $MKDIR -p $WORK_DIR

    (cd $OTHER_DIR && $FIND . ! -type d | $SORT > $WORK_DIR/files_other)
    (cd $THIS_DIR && $FIND . ! -type d | $SORT > $WORK_DIR/files_this)

    $DIFF $WORK_DIR/files_other $WORK_DIR/files_this > $WORK_DIR/files_diff

    echo -n File names...
    if [ -s $WORK_DIR/files_diff ]; then
        echo Differences found.
        REGRESSIONS=true
        # Differences in files found.
        ONLY_OTHER=$($GREP '<' $WORK_DIR/files_diff)
        if [ "$ONLY_OTHER" ]; then
            echo Only in $OTHER
            $GREP '<' $WORK_DIR/files_diff | $SED 's|< ./|    |g'
        fi
        ONLY_THIS=$($GREP '>' $WORK_DIR/files_diff)
        if [ "$ONLY_THIS" ]; then
            echo Only in $THIS
            $GREP '>' $WORK_DIR/files_diff | $SED 's|> ./|    |g'
        fi
    else
        echo Identical!
    fi
}


################################################################################
# Compare permissions

compare_permissions() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    mkdir -p $WORK_DIR

    echo -n Permissions...
    found=""
    for f in `cd $OTHER_DIR && $FIND . -type f`
    do
        if [ ! -f ${OTHER_DIR}/$f ]; then continue; fi
        if [ ! -f ${THIS_DIR}/$f ]; then continue; fi
        OP=`ls -l ${OTHER_DIR}/$f | awk '{printf("%.10s\n", $1);}'`
        TP=`ls -l ${THIS_DIR}/$f | awk '{printf("%.10s\n", $1);}'`
        if [ "$OP" != "$TP" ]
        then
            if [ -z "$found" ]; then echo ; found="yes"; fi
            $PRINTF "\tother: ${OP} this: ${TP}\t$f\n"
        fi
    done
    if [ -z "$found" ]; then
        echo "Identical!"
    else
        REGRESSIONS=true
    fi
}

################################################################################
# Compare file command output

compare_file_types() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    $MKDIR -p $WORK_DIR

    FILE_TYPES_FILTER="$SED \
        -e 's/BuildID[^,]*//' \
        -e 's/last modified: .*//' \
        "

    echo -n File types...
    found=""
    # The file command does not know about jmod files and this sometimes results
    # in different types being detected more or less randomly.
    for f in $(cd $OTHER_DIR && $FIND . ! -type d -a ! -name "*.jmod")
    do
        if [ ! -f ${OTHER_DIR}/$f ]; then continue; fi
        if [ ! -f ${THIS_DIR}/$f ]; then continue; fi
        OF=$(cd ${OTHER_DIR} && $FILE -h $f | eval $FILE_TYPES_FILTER)
        TF=$(cd ${THIS_DIR} && $FILE -h $f | eval $FILE_TYPES_FILTER)
        if [ "$OF" != "$TF" ]
        then
            if [ "`echo $OF | $GREP -c 'Zip archive data'`" -gt 0 ] \
                && [ "`echo $TF | $GREP -c 'Zip archive data'`" -gt 0 ]
            then
                # the way we produce zip-files make it so that directories are stored in
                # old file but not in new (only files with full-path) this makes file
                # report them as different
                continue
            elif [ "`echo $OF | $GREP -c 'MSVC program database ver 7.00'`" -gt 0 ] \
                     && [ "`echo $TF | $GREP -c 'MSVC program database ver 7.00'`" -gt 0 ]
            then
                # For Windows pdb files the file command reports some kind of size data
                # which may sometimes come out randomly different.
                continue
            else
                if [ -z "$found" ]; then echo ; found="yes"; fi
                $PRINTF "\tother: ${OF}\n\tthis : ${TF}\n"
            fi
        fi
    done
    if [ -z "$found" ]; then
        echo "Identical!"
    else
        REGRESSIONS=true
    fi
}

################################################################################
# Compare the rest of the files

compare_general_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    GENERAL_FILES=$(cd $THIS_DIR && $FIND . -type f ! -name "*.so" ! -name "*.jar" \
        ! -name "*.zip" ! -name "*.debuginfo" ! -name "*.dylib" ! -name "jexec" \
        ! -name "modules" ! -name "ct.sym" ! -name "*.diz" ! -name "*.dll" \
        ! -name "*.cpl" ! -name "*.pdb" ! -name "*.exp" ! -name "*.ilk" \
        ! -name "*.lib" ! -name "*.jmod" ! -name "*.exe" \
        ! -name "*.obj" ! -name "*.o" ! -name "jspawnhelper" ! -name "*.a" \
        ! -name "*.tar.gz" ! -name "*.jsa" ! -name "gtestLauncher" \
        ! -name "*.map" \
        | $GREP -v "./bin/"  | $SORT | $FILTER)

    echo Other files with binary differences...
    for f in $GENERAL_FILES
    do
        # Skip all files in test/*/native
        if [[ "$f" == */native/* ]]; then
            continue
        fi
        if [ -e $OTHER_DIR/$f ]; then
            SUFFIX="${f##*.}"
            if [ "$(basename $f)" = "release" ]; then
                # In release file, ignore differences in source rev numbers
                OTHER_FILE=$WORK_DIR/$f.other
                THIS_FILE=$WORK_DIR/$f.this
                $MKDIR -p $(dirname $OTHER_FILE)
                $MKDIR -p $(dirname $THIS_FILE)
                RELEASE_FILTER="$SED -e 's/SOURCE=".*"/SOURCE=<src-rev>/g'"
                $CAT $OTHER_DIR/$f | eval "$RELEASE_FILTER" > $OTHER_FILE
                $CAT $THIS_DIR/$f  | eval "$RELEASE_FILTER" > $THIS_FILE
            elif [ "$SUFFIX" = "svg" ]; then
                # GraphViz has non-determinism when generating svg files
                OTHER_FILE=$WORK_DIR/$f.other
                THIS_FILE=$WORK_DIR/$f.this
                $MKDIR -p $(dirname $OTHER_FILE) $(dirname $THIS_FILE)
                SVG_FILTER="$SED \
                    -e 's/edge[0-9][0-9]*/edgeX/g'
                    "
                $CAT $OTHER_DIR/$f | eval "$SVG_FILTER" > $OTHER_FILE
                $CAT $THIS_DIR/$f | eval "$SVG_FILTER" > $THIS_FILE
            elif [ "$SUFFIX" = "jar_contents" ]; then
                # The jar_contents files may have some lines in random order
                OTHER_FILE=$WORK_DIR/$f.other
                THIS_FILE=$WORK_DIR/$f.this
                $MKDIR -p $(dirname $OTHER_FILE) $(dirname $THIS_FILE)
                $RM $OTHER_FILE $THIS_FILE
                $CAT $OTHER_DIR/$f | $SORT > $OTHER_FILE
                $CAT $THIS_DIR/$f  | $SORT > $THIS_FILE
            else
                OTHER_FILE=$OTHER_DIR/$f
                THIS_FILE=$THIS_DIR/$f
            fi
            DIFF_OUT=$($DIFF $OTHER_FILE $THIS_FILE 2>&1)
            if [ -n "$DIFF_OUT" ]; then
                echo $f
                REGRESSIONS=true
                if [ "$SHOW_DIFFS" = "true" ]; then
                    echo "$DIFF_OUT"
                fi
            fi
        fi
    done


}

################################################################################
# Compare zip file

compare_zip_file() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    ZIP_FILE=$4
    # Optionally provide different name for other zipfile
    OTHER_ZIP_FILE=$5

    THIS_ZIP=$THIS_DIR/$ZIP_FILE
    if [ -n "$OTHER_ZIP_FILE" ]; then
        OTHER_ZIP=$OTHER_DIR/$OTHER_ZIP_FILE
    else
        OTHER_ZIP=$OTHER_DIR/$ZIP_FILE
    fi

    THIS_SUFFIX="${THIS_ZIP##*.}"
    OTHER_SUFFIX="${OTHER_ZIP##*.}"
    if [ "$THIS_SUFFIX" != "$OTHER_SUFFIX" ]; then
        echo "The files do not have the same suffix type! ($THIS_SUFFIX != $OTHER_SUFFIX)"
        return 2
    fi

    TYPE="$THIS_SUFFIX"

    if $CMP $OTHER_ZIP $THIS_ZIP > /dev/null
    then
        return 0
    fi
    # Not quite identical, the might still contain the same data.
    # Unpack the jar/zip files in temp dirs

    THIS_UNZIPDIR=$WORK_DIR/$ZIP_FILE.this
    OTHER_UNZIPDIR=$WORK_DIR/$ZIP_FILE.other
    $RM -rf $THIS_UNZIPDIR $OTHER_UNZIPDIR
    $MKDIR -p $THIS_UNZIPDIR
    $MKDIR -p $OTHER_UNZIPDIR
    if [ "$TYPE" = "jar" -o "$TYPE" = "war" -o "$TYPE" = "zip" ]
    then
        (cd $THIS_UNZIPDIR && $UNARCHIVE $THIS_ZIP)
        (cd $OTHER_UNZIPDIR && $UNARCHIVE $OTHER_ZIP)
    elif [ "$TYPE" = "gz" ]
    then
        (cd $THIS_UNZIPDIR && $GUNZIP -c $THIS_ZIP | $TAR xf -)
        (cd $OTHER_UNZIPDIR && $GUNZIP -c $OTHER_ZIP | $TAR xf -)
    elif [ "$TYPE" = "jmod" ]
    then
        (cd $THIS_UNZIPDIR && $JMOD extract $THIS_ZIP)
        (cd $OTHER_UNZIPDIR && $JMOD extract $OTHER_ZIP)
    else
        (cd $THIS_UNZIPDIR && $JIMAGE extract $THIS_ZIP)
        (cd $OTHER_UNZIPDIR && $JIMAGE extract $OTHER_ZIP)
    fi

    CONTENTS_DIFF_FILE=$WORK_DIR/$ZIP_FILE.diff
    $DIFF -rq $OTHER_UNZIPDIR $THIS_UNZIPDIR > $CONTENTS_DIFF_FILE

    ONLY_OTHER=$($GREP "^Only in $OTHER_UNZIPDIR" $CONTENTS_DIFF_FILE)
    ONLY_THIS=$($GREP "^Only in $THIS_UNZIPDIR" $CONTENTS_DIFF_FILE)

    return_value=0

    if [ -n "$ONLY_OTHER" ]; then
        echo "        Only OTHER $ZIP_FILE contains:"
        echo "$ONLY_OTHER" | sed "s|Only in $OTHER_UNZIPDIR|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    if [ -n "$ONLY_THIS" ]; then
        echo "        Only THIS $ZIP_FILE contains:"
        echo "$ONLY_THIS" | sed "s|Only in $THIS_UNZIPDIR|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    if [ "$CMP_ZIPS_CONTENTS" = "true" ]; then
        DIFFING_FILES=$($GREP -e "differ$" $CONTENTS_DIFF_FILE \
            | $CUT -f 2 -d ' ' | $SED "s|$OTHER_UNZIPDIR/||g")

        # Separate executable/library files from other files in zip.
        DIFFING_TEXT_FILES=
        DIFFING_EXEC_FILES=
        for file in $DIFFING_FILES; do
            SUFFIX="${file##*.}"
            if [ "$SUFFIX" = "exe" -o "$SUFFIX" = "dll" -o "$SUFFIX" = "so" \
                 -o "$SUFFIX" = "dylib" ]; then
                DIFFING_EXEC_FILES="$DIFFING_EXEC_FILES $file"
            else
                DIFFING_TEXT_FILES="$DIFFING_TEXT_FILES $file"
            fi
        done

        $RM -f $WORK_DIR/$ZIP_FILE.diffs
        for file in $DIFFING_TEXT_FILES; do
            if [[ "$ACCEPTED_JARZIP_CONTENTS $EXCEPTIONS" != *"$file"* ]]; then
                diff_text $OTHER_UNZIPDIR/$file $THIS_UNZIPDIR/$file >> $WORK_DIR/$ZIP_FILE.diffs
            fi
        done

        if [ -s "$WORK_DIR/$ZIP_FILE.diffs" ]; then
            return_value=1
            echo "        Differing files in $ZIP_FILE"
            $CAT $WORK_DIR/$ZIP_FILE.diffs | $GREP 'differ$' | cut -f 2 -d ' ' | \
                $SED "s|$OTHER_UNZIPDIR|            |g" > $WORK_DIR/$ZIP_FILE.difflist
            $CAT $WORK_DIR/$ZIP_FILE.difflist

            if [ -n "$SHOW_DIFFS" ]; then
                for i in $(cat $WORK_DIR/$ZIP_FILE.difflist) ; do
                    if [ -f "${OTHER_UNZIPDIR}/$i.javap" ]; then
                        $DIFF ${OTHER_UNZIPDIR}/$i.javap ${THIS_UNZIPDIR}/$i.javap
                    elif [ -f "${OTHER_UNZIPDIR}/$i.cleaned" ]; then
                        $DIFF ${OTHER_UNZIPDIR}/$i.cleaned ${THIS_UNZIPDIR}/$i
                    else
                        $DIFF ${OTHER_UNZIPDIR}/$i ${THIS_UNZIPDIR}/$i
                    fi
                done
            fi
        fi

        # Use the compare_bin_file function for comparing the executable files.
        for file in $DIFFING_EXEC_FILES; do
            compare_bin_file $THIS_UNZIPDIR $OTHER_UNZIPDIR $WORK_DIR/$ZIP_FILE.bin \
                             $file
            if [ "$?" != "0" ]; then
                return_value=1
            fi
        done
    fi

    return $return_value
}

################################################################################
# Compare jmod file

compare_jmod_file() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    JMOD_FILE=$4

    THIS_JMOD=$THIS_DIR/$JMOD_FILE
    OTHER_JMOD=$OTHER_DIR/$JMOD_FILE

    if $CMP $OTHER_JMOD $THIS_JMOD > /dev/null; then
        return 0
    fi

    THIS_JMOD_LIST=$WORK_DIR/$JMOD_FILE.list.this
    OTHER_JMOD_LIST=$WORK_DIR/$JMOD_FILE.list.other
    mkdir -p $(dirname $THIS_JMOD_LIST) $(dirname $OTHER_JMOD_LIST)

    $JMOD list $THIS_JMOD | sort > $THIS_JMOD_LIST
    $JMOD list $OTHER_JMOD | sort > $OTHER_JMOD_LIST
    JMOD_LIST_DIFF_FILE=$WORK_DIR/$JMOD_FILE.list.diff
    $DIFF $THIS_JMOD_LIST $OTHER_JMOD_LIST > $JMOD_LIST_DIFF_FILE

    ONLY_THIS=$($GREP "^<" $JMOD_LIST_DIFF_FILE)
    ONLY_OTHER=$($GREP "^>" $JMOD_LIST_DIFF_FILE)

    if [ -n "$ONLY_OTHER" ]; then
        echo "        Only OTHER $JMOD_FILE contains:"
        echo "$ONLY_OTHER" | sed "s|^>|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    if [ -n "$ONLY_THIS" ]; then
        echo "        Only THIS $JMOD_FILE contains:"
        echo "$ONLY_THIS" | sed "s|^<|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    return $return_value
}

################################################################################
# Compare all zip files

compare_all_zip_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    ZIPS=$(cd $THIS_DIR && $FIND . -type f -name "*.zip" -o -name "*.tar.gz" \
        | $SORT | $FILTER )

    if [ -n "$ZIPS" ]; then
        echo Zip/tar.gz files...

        return_value=0
        for f in $ZIPS; do
            if [ -f "$OTHER_DIR/$f" ]; then
                compare_zip_file $THIS_DIR $OTHER_DIR $WORK_DIR $f
                if [ "$?" != "0" ]; then
                    return_value=1
                    REGRESSIONS=true
                fi
            fi
        done
    fi

    return $return_value
}

################################################################################
# Compare all jmod files

compare_all_jmod_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    JMODS=$(cd $THIS_DIR && $FIND . -type f -name "*.jmod" | $SORT | $FILTER )

    if [ -n "$JMODS" ]; then
        echo Jmod files...

        return_value=0
        for f in $JMODS; do
            if [ -f "$OTHER_DIR/$f" ]; then
                compare_jmod_file $THIS_DIR $OTHER_DIR $WORK_DIR $f
                if [ "$?" != "0" ]; then
                    return_value=1
                    REGRESSIONS=true
                fi
            fi
        done
    fi

    return $return_value
}

################################################################################
# Compare all jar files

compare_all_jar_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    # TODO filter?
    ZIPS=$(cd $THIS_DIR && $FIND . -type f -name "*.jar" -o -name "*.war" \
        -o -name "modules" | $SORT | $FILTER)

    if [ -n "$ZIPS" ]; then
        echo Jar files...

        return_value=0
        for f in $ZIPS; do
            if [ -f "$OTHER_DIR/$f" ]; then
                compare_zip_file $THIS_DIR $OTHER_DIR $WORK_DIR $f
                if [ "$?" != "0" ]; then
                    return_value=1
                    REGRESSIONS=true
                fi
            fi
        done
    fi

    return $return_value
}

################################################################################
# Compare binary (executable/library) file

compare_bin_file() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    BIN_FILE=$4
    OTHER_BIN_FILE=$5

    THIS_FILE=$THIS_DIR/$BIN_FILE
    if [ -n "$OTHER_BIN_FILE" ]; then
        OTHER_FILE=$OTHER_DIR/$OTHER_BIN_FILE
    else
        OTHER_FILE=$OTHER_DIR/$BIN_FILE
    fi
    NAME=$(basename $BIN_FILE)
    WORK_FILE_BASE=$WORK_DIR/$BIN_FILE
    FILE_WORK_DIR=$(dirname $WORK_FILE_BASE)

    $MKDIR -p $FILE_WORK_DIR

    # Make soft links to original files from work dir to facilitate debugging
    $LN -f -s $THIS_FILE $WORK_FILE_BASE.this
    $LN -f -s $OTHER_FILE $WORK_FILE_BASE.other

    ORIG_THIS_FILE="$THIS_FILE"
    ORIG_OTHER_FILE="$OTHER_FILE"

    if [ "$STRIP_ALL" = "true" ] || [[ "$STRIP_BEFORE_COMPARE" = *"$BIN_FILE"* ]]; then
        THIS_STRIPPED_FILE=$FILE_WORK_DIR/this/$NAME
        OTHER_STRIPPED_FILE=$FILE_WORK_DIR/other/$NAME
        $MKDIR -p $FILE_WORK_DIR/this $FILE_WORK_DIR/other
        $CP $THIS_FILE $THIS_STRIPPED_FILE
        $CP $OTHER_FILE $OTHER_STRIPPED_FILE
        $STRIP $THIS_STRIPPED_FILE
        $STRIP $OTHER_STRIPPED_FILE
        THIS_FILE="$THIS_STRIPPED_FILE"
        OTHER_FILE="$OTHER_STRIPPED_FILE"
    fi

    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
        unset _NT_SYMBOL_PATH
        if [ "$(uname -o)" = "Cygwin" ]; then
            THIS=$(cygpath -msa $THIS)
            OTHER=$(cygpath -msa $OTHER)
        fi
        # Build an _NT_SYMBOL_PATH that contains all known locations for
        # pdb files.
        PDB_DIRS="$(ls -d \
            {$OTHER,$THIS}/support/modules_{cmds,libs}/{*,*/*} \
            {$OTHER,$THIS}/support/native/jdk.jpackage/* \
            )"
        export _NT_SYMBOL_PATH="$(echo $PDB_DIRS | tr ' ' ';')"
    fi

    if cmp $OTHER_FILE $THIS_FILE > /dev/null; then
        # The files were bytewise identical.
        if [ -n "$VERBOSE" ]; then
            echo "        :           :         :         :          :          : $BIN_FILE"
        fi
        return 0
    fi
    if [ -z "$SKIP_BIN_DIFF" ]; then
        BIN_MSG=" diff "
        if [[ "$ACCEPTED_BIN_DIFF" != *"$BIN_FILE"* ]]; then
            DIFF_BIN=true
            if [[ "$KNOWN_BIN_DIFF" != *"$BIN_FILE"* ]]; then
                BIN_MSG="*$BIN_MSG*"
                REGRESSIONS=true
            else
                BIN_MSG=" $BIN_MSG "
            fi
        else
            BIN_MSG="($BIN_MSG)"
            DIFF_BIN=
        fi
    else
        BIN_MSG=
        DIFF_BIN=
    fi

    if [ -n "$STAT" ]; then
        THIS_SIZE=$($STAT $STAT_PRINT_SIZE "$THIS_FILE")
        OTHER_SIZE=$($STAT $STAT_PRINT_SIZE "$OTHER_FILE")
    else
        THIS_SIZE=$(ls -l "$THIS_FILE" | awk '{ print $5 }')
        OTHER_SIZE=$(ls -l "$OTHER_FILE" | awk '{ print $5 }')
    fi
    if [ $THIS_SIZE -ne $OTHER_SIZE ]; then
        DIFF_SIZE_NUM=$($EXPR $THIS_SIZE - $OTHER_SIZE)
        DIFF_SIZE_REL=$($EXPR $THIS_SIZE \* 100 / $OTHER_SIZE)
        SIZE_MSG=$($PRINTF "%3d%% %4d" $DIFF_SIZE_REL $DIFF_SIZE_NUM)
        if [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* || "$ACCEPTED_SMALL_SIZE_DIFF" = "true" ]] \
            && [ "$DIFF_SIZE_REL" -gt 98 ] \
            && [ "$DIFF_SIZE_REL" -lt 102 ]; then
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        elif [ "$OPENJDK_TARGET_OS" = "windows" ] \
            && [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* ]] \
            && [ "$DIFF_SIZE_NUM" = 512 ]; then
            # On windows, size of binaries increase in 512 increments.
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        elif [ "$OPENJDK_TARGET_OS" = "windows" ] \
            && [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* ]] \
            && [ "$DIFF_SIZE_NUM" = -512 ]; then
            # On windows, size of binaries increase in 512 increments.
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        else
            if [[ "$ACCEPTED_SIZE_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_SIZE=true
                if [[ "$KNOWN_SIZE_DIFF" != *"$BIN_FILE"* ]]; then
                    SIZE_MSG="*$SIZE_MSG*"
                    REGRESSIONS=true
                else
                    SIZE_MSG=" $SIZE_MSG "
                fi
            else
                SIZE_MSG="($SIZE_MSG)"
                DIFF_SIZE=
            fi
        fi
    else
        SIZE_MSG="           "
        DIFF_SIZE=
        if [[ "$KNOWN_SIZE_DIFF $ACCEPTED_SIZE_DIFF" = *"$BIN_FILE"* ]]; then
            SIZE_MSG="     !     "
        fi
    fi

    if [ "$SORT_ALL_SYMBOLS" = "true" ] || [[ "$SORT_SYMBOLS" = *"$BIN_FILE"* ]]; then
        SYM_SORT_CMD="sort"
    else
        SYM_SORT_CMD="cat"
    fi

    if [ -n "$SYMBOLS_DIFF_FILTER" ] && [ -z "$NEED_SYMBOLS_DIFF_FILTER" ] \
            || [[ "$NEED_SYMBOLS_DIFF_FILTER" = *"$BIN_FILE"* ]]; then
        this_SYMBOLS_DIFF_FILTER="$SYMBOLS_DIFF_FILTER"
    else
        this_SYMBOLS_DIFF_FILTER="$CAT"
    fi

    # Check symbols
    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
        # The output from dumpbin on windows differs depending on if the debug symbol
        # files are still around at the location the binary is pointing too. Need
        # to filter out that extra information.
        $DUMPBIN -exports $OTHER_FILE | $GREP  -E '^ +[0-9A-F]+ +[0-9A-F]+ [0-9A-F]+' | sed 's/ = .*//g' | cut -c27- | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
        $DUMPBIN -exports $THIS_FILE  | $GREP  -E '^ +[0-9A-F]+ +[0-9A-F]+ [0-9A-F]+' | sed 's/ = .*//g' | cut -c27- | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    elif [ "$OPENJDK_TARGET_OS" = "aix" ]; then
        $OBJDUMP -T $ORIG_OTHER_FILE 2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
        $OBJDUMP -T $ORIG_THIS_FILE  2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    elif [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
        $NM -j $ORIG_OTHER_FILE 2> /dev/null | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
        $NM -j $ORIG_THIS_FILE  2> /dev/null | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    else
        $NM -a $ORIG_OTHER_FILE 2> /dev/null | $GREP -v $NAME \
            | $AWK '{print $2, $3, $4, $5}' \
            | eval "$this_SYMBOLS_DIFF_FILTER" \
            | $SYM_SORT_CMD \
            > $WORK_FILE_BASE.symbols.other
        $NM -a $ORIG_THIS_FILE  2> /dev/null | $GREP -v $NAME \
            | $AWK '{print $2, $3, $4, $5}' \
            | eval "$this_SYMBOLS_DIFF_FILTER" \
            | $SYM_SORT_CMD \
            > $WORK_FILE_BASE.symbols.this
    fi

    $DIFF $WORK_FILE_BASE.symbols.other $WORK_FILE_BASE.symbols.this > $WORK_FILE_BASE.symbols.diff
    if [ -s $WORK_FILE_BASE.symbols.diff ]; then
        SYM_MSG=" diff  "
        if [[ "$ACCEPTED_SYM_DIFF" != *"$BIN_FILE"* ]]; then
            DIFF_SYM=true
            if [[ "$KNOWN_SYM_DIFF" != *"$BIN_FILE"* ]]; then
                SYM_MSG="*$SYM_MSG*"
                REGRESSIONS=true
            else
                SYM_MSG=" $SYM_MSG "
            fi
        else
            SYM_MSG="($SYM_MSG)"
            DIFF_SYM=
        fi
    else
        SYM_MSG="         "
        DIFF_SYM=
        if [[ "$KNOWN_SYM_DIFF $ACCEPTED_SYM_DIFF" = *"$BIN_FILE"* ]]; then
            SYM_MSG="    !    "
        fi
    fi

    # Check dependencies
    if [ -n "$LDD_CMD" ]; then
        if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
            LDD_FILTER="$GREP \.dll"
        else
            LDD_FILTER="$CAT"
        fi
        (cd $FILE_WORK_DIR && $CP $OTHER_FILE . && $LDD_CMD $NAME 2>/dev/null \
                    | $LDD_FILTER | $AWK '{ print $1;}' | $SORT \
                    | $TEE $WORK_FILE_BASE.deps.other \
                    | $UNIQ > $WORK_FILE_BASE.deps.other.uniq)
        (cd $FILE_WORK_DIR && $CP $THIS_FILE . && $LDD_CMD $NAME 2</dev/null \
                    | $LDD_FILTER | $AWK '{ print $1;}' | $SORT \
                    | $TEE $WORK_FILE_BASE.deps.this \
                    | $UNIQ > $WORK_FILE_BASE.deps.this.uniq)
        (cd $FILE_WORK_DIR && $RM -f $NAME)

        $DIFF $WORK_FILE_BASE.deps.other $WORK_FILE_BASE.deps.this \
              > $WORK_FILE_BASE.deps.diff
        $DIFF $WORK_FILE_BASE.deps.other.uniq $WORK_FILE_BASE.deps.this.uniq \
              > $WORK_FILE_BASE.deps.diff.uniq

        if [ -s $WORK_FILE_BASE.deps.diff ]; then
            if [ -s $WORK_FILE_BASE.deps.diff.uniq ]; then
                DEP_MSG=" diff  "
            else
                DEP_MSG=" redun "
            fi
            if [[ "$ACCEPTED_DEP_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_DEP=true
                if [[ "$KNOWN_DEP_DIFF" != *"$BIN_FILE"* ]]; then
                    DEP_MSG="*$DEP_MSG*"
                    REGRESSIONS=true
                else
                    DEP_MSG=" $DEP_MSG "
                fi
            else
                DEP_MSG="($DEP_MSG)"
                DIFF_DEP=
            fi
        else
            DEP_MSG="         "
            DIFF_DEP=
            if [[ "$KNOWN_DEP_DIFF $ACCEPTED_DEP_DIFF" = *"$BIN_FILE"* ]]; then
                DEP_MSG="     !      "
            fi
        fi
    else
        DEP_MSG="    -    "
    fi

    # Some linux compilers add a unique Build ID
    if [ "$OPENJDK_TARGET_OS" = "linux" ]; then
      BUILD_ID_FILTER="$SED -r 's/(Build ID:) [0-9a-f]{40}/\1/'"
    else
      BUILD_ID_FILTER="$CAT"
    fi

    # Compare fulldump output
    if [ -n "$FULLDUMP_CMD" ] && [ -z "$SKIP_FULLDUMP_DIFF" ]; then
        if [ -z "$FULLDUMP_DIFF_FILTER" ]; then
            FULLDUMP_DIFF_FILTER="$CAT"
        fi
        $FULLDUMP_CMD $OTHER_FILE | eval "$BUILD_ID_FILTER" | eval "$FULLDUMP_DIFF_FILTER" \
            > $WORK_FILE_BASE.fulldump.other 2>&1 &
        $FULLDUMP_CMD $THIS_FILE  | eval "$BUILD_ID_FILTER" | eval "$FULLDUMP_DIFF_FILTER" \
            > $WORK_FILE_BASE.fulldump.this  2>&1 &
        wait

        $DIFF $WORK_FILE_BASE.fulldump.other $WORK_FILE_BASE.fulldump.this \
            > $WORK_FILE_BASE.fulldump.diff

        if [ -s $WORK_FILE_BASE.fulldump.diff ]; then
            FULLDUMP_DIFF_SIZE=$(ls -n $WORK_FILE_BASE.fulldump.diff | awk '{print $5}')
            FULLDUMP_MSG=$($PRINTF "%8d" $FULLDUMP_DIFF_SIZE)
            if [[ "$ACCEPTED_FULLDUMP_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_FULLDUMP=true
                if [[ "$KNOWN_FULLDUMP_DIFF" != *"$BIN_FILE"* ]]; then
                    FULLDUMP_MSG="*$FULLDUMP_MSG*"
                    REGRESSIONS=true
                else
                    FULLDUMP_MSG=" $FULLDUMP_MSG "
                fi
            else
                FULLDUMP_MSG="($FULLDUMP_MSG)"
                DIFF_FULLDUMP=
            fi
        else
            FULLDUMP_MSG="          "
            DIFF_FULLDUMP=
            if [[ "$KNOWN_FULLDUMP_DIFF $ACCEPTED_FULLDUMP_DIFF" = *"$BIN_FILE"* ]]; then
                FULLDUMP_MSG="    !    "
            fi
        fi
    fi

    # Compare disassemble output
    if [ -n "$DIS_CMD" ] && [ -z "$SKIP_DIS_DIFF" ]; then
        this_DIS_DIFF_FILTER="$CAT"
        if [ -n "$DIS_DIFF_FILTER" ]; then
            if [ -z "$NEED_DIS_DIFF_FILTER" ] \
                || [[ "$NEED_DIS_DIFF_FILTER" = *"$BIN_FILE"* ]]; then
                this_DIS_DIFF_FILTER="$DIS_DIFF_FILTER"
            fi
        fi
        if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
            DIS_GREP_ARG=-a
        else
            DIS_GREP_ARG=
        fi
        $DIS_CMD $OTHER_FILE | $GREP $DIS_GREP_ARG -v $NAME \
            | eval "$this_DIS_DIFF_FILTER" > $WORK_FILE_BASE.dis.other 2>&1 &
        $DIS_CMD $THIS_FILE  | $GREP $DIS_GREP_ARG -v $NAME \
            | eval "$this_DIS_DIFF_FILTER" > $WORK_FILE_BASE.dis.this  2>&1 &
        wait

        $DIFF $WORK_FILE_BASE.dis.other $WORK_FILE_BASE.dis.this > $WORK_FILE_BASE.dis.diff

        if [ -s $WORK_FILE_BASE.dis.diff ]; then
            DIS_DIFF_SIZE=$(ls -n $WORK_FILE_BASE.dis.diff | awk '{print $5}')
            DIS_MSG=$($PRINTF "%8d" $DIS_DIFF_SIZE)
            if [[ "$ACCEPTED_DIS_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_DIS=true
                if [ "$MAX_KNOWN_DIS_DIFF_SIZE" = "" ]; then
                    MAX_KNOWN_DIS_DIFF_SIZE="0"
                fi
                if [[ "$KNOWN_DIS_DIFF" = *"$BIN_FILE"* ]] \
                    && [ "$DIS_DIFF_SIZE" -lt "$MAX_KNOWN_DIS_DIFF_SIZE" ]; then
                    DIS_MSG=" $DIS_MSG "
                else
                    DIS_MSG="*$DIS_MSG*"
                    REGRESSIONS=true
                fi
            else
                DIS_MSG="($DIS_MSG)"
                DIFF_DIS=
            fi
        else
            DIS_MSG="          "
            DIFF_DIS=
            if [[ "$KNOWN_DEP_DIFF $ACCEPTED_DEP_DIFF" = *"$BIN_FILE"* ]]; then
                DIS_MSG="    !    "
            fi
        fi
    fi


    if [ -n "$DIFF_BIN$DIFF_SIZE$DIFF_SYM$DIFF_DEP$DIFF_FULLDUMP$DIFF_DIS" ] || [ -n "$VERBOSE" ]; then
        if [ -n "$BIN_MSG" ]; then echo -n "$BIN_MSG:"; fi
        if [ -n "$SIZE_MSG" ]; then echo -n "$SIZE_MSG:"; fi
        if [ -n "$SYM_MSG" ]; then echo -n "$SYM_MSG:"; fi
        if [ -n "$DEP_MSG" ]; then echo -n "$DEP_MSG:"; fi
        if [ -n "$FULLDUMP_MSG" ]; then echo -n "$FULLDUMP_MSG:"; fi
        if [ -n "$DIS_MSG" ]; then echo -n "$DIS_MSG:"; fi
        echo " $BIN_FILE"
        if [ "$SHOW_DIFFS" = "true" ]; then
            if [ -s "$WORK_FILE_BASE.symbols.diff" ]; then
                echo "Symbols diff:"
                $CAT $WORK_FILE_BASE.symbols.diff
            fi
            if [ -s "$WORK_FILE_BASE.deps.diff" ]; then
                echo "Deps diff:"
                $CAT $WORK_FILE_BASE.deps.diff
            fi
            if [ -s "$WORK_FILE_BASE.fulldump.diff" ]; then
                echo "Fulldump diff:"
                $CAT $WORK_FILE_BASE.fulldump.diff
            fi
            if [ -s "$WORK_FILE_BASE.dis.diff" ]; then
                echo "Disassembly diff:"
                $CAT $WORK_FILE_BASE.dis.diff
            fi
        fi
        return 1
    fi
    return 0
}

################################################################################
# Print binary diff header

print_binary_diff_header() {
    if [ -z "$SKIP_BIN_DIFF" ]; then echo -n " Binary :"; fi
    if [ -z "$SKIP_SIZE_DIFF" ]; then echo -n "   Size    :"; fi
    if [ -z "$SKIP_SYM_DIFF" ]; then echo -n " Symbols :"; fi
    if [ -z "$SKIP_DEP_DIFF" ]; then echo -n "  Deps   :"; fi
    if [ -z "$SKIP_FULLDUMP_DIFF" ]; then echo -n " Fulldump :"; fi
    if [ -z "$SKIP_DIS_DIFF" ]; then echo -n " Disass   :"; fi
    echo
}

################################################################################
# Compare all libraries

compare_all_libs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    LIBS=$(cd $THIS_DIR && $FIND . -type f \( -name 'lib*.so' -o -name '*.dylib' \
        -o -name '*.dll' -o -name '*.obj' -o -name '*.o' -o -name '*.a' \
        -o -name '*.cpl' \) | $SORT | $FILTER)

    # On macos, filter out the dSYM debug symbols files as they are also
    # named *.dylib.
    if [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
        LIBS=$(echo "$LIBS" | $GREP -v '\.dSYM/')
    fi

    if [ -n "$LIBS" ]; then
        echo Libraries...
        print_binary_diff_header
        for l in $LIBS; do
            if [ -f "$OTHER_DIR/$l" ]; then
                compare_bin_file $THIS_DIR $OTHER_DIR $WORK_DIR $l
                if [ "$?" != "0" ]; then
                    return_value=1
                fi
            fi
        done
    fi

    return $return_value
}

################################################################################
# Compare all executables

compare_all_execs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
        EXECS=$(cd $THIS_DIR && $FIND . -type f -name '*.exe' | $SORT | $FILTER)
    else
        EXECS=$(cd $THIS_DIR && $FIND . -name db -prune -o -type f -perm -100 \! \
            \( -name '*.so' -o -name '*.dylib' -o -name '*.dll' -o -name '*.cgi' \
            -o -name '*.jar' -o -name '*.diz' -o -name 'jcontrol' -o -name '*.properties' \
            -o -name '*.data' -o -name '*.bfc' -o -name '*.src' -o -name '*.txt' \
            -o -name '*.cfg' -o -name 'meta-index' -o -name '*.properties.ja' \
            -o -name '*.xml' -o -name '*.html' -o -name '*.png' -o -name 'README' \
            -o -name '*.zip' -o -name '*.jimage' -o -name '*.java' -o -name '*.mf' \
            -o -name '*.jpg' -o -name '*.wsdl' -o -name '*.js' -o -name '*.sh' \
            -o -name '*.bat' -o -name '*LICENSE' -o -name '*.d' -o -name '*store' \
            -o -name 'blocked' -o -name '*certs' -o -name '*.ttf' \
            -o -name '*.jfc' -o -name '*.dat'  -o -name 'release' -o -name '*.dir'\
            -o -name '*.sym' -o -name '*.idl' -o -name '*.h' -o -name '*.access' \
            -o -name '*.template' -o -name '*.policy' -o -name '*.security' \
            -o -name 'COPYRIGHT' -o -name '*.1' -o -name '*.debuginfo' \
            -o -name 'classlist' \) | $SORT | $FILTER)
    fi

    if [ -n "$EXECS" ]; then
        echo Executables...
        print_binary_diff_header
        for e in $EXECS; do
            if [ -f "$OTHER_DIR/$e" ]; then
                compare_bin_file $THIS_DIR $OTHER_DIR $WORK_DIR $e
                if [ "$?" != "0" ]; then
                    return_value=1
                fi
            fi
        done
    fi

    return $return_value
}

################################################################################
# Initiate configuration

THIS="$SCRIPT_DIR"
echo "$THIS"
THIS_SCRIPT="$0"

if [ -z "$1" ] || [ "$1" = "-h" ] || [ "$1" = "-?" ] || [ "$1" = "/h" ] || [ "$1" = "/?" ] || [ "$1" = "-help" ] || [ "$1" = "--help" ]; then
    echo "bash ./compare.sh [OPTIONS] [FILTER]"
    echo ""
    echo "-all                Compare all files in all known ways"
    echo "-names              Compare the file names and directory structure"
    echo "-perms              Compare the permission bits on all files and directories"
    echo "-types              Compare the output of the file command on all files"
    echo "-general            Compare the files not convered by the specialized comparisons"
    echo "-zips               Compare the contents of all zip files and files in them"
    echo "-zips-names         Compare the file names inside all zip files"
    echo "-jars               Compare the contents of all jar files"
    echo "-jmods              Compare the listings of all jmod files"
    echo "-libs               Compare all native libraries"
    echo "-execs              Compare all executables"
    echo "-v                  Verbose output, does not hide known differences"
    echo "-vv                 More verbose output, shows diff output of all comparisons"
    echo "-o [OTHER]          Compare with build in other directory. Will default to the old build directory"
    echo ""
    echo "--sort-symbols      Sort all symbols before comparing"
    echo "--strip             Strip all binaries before comparing"
    echo "--clean             Clean all previous comparison results first"
    echo ""
    echo "[FILTER]            List filenames in the image to compare, works for jars, zips, libs and execs"
    echo "Example:"
    echo "bash ./make/scripts/compareimages.sh CodePointIM.jar"
    echo ""
    echo "-2zips <file1> <file2> Compare two zip files only"
    echo "-2bins <file1> <file2> Compare two binary files only"
    echo "-2dirs <dir1> <dir2> Compare two directories as if they were images"
    echo ""
    exit 10
fi

CMP_NAMES=false
CMP_PERMS=false
CMP_TYPES=false
CMP_GENERAL=false
CMP_ZIPS=false
CMP_ZIPS_CONTENTS=true
CMP_JARS=false
CMP_JMODS=false
CMP_LIBS=false
CMP_EXECS=false

while [ -n "$1" ]; do
    case "$1" in
        -v)
            VERBOSE=true
            ;;
        -vv)
            VERBOSE=true
            SHOW_DIFFS=true
            ;;
        -o)
            OTHER="$2"
            shift
            ;;
        -all)
            CMP_NAMES=true
            if [ "$OPENJDK_TARGET_OS" != "windows" ]; then
                CMP_PERMS=true
            fi
            CMP_TYPES=true
            CMP_GENERAL=true
            CMP_ZIPS=true
            CMP_JARS=true
            CMP_JMODS=true
            CMP_LIBS=true
            CMP_EXECS=true
            ;;
        -names)
            CMP_NAMES=true
            ;;
        -perms)
            CMP_PERMS=true
            ;;
        -types)
            CMP_TYPES=true
            ;;
        -general)
            CMP_GENERAL=true
            ;;
        -zips)
            CMP_ZIPS=true
            CMP_ZIPS_CONTENTS=true
            ;;
        -zips-names)
            CMP_ZIPS=true
            CMP_ZIPS_CONTENTS=false
            ;;
        -jars)
            CMP_JARS=true
            ;;
        -jmods)
            CMP_JMODS=true
            ;;
        -libs)
            CMP_LIBS=true
            ;;
        -execs)
            CMP_EXECS=true
            ;;
        -2dirs)
            THIS="$(cd "$2" > /dev/null && pwd )"
            OTHER="$(cd "$3" > /dev/null && pwd )"
            THIS_BASE_DIR="$THIS"
            OTHER_BASE_DIR="$OTHER"
            SKIP_DEFAULT=true
            shift
            shift
            ;;
        -2zips)
            CMP_2_ZIPS=true
            THIS_FILE=$2
            OTHER_FILE=$3
            shift
            shift
            ;;
        -2bins)
            CMP_2_BINS=true
            THIS_FILE=$2
            OTHER_FILE=$3
            shift
            shift
            ;;
        --sort-symbols)
            SORT_ALL_SYMBOLS=true
            ;;
        --strip)
            STRIP_ALL=true
            ;;
        --clean)
            CLEAN_OUTPUT=true
            ;;
        *)
            CMP_NAMES=false
            CMP_PERMS=false
            CMP_TYPES=false
            CMP_ZIPS=true
            CMP_JARS=true
            CMP_JMODS=true
            CMP_LIBS=true
            CMP_EXECS=true

            if [ -z "$FILTER" ]; then
                FILTER="$GREP"
            fi
            FILTER="$FILTER -e $1"
            ;;
    esac
    shift
done

if [ "$STRIP_ALL" = "true" ] && [ -z "$STRIP" ]; then
  echo Warning: Not stripping even with --strip, since strip is missing on this platform
  STRIP_ALL=false
fi

COMPARE_ROOT=$OUTPUTDIR/compare-support
if [ "$CLEAN_OUTPUT" = "true" ]; then
    echo Cleaning old output in $COMPARE_ROOT.
    $RM -rf $COMPARE_ROOT
fi
$MKDIR -p $COMPARE_ROOT
if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
    if [ "$(uname -o)" = "Cygwin" ]; then
        COMPARE_ROOT=$(cygpath -msa $COMPARE_ROOT)
    fi
fi

if [ "$CMP_2_ZIPS" = "true" ]; then
    THIS_DIR="$(dirname $THIS_FILE)"
    THIS_DIR="$(cd "$THIS_DIR" > /dev/null && pwd )"
    OTHER_DIR="$(dirname $OTHER_FILE)"
    OTHER_DIR="$(cd "$OTHER_DIR" > /dev/null && pwd )"
    THIS_FILE_NAME="$(basename $THIS_FILE)"
    OTHER_FILE_NAME="$(basename $OTHER_FILE)"
    echo Comparing $THIS_DIR/$THIS_FILE_NAME and $OTHER_DIR/$OTHER_FILE_NAME
    compare_zip_file $THIS_DIR $OTHER_DIR $COMPARE_ROOT/2zips $THIS_FILE_NAME $OTHER_FILE_NAME
    exit
fi

if [ "$CMP_2_BINS" = "true" ]; then
    THIS_DIR="$(dirname $THIS_FILE)"
    THIS_DIR="$(cd "$THIS_DIR" > /dev/null && pwd )"
    OTHER_DIR="$(dirname $OTHER_FILE)"
    OTHER_DIR="$(cd "$OTHER_DIR" > /dev/null && pwd )"
    THIS_FILE_NAME="$(basename $THIS_FILE)"
    OTHER_FILE_NAME="$(basename $OTHER_FILE)"
    echo Comparing $THIS_DIR/$THIS_FILE_NAME and $OTHER_DIR/$OTHER_FILE_NAME
    compare_bin_file $THIS_DIR $OTHER_DIR $COMPARE_ROOT/2bins $THIS_FILE_NAME $OTHER_FILE_NAME
    exit
fi

if [ "$CMP_NAMES" = "false" ] \
       && [ "$CMP_TYPES" = "false" ] \
       && [ "$CMP_PERMS" = "false" ] \
       && [ "$CMP_GENERAL" = "false" ] \
       && [ "$CMP_ZIPS" = "false" ] \
       && [ "$CMP_JARS" = "false" ] \
       && [ "$CMP_JMODS" = "false" ] \
       && [ "$CMP_LIBS" = "false" ] \
       && [ "$CMP_EXECS" = "false" ]; then
    CMP_NAMES=true
    CMP_PERMS=true
    CMP_TYPES=true
    CMP_GENERAL=true
    CMP_ZIPS=true
    CMP_JARS=true
    CMP_JMODS=true
    CMP_LIBS=true
    CMP_EXECS=true
fi

if [ -z "$FILTER" ]; then
    FILTER="$CAT"
fi

if [ "$SKIP_DEFAULT" != "true" ]; then
    if [ -z "$OTHER" ]; then
        echo "Nothing to compare to, set with -o"
        exit 1
    else
        if [ ! -d "$OTHER" ]; then
            echo "Other build directory does not exist:"
            echo "$OTHER"
            exit 1
        fi
        OTHER="$( cd "$OTHER" > /dev/null && pwd )"
        echo "Comparing to:"
        echo "$OTHER"
        echo
    fi

    # Find the common images to compare, prioritizing later build stages
    if [ -d "$THIS/images/jdk" ] && [ -d "$OTHER/images/jdk" ]; then
        THIS_JDK="$THIS/images/jdk"
        OTHER_JDK="$OTHER/images/jdk"
        echo "Selecting normal images for JDK compare"
    elif [ -d "$(ls -d $THIS/licensee-src/build/*/images/jdk 2> /dev/null)" ] \
        && [ -d "$(ls -d $OTHER/licensee-src/build/*/images/jdk 2> /dev/null)" ]
    then
        echo "Selecting licensee images for compare"
        # Simply override the THIS and OTHER dir with the build dir from
        # the nested licensee source build for the rest of the script
        # execution.
        OLD_THIS="$THIS"
        OLD_OTHER="$OTHER"
        THIS="$(ls -d $THIS/licensee-src/build/*)"
        OTHER="$(ls -d $OTHER/licensee-src/build/*)"
        THIS_JDK="$THIS/images/jdk"
        OTHER_JDK="$OTHER/images/jdk"
        # Rewrite the path to tools that are used from the build
        JIMAGE="$(echo "$JIMAGE" | $SED "s|$OLD_THIS|$THIS|g")"
        JMOD="$(echo "$JMOD" | $SED "s|$OLD_THIS|$THIS|g")"
        JAVAP="$(echo "$JAVAP" | $SED "s|$OLD_THIS|$THIS|g")"
    else
        echo "No common images found."
        exit 1
    fi
    echo "  $THIS_JDK"
    echo "  $OTHER_JDK"

    if [ -d "$THIS/images/jdk-bundle" -o -d "$THIS/deploy/images/jdk-bundle" ] \
             && [ -d "$OTHER/images/jdk-bundle" -o -d "$OTHER/deploy/images/jdk-bundle" ]; then
        if [ -d "$THIS/deploy/images/jdk-bundle" ]; then
            THIS_JDK_BUNDLE="$THIS/deploy/images/jdk-bundle"
        else
            THIS_JDK_BUNDLE="$THIS/images/jdk-bundle"
        fi
        if [ -d "$OTHER/deploy/images/jdk-bundle" ]; then
            OTHER_JDK_BUNDLE="$OTHER/deploy/images/jdk-bundle"
        else
            OTHER_JDK_BUNDLE="$OTHER/images/jdk-bundle"
        fi
        echo "Also comparing jdk macosx bundles"
        echo "  $THIS_JDK_BUNDLE"
        echo "  $OTHER_JDK_BUNDLE"
    fi

    THIS_SEC_DIR="$THIS/images"
    OTHER_SEC_DIR="$OTHER/images"
    if [ -f "$THIS_SEC_DIR/sec-bin.zip" ] && [ -f "$OTHER_SEC_DIR/sec-bin.zip" ]; then
        OTHER_SEC_BIN="$OTHER_SEC_DIR/sec-bin.zip"
        THIS_SEC_BIN="$THIS_SEC_DIR/sec-bin.zip"
        if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
            if [ "$OPENJDK_TARGET_CPU" = "x86_64" ]; then
                JGSS_WINDOWS_BIN="jgss-windows-x64-bin.zip"
            else
                JGSS_WINDOWS_BIN="jgss-windows-i586-bin.zip"
            fi
            OTHER_SEC_WINDOWS_BIN="$OTHER_SEC_DIR/sec-windows-bin.zip"
            OTHER_JGSS_WINDOWS_BIN="$OTHER_SEC_DIR/$JGSS_WINDOWS_BIN"
            THIS_SEC_WINDOWS_BIN="$THIS_SEC_DIR/sec-windows-bin.zip"
            THIS_JGSS_WINDOWS_BIN="$THIS_SEC_DIR/$JGSS_WINDOWS_BIN"
        fi
    fi

    if [ -d "$THIS/images/docs" ] && [ -d "$OTHER/images/docs" ]; then
        THIS_DOCS="$THIS/images/docs"
        OTHER_DOCS="$OTHER/images/docs"
        echo "Also comparing docs"
    else
        echo "WARNING! Docs haven't been built and won't be compared."
    fi

    if [ -d "$THIS/images/test" ] && [ -d "$OTHER/images/test" ]; then
        THIS_TEST="$THIS/images/test"
        OTHER_TEST="$OTHER/images/test"
        echo "Also comparing test image"
    else
        echo "WARNING! Test haven't been built and won't be compared."
    fi
fi

################################################################################
# Do the work

if [ "$CMP_NAMES" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_dirs $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
        echo -n "JDK "
        compare_files $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_JDK_BUNDLE" ] && [ -n "$OTHER_JDK_BUNDLE" ]; then
        echo -n "JDK Bundle "
        compare_dirs $THIS_JDK_BUNDLE $OTHER_JDK_BUNDLE $COMPARE_ROOT/jdk-bundle

        echo -n "JDK Bundle "
        compare_files $THIS_JDK_BUNDLE $OTHER_JDK_BUNDLE $COMPARE_ROOT/jdk-bundle
    fi
    if [ -n "$THIS_DOCS" ] && [ -n "$OTHER_DOCS" ]; then
        echo -n "Docs "
        compare_dirs $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
        echo -n "Docs "
        compare_files $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        compare_dirs $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
        echo -n "Test "
        compare_files $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_dirs $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
        compare_files $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_LIBS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_all_libs $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        STRIP_ALL_bak="$STRIP_ALL"
        if [ "$STRIP_TESTS_BEFORE_COMPARE" = "true" ]; then
          STRIP_ALL="true"
        fi
        compare_all_libs $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
        STRIP_ALL="$STRIP_ALL_bak"
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_all_libs $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_EXECS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_all_execs $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        STRIP_ALL_bak="$STRIP_ALL"
        if [ "$STRIP_TESTS_BEFORE_COMPARE" = "true" ]; then
          STRIP_ALL="true"
        fi
        compare_all_execs $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
        STRIP_ALL="$STRIP_ALL_bak"
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_all_execs $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_GENERAL" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_general_files $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_JDK_BUNDLE" ] && [ -n "$OTHER_JDK_BUNDLE" ]; then
        echo -n "JDK Bundle "
        compare_general_files $THIS_JDK_BUNDLE $OTHER_JDK_BUNDLE $COMPARE_ROOT/jdk-bundle
    fi
    if [ -n "$THIS_DOCS" ] && [ -n "$OTHER_DOCS" ]; then
        echo -n "Docs "
        compare_general_files $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        compare_general_files $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_general_files $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_ZIPS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_all_zip_files $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        compare_all_zip_files $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
    fi
    if [ -n "$THIS_SEC_BIN" ] && [ -n "$OTHER_SEC_BIN" ]; then
        if [ -n "$(echo $THIS_SEC_BIN | $FILTER)" ]; then
            echo "sec-bin.zip..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin sec-bin.zip
        fi
    fi
    if [ -n "$THIS_SEC_WINDOWS_BIN" ] && [ -n "$OTHER_SEC_WINDOWS_BIN" ]; then
        if [ -n "$(echo $THIS_SEC_WINDOWS_BIN | $FILTER)" ]; then
            echo "sec-windows-bin.zip..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin sec-windows-bin.zip
        fi
    fi
    if [ -n "$THIS_JGSS_WINDOWS_BIN" ] && [ -n "$OTHER_JGSS_WINDOWS_BIN" ]; then
        if [ -n "$(echo $THIS_JGSS_WINDOWS_BIN | $FILTER)" ]; then
            echo "$JGSS_WINDOWS_BIN..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin $JGSS_WINDOWS_BIN
        fi
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_all_zip_files $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_JARS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_all_jar_files $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        compare_all_jar_files $THIS_TEST $OTHER_TEST $COMPARE_ROOT/test
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_all_jar_files $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_JMODS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        compare_all_jmod_files $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_all_jmod_files $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_PERMS" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_permissions $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_permissions $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

if [ "$CMP_TYPES" = "true" ]; then
    if [ -n "$THIS_JDK" ] && [ -n "$OTHER_JDK" ]; then
        echo -n "JDK "
        compare_file_types $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_JDK_BUNDLE" ] && [ -n "$OTHER_JDK_BUNDLE" ]; then
        echo -n "JDK Bundle "
        compare_file_types $THIS_JDK_BUNDLE $OTHER_JDK_BUNDLE $COMPARE_ROOT/jdk-bundle
    fi
    if [ -n "$THIS_TEST" ] && [ -n "$OTHER_TEST" ]; then
        echo -n "Test "
        compare_file_types $THIS_JDK $OTHER_JDK $COMPARE_ROOT/jdk
    fi
    if [ -n "$THIS_BASE_DIR" ] && [ -n "$OTHER_BASE_DIR" ]; then
        compare_file_types $THIS_BASE_DIR $OTHER_BASE_DIR $COMPARE_ROOT/base_dir
    fi
fi

echo

if [ -n "$REGRESSIONS" ]; then
    echo "REGRESSIONS FOUND!"
    echo
    exit 1
else
    echo "No regressions found"
    echo
    exit 0
fi
