/*
 *
 *  oFono - Open Source Telephony
 *
 *  Copyright (C) 2013 Canonical Ltd.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <glib.h>
#include <errno.h>

#include <ofono/modem.h>
#include <ofono/gprs-context.h>
#include <ofono/types.h>

#include "gril.h"
#include "grilunsol.h"

/*
 * TODO: It may make sense to split this file into
 * domain-specific files ( eg. test-grilrequest-gprs-context.c )
 * once more tests are added.
 */

/*
 * As all our architectures are little-endian except for
 * PowerPC, and the Binder wire-format differs slightly
 * depending on endian-ness, the following guards against test
 * failures when run on PowerPC.
 */
#if BYTE_ORDER == LITTLE_ENDIAN

typedef struct signal_strength_test signal_strength_test;
struct signal_strength_test {
	int strength;
	const struct ril_msg msg;
};

static const struct ril_msg unsol_data_call_list_changed_invalid_1 = {
	.buf = "",
	.buf_len = 0,
	.unsolicited = TRUE,
	.req = RIL_UNSOL_DATA_CALL_LIST_CHANGED,
	.serial_no = 0,
	.error = 0,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_DATA_CALL_LIST_CHANGED message
 * with the following parameters:
 *
 * (version=7,num=1 [status=0,retry=-1,cid=0,active=1,type=IP,
 * ifname=rmnet_usb0,address=10.209.114.102/30,
 * dns=172.16.145.103 172.16.145.103,gateways=10.209.114.101]}
 */
static const char unsol_data_call_list_changed_parcel1[] = {
	0x00, 0x00, 0x00, 0xd4, 0x01, 0x00, 0x00, 0x00, 0xf2, 0x03, 0x00, 0x00,
	0x07, 0x00, 0x00, 0x00,	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,	0x01, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x49, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x72, 0x00, 0x6d, 0x00, 0x6e, 0x00, 0x65, 0x00,
	0x74, 0x00, 0x5f, 0x00,	0x75, 0x00, 0x73, 0x00, 0x62, 0x00, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,	0x31, 0x00, 0x30, 0x00,
	0x2e, 0x00, 0x32, 0x00, 0x30, 0x00, 0x39, 0x00, 0x2e, 0x00, 0x31, 0x00,
	0x31, 0x00, 0x34, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x30, 0x00, 0x32, 0x00,
	0x2f, 0x00, 0x33, 0x00,	0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x31, 0x00, 0x37, 0x00, 0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x36, 0x00,
	0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x35, 0x00, 0x2e, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x20, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00,
	0x2e, 0x00, 0x31, 0x00,	0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x35, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x30, 0x00,	0x33, 0x00, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x31, 0x00, 0x30, 0x00, 0x2e, 0x00, 0x32, 0x00,
	0x30, 0x00, 0x39, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x31, 0x00, 0x34, 0x00,
	0x2e, 0x00, 0x31, 0x00,	0x30, 0x00, 0x31, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg unsol_data_call_list_changed_valid_1 = {
	.buf = (gchar *) &unsol_data_call_list_changed_parcel1,
	.buf_len = sizeof(unsol_data_call_list_changed_parcel1),
	.unsolicited = TRUE,
	.req = RIL_UNSOL_DATA_CALL_LIST_CHANGED,
	.serial_no = 0,
	.error = 0,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_RESPONSE_NEW_SMS message
 * with the following parameter (SMSC address length is 7):
 *
 * {07914306073011F0040B914336543980F50000310113212002400AC8373B0C6AD7DDE437}
 */
static const char unsol_response_new_sms_parcel1[] = {
	0x48, 0x00, 0x00, 0x00, 0x30, 0x00, 0x37, 0x00, 0x39, 0x00, 0x31, 0x00,
	0x34, 0x00, 0x33, 0x00, 0x30, 0x00, 0x36, 0x00, 0x30, 0x00, 0x37, 0x00,
	0x33, 0x00, 0x30, 0x00, 0x31, 0x00, 0x31, 0x00, 0x46, 0x00, 0x30, 0x00,
	0x30, 0x00, 0x34, 0x00, 0x30, 0x00, 0x42, 0x00, 0x39, 0x00, 0x31, 0x00,
	0x34, 0x00, 0x33, 0x00, 0x33, 0x00, 0x36, 0x00, 0x35, 0x00, 0x34, 0x00,
	0x33, 0x00, 0x39, 0x00, 0x38, 0x00, 0x30, 0x00, 0x46, 0x00, 0x35, 0x00,
	0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x33, 0x00, 0x31, 0x00,
	0x30, 0x00, 0x31, 0x00, 0x31, 0x00, 0x33, 0x00, 0x32, 0x00, 0x31, 0x00,
	0x32, 0x00, 0x30, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x30, 0x00,
	0x30, 0x00, 0x41, 0x00, 0x43, 0x00, 0x38, 0x00, 0x33, 0x00, 0x37, 0x00,
	0x33, 0x00, 0x42, 0x00, 0x30, 0x00, 0x43, 0x00, 0x36, 0x00, 0x41, 0x00,
	0x44, 0x00, 0x37, 0x00, 0x44, 0x00, 0x44, 0x00, 0x45, 0x00, 0x34, 0x00,
	0x33, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg unsol_response_new_sms_valid_1 = {
	.buf = (gchar *) &unsol_response_new_sms_parcel1,
	.buf_len = sizeof(unsol_response_new_sms_parcel1),
	.unsolicited = TRUE,
	.req = RIL_UNSOL_RESPONSE_NEW_SMS,
	.serial_no = 0,
	.error = 0,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_RESPONSE_NEW_SMS message
 * with the following parameter (SMSC address length is 6):
 *
 * {069143060730F0040B914336543980F50000310113212002400AC8373B0C6AD7DDE437}
 */
static const char unsol_response_new_sms_parcel2[] = {
	0x46, 0x00, 0x00, 0x00, 0x30, 0x00, 0x36, 0x00, 0x39, 0x00, 0x31, 0x00,
	0x34, 0x00, 0x33, 0x00, 0x30, 0x00, 0x36, 0x00, 0x30, 0x00, 0x37, 0x00,
	0x33, 0x00, 0x30, 0x00, 0x46, 0x00, 0x30, 0x00,	0x30, 0x00, 0x34, 0x00,
	0x30, 0x00, 0x42, 0x00, 0x39, 0x00, 0x31, 0x00,	0x34, 0x00, 0x33, 0x00,
	0x33, 0x00, 0x36, 0x00, 0x35, 0x00, 0x34, 0x00,	0x33, 0x00, 0x39, 0x00,
	0x38, 0x00, 0x30, 0x00, 0x46, 0x00, 0x35, 0x00,	0x30, 0x00, 0x30, 0x00,
	0x30, 0x00, 0x30, 0x00, 0x33, 0x00, 0x31, 0x00,	0x30, 0x00, 0x31, 0x00,
	0x31, 0x00, 0x33, 0x00, 0x32, 0x00, 0x31, 0x00,	0x32, 0x00, 0x30, 0x00,
	0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x30, 0x00,	0x30, 0x00, 0x41, 0x00,
	0x43, 0x00, 0x38, 0x00, 0x33, 0x00, 0x37, 0x00,	0x33, 0x00, 0x42, 0x00,
	0x30, 0x00, 0x43, 0x00, 0x36, 0x00, 0x41, 0x00,	0x44, 0x00, 0x37, 0x00,
	0x44, 0x00, 0x44, 0x00, 0x45, 0x00, 0x34, 0x00,	0x33, 0x00, 0x37, 0x00,
	0x00, 0x00, 0x00, 0x00
};

static const struct ril_msg unsol_response_new_sms_valid_2 = {
	.buf = (gchar *) &unsol_response_new_sms_parcel2,
	.buf_len = sizeof(unsol_response_new_sms_parcel2),
	.unsolicited = TRUE,
	.req = RIL_UNSOL_RESPONSE_NEW_SMS,
	.serial_no = 0,
	.error = 0,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_SUPP_SVC_NOTIFICATION message
 * with the following parameters:
 *
 * {1,2,0,0,} -> call has been put on hold
 */
static const guchar unsol_supp_svc_notif_parcel1[] = {
	0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff
};

static const struct ril_msg unsol_supp_svc_notif_valid_1 = {
	.buf = (gchar *) &unsol_supp_svc_notif_parcel1,
	.buf_len = sizeof(unsol_supp_svc_notif_parcel1),
	.unsolicited = TRUE,
	.req = RIL_UNSOL_SUPP_SVC_NOTIFICATION,
	.serial_no = 0,
	.error = 0,
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_SIGNAL_STRENGTH message
 * with the following parameters:
 *
 * (gw: 14, cdma: -1, evdo: -1, lte: 99)
 *
 * Note, the return value for gw sigmal is: (gw * 100) / 31, which
 * in this case equals 45.
 */
static const guchar unsol_signal_strength_parcel1[] = {
	0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,	0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x63, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x7f,
	0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0x7f
};

static const signal_strength_test unsol_signal_strength_valid_1 = {
	.strength = 45,
	.msg = {
		.buf = (gchar *) &unsol_signal_strength_parcel1,
		.buf_len = sizeof(unsol_signal_strength_parcel1),
		.unsolicited = TRUE,
		.req = RIL_UNSOL_SIGNAL_STRENGTH,
		.serial_no = 0,
		.error = 0,
	}
};

/*
 * The following hexadecimal data represents a serialized Binder parcel
 * instance containing a valid RIL_UNSOL_ON_USSD message with the following
 * parameters:
 *
 * {0,Spain 12:56 09/12/13  Canary 11:56 09/12/13  }
 */
static const guchar unsol_on_ussd_parcel1[] = {
	0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
	0x2d, 0x00, 0x00, 0x00, 0x53, 0x00, 0x70, 0x00, 0x61, 0x00, 0x69, 0x00,
	0x6e, 0x00, 0x20, 0x00, 0x31, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x35, 0x00,
	0x36, 0x00, 0x20, 0x00, 0x30, 0x00, 0x39, 0x00, 0x2f, 0x00, 0x31, 0x00,
	0x32, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x33, 0x00, 0x20, 0x00, 0x20, 0x00,
	0x43, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x61, 0x00, 0x72, 0x00, 0x79, 0x00,
	0x20, 0x00, 0x31, 0x00, 0x31, 0x00, 0x3a, 0x00, 0x35, 0x00, 0x36, 0x00,
	0x20, 0x00, 0x30, 0x00, 0x39, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x32, 0x00,
	0x2f, 0x00, 0x31, 0x00, 0x33, 0x00, 0x20, 0x00, 0x20, 0x00, 0x00, 0x00
};

static const struct ril_msg unsol_on_ussd_valid_1 = {
	.buf = (gchar *) &unsol_on_ussd_parcel1,
	.buf_len = sizeof(unsol_on_ussd_parcel1),
	.unsolicited = TRUE,
	.req = RIL_UNSOL_ON_USSD,
	.serial_no = 0,
	.error = 0,
};

static void test_unsol_data_call_list_changed_invalid(gconstpointer data)
{
	struct ofono_error error;
	struct unsol_data_call_list *unsol;

	unsol = g_ril_unsol_parse_data_call_list(NULL, data, &error);
	g_assert(unsol != NULL);
	g_ril_unsol_free_data_call_list(unsol);

	g_assert(error.type == OFONO_ERROR_TYPE_FAILURE &&
			error.error == -EINVAL);
}

static void test_unsol_data_call_list_changed_valid(gconstpointer data)
{
	struct ofono_error error;
	struct unsol_data_call_list *unsol;

	unsol = g_ril_unsol_parse_data_call_list(NULL, data, &error);
	g_assert(unsol != NULL);
	g_ril_unsol_free_data_call_list(unsol);

	g_assert(error.type == OFONO_ERROR_TYPE_NO_ERROR &&
			error.error == 0);
}

static void test_signal_strength_valid(gconstpointer data)
{
	const signal_strength_test *test = data;
	int strength = g_ril_unsol_parse_signal_strength(NULL, &test->msg);
	g_assert(strength == test->strength);
}

static void test_unsol_response_new_sms_valid(gconstpointer data)
{
	struct unsol_sms_data *sms_data;

	sms_data = g_ril_unsol_parse_new_sms(NULL, data);

	g_assert(sms_data != NULL);
	g_assert(sms_data->data != NULL);
	g_assert(sms_data->length > 0);

	g_ril_unsol_free_sms_data(sms_data);
}

static void test_unsol_supp_svc_notif_valid(gconstpointer data)
{
	struct unsol_supp_svc_notif *unsol;

	unsol = g_ril_unsol_parse_supp_svc_notif(NULL,
						(struct ril_msg *) data);
	g_assert(unsol != NULL);
	g_ril_unsol_free_supp_svc_notif(unsol);
}

static void test_unsol_on_ussd_valid(gconstpointer data)
{
	struct unsol_ussd *unsol;

	unsol = g_ril_unsol_parse_ussd(NULL, (struct ril_msg *) data);

	g_assert(unsol != NULL);
	g_ril_unsol_free_ussd(unsol);
}
#endif

int main(int argc, char **argv)
{
	g_test_init(&argc, &argv, NULL);
/*
 * As all our architectures are little-endian except for
 * PowerPC, and the Binder wire-format differs slightly
 * depending on endian-ness, the following guards against test
 * failures when run on PowerPC.
 */
#if BYTE_ORDER == LITTLE_ENDIAN

	g_test_add_data_func("/testgrilunsol/gprs-context: "
				"invalid DATA_CALL_LIST_CHANGED Test 1",
				&unsol_data_call_list_changed_invalid_1,
				test_unsol_data_call_list_changed_invalid);

	g_test_add_data_func("/testgrilunsol/gprs-context: "
				"valid DATA_CALL_LIST_CHANGED Test 1",
				&unsol_data_call_list_changed_valid_1,
				test_unsol_data_call_list_changed_valid);

	g_test_add_data_func("/testgrilunsol/sms: "
				"valid RESPONSE_NEW_SMS Test 1",
				&unsol_response_new_sms_valid_1,
				test_unsol_response_new_sms_valid);

	g_test_add_data_func("/testgrilunsol/sms: "
				"valid RESPONSE_NEW_SMS Test 2",
				&unsol_response_new_sms_valid_2,
				test_unsol_response_new_sms_valid);

	g_test_add_data_func("/testgrilunsol/voicecall: "
				"valid SUPP_SVC_NOTIF Test 1",
				&unsol_supp_svc_notif_valid_1,
				test_unsol_supp_svc_notif_valid);

	g_test_add_data_func("/testgrilunsol/voicecall: "
				"valid SIGNAL_STRENGTH Test 1",
				&unsol_signal_strength_valid_1,
				test_signal_strength_valid);

	g_test_add_data_func("/testgrilunsol/ussd: "
				"valid ON_USSD Test 1",
				&unsol_on_ussd_valid_1,
				test_unsol_on_ussd_valid);

#endif
	return g_test_run();
}
