We can customize parsing by providing different options to the parser;

```js
const parser = require('fast-xml-parser');

const options = {
    ignoreAttributes : false
};
let jsonObj = parser.parse(xmlDataStr, options);
```

Let's understand each option in detail with necessary examples

## allowBooleanAttributes

To allow attributes without value. 

By default boolean attributes are ignored
```js
    const xmlData = `<root a="nice" checked><a>wow</a></root>`;

    const result = parser.parse(xmlData, {
        ignoreAttributes: false,
        attributeNamePrefix : "@_"
    });
```
Output
```json
{
    "root": {
        "@_a": "nice",
        "a": "wow"
    }
}
```
Once you allow boolean attributes, they are parsed and set to `true`.
```js
const xmlData = `<root a="nice" checked><a>wow</a></root>`;

const result = parser.parse(xmlData, {
    ignoreAttributes: false,
    attributeNamePrefix : "@_",
    allowBooleanAttributes: true
});
```
Output
```json
{
    "root": {
        "@_a": "nice",
        "@_checked": true,
        "a": "wow"
    }
}
```

## attributeNamePrefix

To recognize attributes in the JS object separately. You can prepend some string with each attribute name.

```js
const xmlData = `<root a="nice" ><a>wow</a></root>`;

const result = parser.parse(xmlData, {
    ignoreAttributes: false,
    attributeNamePrefix : "@_"
});
```
Output
```json
{
    "root": {
        "@_a": "nice",
        "a": "wow"
    }
}
```
## attrNodeName

To group all the attributes of a tag under given property name.

```js
const xmlData = `<root a="nice" b="very nice" ><a>wow</a></root>`;

const result = parser.parse(xmlData, {
    ignoreAttributes: false,
    attrNodeName : "@_"
});
```
Output
```json
{
    "root": {
        "@_": {
            "@_a": "nice",
            "@_b": "very nice"
        },
        "a": "wow"
    }
}
```

## ignoreAttributes

By default `ignoreAttributes` is set to `true`. It means, attributes are ignored by the parser. If you set any configuration related to attributes without setting `ignoreAttributes: false`, it is useless.

Eg
```js
const xmlData = `<root a="nice" ><a>wow</a></root>`;

const result = parser.parse(xmlData, {
    // ignoreAttributes: false,
    attributeNamePrefix : "@_"
});
```
Output
```json
{
    "root": {
        "a": "wow"
    }
}
```

## ignoreNameSpace

Remove namespace string from tag and attribute names.

Default is `ignoreNameSpace: false`
```js
const xmlData = `<root some:a="nice" ><any:a>wow</any:a></root>`;

const result = parser.parse(xmlData, {
    ignoreAttributes: false,
    attributeNamePrefix : "@_",
    //ignoreNameSpace: true
});
```
Output
```json
{
    "root": {
        "@_some:a": "nice",
        "any:a": "wow"
    }
}
```

Setting `ignoreNameSpace: true`
```js
const xmlData = `<root some:a="nice" ><any:a>wow</any:a></root>`;

const result = parser.parse(xmlData, {
    ignoreAttributes: false,
    attributeNamePrefix : "@_",
    ignoreNameSpace: true
});
```
Output
```json
{
    "root": {
        "@_a": "nice",
        "a": "wow"
    }
}
```