/* 
 * Copyright (c) 2011, 2013 Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "mforms/mforms.h"
#include "wf_view.h"
#include "wf_find_panel.h"

#include "base/log.h"
#include "ConvUtils.h"

using namespace Windows::Forms;

using namespace MySQL::Forms;
using namespace MySQL::Utilities;

DEFAULT_LOG_DOMAIN(DOMAIN_MFORMS_WRAPPER)

//----------------- Static functions ---------------------------------------------------------------

bool FindPanelImpl::create(mforms::FindPanel* panel)
{
  FindPanelImpl^ find_panel_impl = gcnew FindPanelImpl(panel);

  if (find_panel_impl != nullptr)
  {
    find_panel_impl->nativeFindPanel = ViewImpl::create<FindPanel>(panel, find_panel_impl);
    find_panel_impl->nativeFindPanel->Backend = find_panel_impl;

    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

int FindPanelImpl::perform_action(mforms::FindPanel* panel, mforms::FindPanelAction action)
{
  FindPanelImpl^ native_panel = (FindPanelImpl^)ObjectImpl::FromUnmanaged(panel);
  if (native_panel != nullptr)
    return native_panel->DoFindReplace(action);

  return 0;
}

//--------------------------------------------------------------------------------------------------

void FindPanelImpl::focus(mforms::FindPanel* panel)
{
  FindPanelImpl^ native_panel = (FindPanelImpl^)ObjectImpl::FromUnmanaged(panel);
  if (native_panel != nullptr)
    native_panel->nativeFindPanel->FocusSearchField();
}

//--------------------------------------------------------------------------------------------------

void FindPanelImpl::enable_replace(mforms::FindPanel* panel, bool flag)
{
  FindPanelImpl^ native_panel = (FindPanelImpl^)ObjectImpl::FromUnmanaged(panel);
  if (native_panel != nullptr)
    native_panel->nativeFindPanel->ShowReplace = flag;
}

//----------------- Actual implementation ----------------------------------------------------------

FindPanelImpl::FindPanelImpl(mforms::FindPanel* panel)
  : ViewImpl(panel)
{
}

//--------------------------------------------------------------------------------------------------

int FindPanelImpl::DoFindReplace(mforms::FindPanelAction action)
{
  mforms::FindPanel* panel = get_backend_control<mforms::FindPanel>(nativeFindPanel);
  mforms::CodeEditor* editor = panel->get_editor();

  if (editor == NULL)
    return 0;

  std::string search_text = NativeToCppStringRaw(nativeFindPanel->SearchText);
  std::string replace_text = NativeToCppStringRaw(nativeFindPanel->ReplaceText);

  mforms::FindFlags flags = mforms::FindDefault;
  if (!nativeFindPanel->IgnoreCase)
    flags |= mforms::FindMatchCase;
  if (nativeFindPanel->Wrap)
    flags |= mforms::FindWrapAround;
  if (nativeFindPanel->WholeWords)
    flags |= mforms::FindWholeWords;
  if (nativeFindPanel->RegularExpression)
    flags |= mforms::FindRegex;

  switch (action)
  {
  case mforms::FindNext:
    return editor->find_and_highlight_text(search_text, flags, true, false);
    break;
  case mforms::FindPrevious:
    return editor->find_and_highlight_text(search_text, flags, true, true);
    break;
  case mforms::Replace:
  {
    int start, length;
    editor->get_selection(start, length);
    if (length > 0)
      editor->replace_selected_text(replace_text);
    break;
  }
  case mforms::ReplaceAndFind:
    editor->replace_selected_text(replace_text);
    return editor->find_and_highlight_text(search_text, flags, true, false);
    break;
  case mforms::FindAndReplace:
    return editor->find_and_replace_text(search_text, replace_text, flags, false);
    break;
  case mforms::ReplaceAll:
    return editor->find_and_replace_text(search_text, replace_text, flags, true);
    break;
  }

  return 0;
}

//--------------------------------------------------------------------------------------------------

void FindPanelImpl::Close()
{
  mforms::FindPanel* panel = get_backend_control<mforms::FindPanel>(nativeFindPanel);
  mforms::CodeEditor* editor = panel->get_editor();
  editor->hide_find_panel();
}

//--------------------------------------------------------------------------------------------------

int FindPanelImpl::FindReplaceAction(MySQL::Utilities::FindPanelAction action)
{
  return DoFindReplace((mforms::FindPanelAction)action);
}

//--------------------------------------------------------------------------------------------------

