/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     eval.c                                                         */
/*                                                                          */
/* description:  routines for evaluating fe-functions and derivatives       */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  some routines for evaluation of a finite element function, its gradient */
/*  and second derivatives; all those with _fast use the preevaluated       */
/*  basis functions at that point.                                          */
/*                                                                          */
/*  PLEASE NOTE: The _fast routines rely on zero values in the unused       */
/*  barycentric coordinates and derivatives. This was done to maintain the  */
/*  calling syntax (the routines don't know the space dimension) and to     */
/*  enable loop unrolling with constant index boundaries.                   */
/*--------------------------------------------------------------------------*/

REAL eval_uh(const REAL_B lambda, const REAL *uh_loc,
	     const BAS_FCTS *b)
{
  int       i;
  BAS_FCT   **phi = b->phi;
  REAL      val = 0.0;

  for (i = 0; i < b->n_bas_fcts; i++)
    val += uh_loc[i]*phi[i](lambda);

  return(val);
}

const REAL *eval_uh_d(const REAL lambda[N_LAMBDA], const REAL_D *uh_loc,
		      const BAS_FCTS *b, REAL_D values)
{
  static REAL_D Values;
  int       i, n;
  BAS_FCT   **phi = b->phi;
  REAL      phil, *val = values ? values : Values;

  for (n = 0; n < DIM_OF_WORLD; n++)
    val[n] = 0;

  for (i = 0; i < b->n_bas_fcts; i++)
  {
    phil = phi[i](lambda);
    for (n = 0; n < DIM_OF_WORLD; n++)
      val[n] += uh_loc[i][n]*phil;
  }

  return((const REAL *) val);
}


REAL eval_uh_fast(const REAL *uh_loc, const REAL *phi_val, int n_bas_fcts)
{
  int       i;
  REAL      val = 0.0;

  for (i = 0; i < n_bas_fcts; i++)
    val += uh_loc[i]*phi_val[i];

  return(val);
}

const REAL *eval_uh_d_fast(const REAL_D *uh_loc, const REAL *phi_val,
			   int n_bas_fcts, REAL_D values)
{
  static REAL_D  Values;
  int            i, n;
  REAL           *val = values ? values : Values;

  for (n = 0; n < DIM_OF_WORLD; n++)
    val[n] = 0.0;

  for (i = 0; i < n_bas_fcts; i++) 
    for (n = 0; n < DIM_OF_WORLD; n++)
      val[n] += uh_loc[i][n]*phi_val[i];

  return((const REAL *) val);
}

const REAL *eval_grd_uh(const REAL lambda[N_LAMBDA], 
			const REAL_D Lambda[N_LAMBDA],
			const REAL *uh_loc, const BAS_FCTS *b, REAL_D grd_uh)
{
  static REAL_D  grd;
  const REAL     *grd_b;
  int            i, j, dim = b->dim;
  GRD_BAS_FCT    **grd_phi = b->grd_phi;
  REAL           *val, grd1[N_LAMBDA] = {0};
  
  val = grd_uh ? grd_uh : grd;
 
  for (i = 0; i < b->n_bas_fcts; i++)
  {
    grd_b = grd_phi[i](lambda);
    for (j = 0; j < dim+1; j++)
      grd1[j] += uh_loc[i]*grd_b[j];
  }

  for (i = 0; i < DIM_OF_WORLD; i++)
  {
    for (val[i] = j = 0; j < dim+1; j++)
      val[i] += Lambda[j][i]*grd1[j];
  }
  
  return((const REAL *) val);
}

const REAL_D *eval_grd_uh_d(const REAL lambda[N_LAMBDA], 
			    const REAL_D Lambda[N_LAMBDA],
			    const REAL_D *uh_loc, const BAS_FCTS *b,
			    REAL_DD grd_uh)
{
  static REAL_DD  grd;
  const REAL      *grd_b;
  int             i, j, n, dim = b->dim;
  GRD_BAS_FCT     **grd_phi = b->grd_phi;
  REAL_D          *val;
  REAL            grd1[DIM_OF_WORLD][N_LAMBDA] = {{0}};
  
  val = grd_uh ? grd_uh : grd;
 
  for (i = 0; i < b->n_bas_fcts; i++)
  {
    grd_b = grd_phi[i](lambda);
    for (j = 0; j < dim+1; j++)
      for (n = 0; n < DIM_OF_WORLD; n++)
	grd1[n][j] += uh_loc[i][n]*grd_b[j];
  }

  for (i = 0; i < DIM_OF_WORLD; i++)
  {
      for (n = 0; n < DIM_OF_WORLD; n++)
	for (val[n][i] = j = 0; j < dim+1; j++)
	  val[n][i] += Lambda[j][i]*grd1[n][j];
  }
  
  return((const REAL_D *) val);
}


const REAL *eval_grd_uh_fast(const REAL_D Lambda[N_LAMBDA], 
			     const REAL *uh_loc,
			     const REAL (*grd_phi)[N_LAMBDA],
			     int n_bas_fcts, REAL_D grd_uh)
{
  static REAL_D  grd;
  int            i, j;
  REAL           *val, grd1[N_LAMBDA] = {0};
  
  val = grd_uh ? grd_uh : grd;

  for (i = 0; i < n_bas_fcts; i++)
    for (j = 0; j < N_LAMBDA; j++)
      grd1[j] += uh_loc[i]*grd_phi[i][j];


  for (i = 0; i < DIM_OF_WORLD; i++)
    for (val[i] = j = 0; j < N_LAMBDA; j++)
      val[i] += Lambda[j][i]*grd1[j];
  
  return((const REAL *) val);
}

const REAL_D *eval_grd_uh_d_fast(const REAL_D Lambda[N_LAMBDA], 
				 const REAL_D *uh_loc, 
				 const REAL (*grd_phi)[N_LAMBDA],
				 int n_bas_fcts, REAL_DD grd_uh)
{
  static REAL_DD  grd;
  int             i, j, n;
  REAL_D          *val;
  REAL            grd1[N_LAMBDA];
  
  val = grd_uh ? grd_uh : grd;

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    for (j = 0; j < N_LAMBDA; j++)
      grd1[j] = 0.0;

    for (i = 0; i < n_bas_fcts; i++)
    {
      for (j = 0; j < N_LAMBDA; j++)
	grd1[j] += uh_loc[i][n]*grd_phi[i][j];
    }

    for (i = 0; i < DIM_OF_WORLD; i++)
    {
      for (val[n][i] = j = 0; j < N_LAMBDA; j++)
	val[n][i] += Lambda[j][i]*grd1[j];
    }
  }
  
  return((const REAL_D *) val);
}

const REAL_D *eval_D2_uh(const REAL lambda[N_LAMBDA],
			 const REAL_D Lambda[N_LAMBDA],
			 const REAL *uh_loc, const BAS_FCTS *b, REAL_DD D2_uh)
{
  static REAL_DD D2;
  const REAL     (*D2_b)[N_LAMBDA];
  int            i, j, k, l, dim = b->dim;
  D2_BAS_FCT     **D2_phi = b->D2_phi;
  REAL           (*val)[DIM_OF_WORLD], D2_tmp[N_LAMBDA][N_LAMBDA] = {{0}};
  
  val = D2_uh ? D2_uh : D2;
  
  for (i = 0; i < b->n_bas_fcts; i++)
  {
    D2_b = D2_phi[i](lambda);
    for (k = 0; k < dim+1; k++)
      for (l = k; l < dim+1; l++)
      D2_tmp[k][l] += uh_loc[i]*D2_b[k][l];
  }

  for (i = 0; i < DIM_OF_WORLD; i++)
    for (j = i; j < DIM_OF_WORLD; j++)
    {
      val[i][j] = 0.0;
      for (k = 0; k < dim+1; k++)
      {
	val[i][j] += Lambda[k][i]*Lambda[k][j]*D2_tmp[k][k];
	for (l = k+1; l < dim+1; l++)
	  val[i][j] +=
	    (Lambda[k][i]*Lambda[l][j] + Lambda[l][i]*Lambda[k][j])*D2_tmp[k][l];
      }
      val[j][i] = val[i][j];
  }
  
  return((const REAL_D *) val);
}

const REAL_DD *eval_D2_uh_d(const REAL lambda[N_LAMBDA],
			    const REAL_D Lambda[N_LAMBDA],
			    const REAL_D *uh_loc, const BAS_FCTS *b, 
			    REAL_DD *D2_uh)
{
  static REAL_DD  D2[DIM_OF_WORLD];
  const REAL      (*D2_b)[N_LAMBDA];
  int             i, j, k, l, n, dim = b->dim;
  D2_BAS_FCT      **D2_phi = b->D2_phi;
  REAL            D2_tmp[N_LAMBDA][N_LAMBDA];
  REAL_DD         *val;
  
  val = D2_uh ? D2_uh : D2;
  
  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    for (k = 0; k < dim + 1; k++)
      for (l = k; l < dim + 1; l++)
	D2_tmp[k][l] = 0.0;

    for (i = 0; i < b->n_bas_fcts; i++)
    {
      D2_b = D2_phi[i](lambda);
      
      for (k = 0; k < dim+1; k++)
	for (l = k; l < dim+1; l++)
	  D2_tmp[k][l] += uh_loc[i][n]*D2_b[k][l];
    }

    for (i = 0; i < DIM_OF_WORLD; i++)
      for (j = i; j < DIM_OF_WORLD; j++)
      {
	val[n][i][j] = 0.0;
	for (k = 0; k < dim+1; k++)
	{
	  val[n][i][j] += Lambda[k][i]*Lambda[k][j]*D2_tmp[k][k];
	  for (l = k+1; l < dim+1; l++)
	    val[n][i][j] += 
	      (Lambda[k][i]*Lambda[l][j] + Lambda[l][i]*Lambda[k][j])
	      *
	      D2_tmp[k][l];
	}
      }
  }
  return((const REAL_DD *) val);
}

const REAL_D *eval_D2_uh_fast(const REAL_D Lambda[N_LAMBDA], 
			      const REAL *uh_loc, 
			      const REAL (*D2_phi)[N_LAMBDA][N_LAMBDA],
			      int n_bas_fcts, REAL_DD D2_uh)
{
  static REAL_DD   D2;
  int              i, j, k, l;
  REAL             (*val)[DIM_OF_WORLD], D2_tmp[N_LAMBDA][N_LAMBDA] = {{0}};
  
  val = D2_uh ? D2_uh : D2;

  for (i = 0; i < n_bas_fcts; i++)
    for (k = 0; k < N_LAMBDA; k++)
      for (l = k; l < N_LAMBDA; l++)
	D2_tmp[k][l] += uh_loc[i]*D2_phi[i][k][l];


  for (i = 0; i < DIM_OF_WORLD; i++)
  {
    for (j = i; j < DIM_OF_WORLD; j++)
    {
      val[i][j] = 0.0;
      for (k = 0; k < N_LAMBDA; k++)
      {
	val[i][j] += Lambda[k][i]*Lambda[k][j]*D2_tmp[k][k];
	for (l = k+1; l < N_LAMBDA; l++)
	{
	  val[i][j] +=
	    (Lambda[k][i]*Lambda[l][j] + Lambda[l][i]*Lambda[k][j])*D2_tmp[k][l];
	}
      }
      val[j][i] = val[i][j]; /* the Hessian is symmetric */
    }
  }
  
  return((const REAL_D *) val);
}

const REAL_DD *eval_D2_uh_d_fast(const REAL_D Lambda[N_LAMBDA], 
				 const REAL_D *uh_loc, 
				 const REAL (*D2_phi)[N_LAMBDA][N_LAMBDA],
				 int n_bas_fcts, REAL_DD *D2_uh)
{
  static REAL_DD   D2[DIM_OF_WORLD];
  int              i, j, k, l, n;
  REAL             D2_tmp[N_LAMBDA][N_LAMBDA];
  REAL_DD          *val;

  val = D2_uh ? D2_uh : D2;
 
  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    for (k = 0; k < N_LAMBDA; k++)
      for (l = k; l < N_LAMBDA; l++) {
	D2_tmp[k][l] = 0.0;
	
	for (i = 0; i < n_bas_fcts; i++)
	  D2_tmp[k][l] += uh_loc[i][n]*D2_phi[i][k][l];
      }

    for (i = 0; i < DIM_OF_WORLD; i++)
    {
      for (j = i; j < DIM_OF_WORLD; j++)
      {
	val[n][i][j] = 0.0;
	for (k = 0; k < N_LAMBDA; k++)
	{
	  val[n][i][j] += Lambda[k][i]*Lambda[k][j]*D2_tmp[k][k];
	  for (l = k+1; l < N_LAMBDA; l++)
	  {
	    val[n][i][j] +=
	      (Lambda[k][i]*Lambda[l][j] + Lambda[l][i]*Lambda[k][j])
	      *
	      D2_tmp[k][l];
	  }
	}
	val[n][j][i] = val[n][i][j];
      }
    }
  }
  return((const REAL_DD *) val);
}

/*--------------------------------------------------------------------------*/
/*  standard routines for getting values of a finite element function and   */
/*  its first (and second) derivatives at the quadrature points             */
/*--------------------------------------------------------------------------*/

const REAL *uh_at_qp(const QUAD_FAST *fast, const REAL *uh_loc, REAL *vec)
{
  FUNCNAME("uh_at_qp");
  static REAL    *quad_vec = nil;
  static size_t  size = 0;
  REAL           *val, **phi;
  int            i, j;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL);
      size = new_size;
    }
    val = quad_vec;
  }

  phi = fast->phi;
  for (i = 0; i < fast->n_points; i++)
    for (val[i] = j = 0; j < fast->n_bas_fcts; j++)
      val[i] += uh_loc[j]*phi[i][j];
    
  return((const REAL *) val);
}

const REAL_D *grd_uh_at_qp(const QUAD_FAST *fast,
			   const REAL_BD Lambda,
			   const REAL *uh_loc, REAL_D *vec)
{
  FUNCNAME("grd_uh_at_qp");
  static REAL_D  *quad_vec = nil;
  static size_t  size = 0;
  REAL_D         *val;
  REAL           (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int            i, j, k, l, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_D);
      size = new_size;
    }
    val = quad_vec;
  }

  for (i = 0; i < fast->n_points; i++)
  {
    grd_phi = fast->grd_phi[i];

    for (k = 0; k < dim+1; k++) {
      grd1[k] = 0.0;

      for (j = 0; j < fast->n_bas_fcts; j++)
	grd1[k] += uh_loc[j]*grd_phi[j][k];
    }

    for (l = 0; l < DIM_OF_WORLD; l++)
    {
      for (val[i][l] = k = 0; k < dim+1; k++)
	val[i][l] += Lambda[k][l]*grd1[k];
    }
  }

  return((const REAL_D *) val);
}

const REAL_D *param_grd_uh_at_qp(const QUAD_FAST *fast,
				 REAL_D Lambda[][N_LAMBDA],
				 const REAL *uh_loc, REAL_D *vec)
{
  FUNCNAME("param_grd_uh_at_qp");
  static REAL_D  *quad_vec = nil;
  static size_t  size = 0;
  REAL_D         *val;
  REAL           (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int            i, j, k, l, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_D);
      size = new_size;
    }
    val = quad_vec;
  }

  for (i = 0; i < fast->n_points; i++)
  {
    grd_phi = fast->grd_phi[i];

    for (k = 0; k < dim+1; k++) {
      grd1[k] = 0.0;

      for (j = 0; j < fast->n_bas_fcts; j++)
	grd1[k] += uh_loc[j]*grd_phi[j][k];
    }

    for (l = 0; l < DIM_OF_WORLD; l++)
    {
      for (val[i][l] = k = 0; k < dim+1; k++)
	val[i][l] += Lambda[i][k][l]*grd1[k];
    }
  }

  return((const REAL_D *) val);
}

const REAL_DD *D2_uh_at_qp(const QUAD_FAST *fast,
			   const REAL_D Lambda[N_LAMBDA],
			   const REAL *uh_loc, REAL_DD *vec)
{
  FUNCNAME("D2_uh_at_qp");
  static REAL_DD  *quad_vec = nil;
  static size_t   size = 0;
  REAL_DD         *val;
  REAL            (*D2_phi)[N_LAMBDA][N_LAMBDA], D2_tmp[N_LAMBDA][N_LAMBDA];
  int             i, j, k, l, iq, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_DD);
      size = new_size;
    }
    val = quad_vec;
  }

  for (iq = 0; iq < fast->n_points; iq++)
  {
    D2_phi = fast->D2_phi[iq];

    for (k = 0; k < dim+1; k++)
      for (l = k; l < dim+1; l++) {
	D2_tmp[k][l] = 0.0;

	for (i = 0; i < fast->n_bas_fcts; i++)
	  D2_tmp[k][l] += uh_loc[i]*D2_phi[i][k][l];
      }

    for (i = 0; i < DIM_OF_WORLD; i++)
      for (j = i; j < DIM_OF_WORLD; j++)
      {
	val[iq][i][j] = 0.0;

	for (k = 0; k < dim+1; k++)
	{
	  val[iq][i][j] += Lambda[k][i]*Lambda[k][j]*D2_tmp[k][k];
	  for (l = k+1; l < dim+1; l++)
	    val[iq][i][j] += 
	      (Lambda[k][i]*Lambda[l][j] + Lambda[l][i]*Lambda[k][j])
	      *
	      D2_tmp[k][l];
	}
      }
  }

  return((const REAL_DD *) val);
}

const REAL_DD *param_D2_uh_at_qp(const QUAD_FAST *fast,
				 const REAL_D Lambda[][N_LAMBDA],
				 const REAL *uh_loc, REAL_DD *vec)
{
  FUNCNAME("param_D2_uh_at_qp");
  ERROR_EXIT("Not yet implemented, sorry.\n");

  return nil;
}

const REAL_D *uh_d_at_qp(const QUAD_FAST *fast, const REAL_D *uh_loc,
			 REAL_D *vec)
{
  FUNCNAME("uh_d_at_qp");
  static REAL_D  *quad_vec = nil;
  static size_t  size = 0;
  REAL           **phi;
  REAL_D         *val;
  int            i, j, k;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_D);
      size = new_size;
    }
    val = quad_vec;
  }

  phi = fast->phi;
  for (i = 0; i < fast->n_points; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      for (val[i][k] = j = 0; j < fast->n_bas_fcts; j++)
	val[i][k] += uh_loc[j][k]*phi[i][j];
    
  return((const REAL_D *) val);
}

const REAL_DD *grd_uh_d_at_qp(const QUAD_FAST *fast,
			      const REAL_D Lambda[N_LAMBDA], 
			      const REAL_D *uh_loc, REAL_DD *vec)
{
  FUNCNAME("grd_uh_d_at_qp");
  static REAL_DD  *quad_vec = nil;
  static size_t   size = 0;
  REAL_DD         *val;
  REAL            (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int             i, j, k, l, m, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_DD);
      size = new_size;
    }
    val = quad_vec;
  }

  for (i = 0; i < fast->n_points; i++)
  {
    grd_phi = fast->grd_phi[i];

    for (k = 0; k < DIM_OF_WORLD; k++)
    {
      for (m = 0; m < dim+1; m++) {
	grd1[m] = 0.0;

	for (j = 0; j < fast->n_bas_fcts; j++)
	  grd1[m] += uh_loc[j][k]*grd_phi[j][m];
      }

      for (l = 0; l < DIM_OF_WORLD; l++)
      {
	for (val[i][k][l] = m = 0; m < dim+1; m++)
	  val[i][k][l] += Lambda[m][l]*grd1[m];
      }
    }
  }
  return((const REAL_DD *) val);
}

const REAL_DD *param_grd_uh_d_at_qp(const QUAD_FAST *fast,
				    REAL_D Lambda[][N_LAMBDA], 
				    const REAL_D *uh_loc, REAL_DD *vec)
{
  FUNCNAME("param_grd_uh_d_at_qp");
  static REAL_DD  *quad_vec = nil;
  static size_t   size = 0;
  REAL_DD         *val;
  REAL            (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int             i, j, k, l, m, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_DD);
      size = new_size;
    }
    val = quad_vec;
  }

  for (i = 0; i < fast->n_points; i++)
  {
    grd_phi = fast->grd_phi[i];

    for (k = 0; k < DIM_OF_WORLD; k++)
    {
      for (m = 0; m < dim+1; m++) {
	grd1[m] = 0.0;

	for (j = 0; j < fast->n_bas_fcts; j++)
	  grd1[m] += uh_loc[j][k]*grd_phi[j][m];
      }

      for (l = 0; l < DIM_OF_WORLD; l++)
      {
	for (val[i][k][l] = m = 0; m < dim+1; m++)
	  val[i][k][l] += Lambda[i][m][l]*grd1[m];
      }
    }
  }
  return((const REAL_DD *) val);
}

const REAL_DD (*D2_uh_d_at_qp(const QUAD_FAST *fast,
			      const REAL_D grd_lam[N_LAMBDA],
			      const REAL_D *uh_loc, 
			      REAL_DD (*vec)[DIM_OF_WORLD]))[DIM_OF_WORLD]
{
  FUNCNAME("D2_uh_d_at_qp");
  static REAL_DD  (*quad_vec)[DIM_OF_WORLD] = nil;
  static size_t   size = 0;
  REAL_DD         (*val)[DIM_OF_WORLD];
  REAL            (*D2_phi)[N_LAMBDA][N_LAMBDA],D2_tmp[N_LAMBDA][N_LAMBDA];
  int             i, j, k, l, n, iq, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->n_points);
      quad_vec = (REAL_DD (*) [DIM_OF_WORLD])
	MEM_REALLOC(quad_vec,size*DIM_OF_WORLD,new_size*DIM_OF_WORLD,REAL_DD);
      size = new_size;
    }
    val = quad_vec;
  }

  for (iq = 0; iq < fast->n_points; iq++)
  {
    D2_phi = fast->D2_phi[iq];

    for (n = 0; n < DIM_OF_WORLD; n++)
    {
      for (k = 0; k < dim+1; k++)
	for (l = k; l < dim+1; l++) {
	  D2_tmp[k][l] = 0.0;

	  for (i = 0; i < fast->n_bas_fcts; i++)
	    D2_tmp[k][l] += uh_loc[i][n]*D2_phi[i][k][l];
	}

      for (i = 0; i < DIM_OF_WORLD; i++)
	for (j = i; j < DIM_OF_WORLD; j++)
	{
	  val[iq][n][i][j] = 0.0;
	  for (k = 0; k < dim+1; k++)
	  {
	    val[iq][n][i][j] += grd_lam[k][i]*grd_lam[k][j]*D2_tmp[k][k];
	    for (l = k+1; l < dim+1; l++)
	      val[iq][n][i][j] +=
		(grd_lam[k][i]*grd_lam[l][j] + grd_lam[l][i]*grd_lam[k][j])
		*
		D2_tmp[k][l];
	  }
	  val[iq][n][j][i] =  val[iq][n][i][j];
	}
    }
  }

  return((const REAL_DD (*) [DIM_OF_WORLD]) val);
}

const REAL_DD (*param_D2_uh_d_at_qp(const QUAD_FAST *fast,
				    const REAL_D grd_lam[][N_LAMBDA],
				    const REAL_D *uh_loc, 
			    REAL_DD (*vec)[DIM_OF_WORLD]))[DIM_OF_WORLD]
{
  FUNCNAME("param_D2_uh_d_at_qp");
  ERROR_EXIT("Not yet implemented, sorry.\n");

  return nil;
}

REAL eval_div_uh_d(const REAL lambda[N_LAMBDA], const REAL_D Lambda[N_LAMBDA],
		   const REAL_D *uh_loc, const BAS_FCTS *b)
{
  REAL            div;
  const REAL      *grd_b;
  int             i, j, n, dim = b->dim;
  GRD_BAS_FCT     **grd_phi = b->grd_phi;
  REAL            grd1[DIM_OF_WORLD][N_LAMBDA] = {{0}};
  
  for (i = 0; i < b->n_bas_fcts; i++)
  {
    grd_b = grd_phi[i](lambda);
    for (j = 0; j < dim+1; j++)
      for (n = 0; n < DIM_OF_WORLD; n++)
	grd1[n][j] += uh_loc[i][n]*grd_b[j];
  }

  for (div = n = 0; n < DIM_OF_WORLD; n++)
    for (j = 0; j < dim+1; j++)
      div += Lambda[j][n]*grd1[n][j];
  
  return(div);
}


REAL eval_div_uh_d_fast(const REAL_D Lambda[N_LAMBDA], const REAL_D *uh_loc, 
			const REAL (*grd_phi)[N_LAMBDA], int n_bas_fcts)
{
  REAL      div;
  int       i, j, n;
  REAL      grd1[N_LAMBDA];
  
  for (div = n = 0; n < DIM_OF_WORLD; n++)
  {
    for (j = 0; j < N_LAMBDA; j++) {
      grd1[j] = 0.0;

      for (i = 0; i < n_bas_fcts; i++)
	grd1[j] += uh_loc[i][n]*grd_phi[i][j];
    }

    for (j = 0; j < N_LAMBDA; j++)
      div += Lambda[j][n]*grd1[j];
  }
  
  return(div);
}

const REAL *div_uh_d_at_qp(const QUAD_FAST *fast,
			   const REAL_D Lambda[N_LAMBDA], 
			   const REAL_D *uh_loc, REAL *vec)
{
  FUNCNAME("div_uh_d_at_qp");
  static REAL     *quad_vec = nil;
  static size_t   size = 0;
  REAL            *val, div;
  REAL            (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int             i, j, n, m, dim = fast->quad->dim;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL);
      size = new_size;
    }
    val = quad_vec;
  }

  for (m = 0; m < fast->n_points; m++)
  {
    grd_phi = fast->grd_phi[m];

    for (div = n = 0; n < DIM_OF_WORLD; n++)
    {
      for (j = 0; j < dim+1; j++) {
	grd1[j] = 0.0;

	for (i = 0; i < fast->n_bas_fcts; i++)
	  grd1[j] += uh_loc[i][n]*grd_phi[i][j];
      }

      for (j = 0; j < dim+1; j++)
	div += Lambda[j][n]*grd1[j];
    }
    val[m] = div;
  }

  return((const REAL *) val);
}

const REAL *param_div_uh_d_at_qp(const QUAD_FAST *fast,
				 const REAL_D Lambda[][N_LAMBDA], 
				 const REAL_D *uh_loc, REAL *vec)
{
  FUNCNAME("param_div_uh_d_at_qp");
  static REAL     *quad_vec = nil;
  static size_t   size = 0;
  REAL            *val, div;
  REAL            (*grd_phi)[N_LAMBDA], grd1[N_LAMBDA];
  int             i, j, n, m, dim = fast->quad->dim;

  if (vec) {
    val = vec;
  }
  else
  {
    if (size < (size_t) fast->quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, fast->quad->n_points);
      
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL);
      size = new_size;
    }
    val = quad_vec;
  }

  for (m = 0; m < fast->n_points; m++)
  {
    grd_phi = fast->grd_phi[m];

    for (div = n = 0; n < DIM_OF_WORLD; n++)
    {
      for (j = 0; j < dim+1; j++) {
	grd1[j] = 0.0;

	for (i = 0; i < fast->n_bas_fcts; i++)
	  grd1[j] += uh_loc[i][n]*grd_phi[i][j];
      }

      for (j = 0; j < dim+1; j++)
	div += Lambda[m][j][n]*grd1[j];
    }
    val[m] = div;
  }

  return((const REAL *) val);
}

/*--------------------------------------------------------------------------*/
/* calculation H1 or L2 Norm of a finite element function                   */
/*--------------------------------------------------------------------------*/

typedef struct eval_traverse_data {

  REAL                norm;
  const QUAD_FAST     *quad_fast;

  const REAL *(*get_real_vec)(const EL *, const DOF_REAL_VEC *, REAL *);
  const DOF_REAL_VEC  *uh;

  const REAL_D *(*get_real_d_vec)(const EL *, const DOF_REAL_D_VEC *, REAL_D *);
  const DOF_REAL_D_VEC  *uh_d;

} EVAL_TRAVERSE_DATA;

static void H1_norm_uh_fct(const EL_INFO *el_info, void *data)
{
  EVAL_TRAVERSE_DATA *ud = (EVAL_TRAVERSE_DATA *)data;
  REAL           det[MAX_N_QUAD_POINTS], norm2, normT;
  REAL_D         Lambda[MAX_N_QUAD_POINTS][N_LAMBDA];
  const REAL    *uh_loc;
  const REAL_D  *grduh_vec;
  PARAMETRIC    *parametric = el_info->mesh->parametric;
  int            iq, j, dim = el_info->mesh->dim;

  if(parametric) {
    parametric->init_element(el_info, parametric);
    parametric->grd_lambda(el_info, ud->quad_fast->quad, 0, nil, Lambda, det);

    uh_loc = (*ud->get_real_vec)(el_info->el, ud->uh, nil);
    grduh_vec = param_grd_uh_at_qp(ud->quad_fast, Lambda, uh_loc, nil);

    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = j = 0; j < DIM_OF_WORLD; j++)
	norm2 += SQR(grduh_vec[iq][j]);
      
      normT += det[iq] * ud->quad_fast->w[iq] * norm2;
    }
    ud->norm += normT;
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_grd_lambda_1d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_grd_lambda_2d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_grd_lambda_3d(el_info, Lambda[0]);
#endif
#endif
    }
 
    uh_loc = (*ud->get_real_vec)(el_info->el, ud->uh, nil);
    grduh_vec = grd_uh_at_qp(ud->quad_fast, (const REAL_D*) Lambda[0],
			     uh_loc, nil);
    
    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = j = 0; j < DIM_OF_WORLD; j++)
	norm2 += SQR(grduh_vec[iq][j]);
      
      normT += ud->quad_fast->w[iq]*norm2;
    }
    ud->norm += det[0] * normT;
  }

  return;
}


REAL H1_norm_uh(const QUAD *quad, const DOF_REAL_VEC *u_h)
{
  FUNCNAME("H1_norm_uh");
  EVAL_TRAVERSE_DATA td[1] = {{0}};
  int      deg;

  if (!(td->uh = u_h))
  {
    ERROR("no DOF vector u_h; returning 0.0\n");
    return(0.0);
  }
  if (!quad)
  {
    deg = 2*td->uh->fe_space->bas_fcts->degree-2;
    quad = get_quadrature(td->uh->fe_space->mesh->dim, deg);
  }
  td->quad_fast = get_quad_fast(td->uh->fe_space->bas_fcts, quad, INIT_GRD_PHI);
  td->get_real_vec = td->uh->fe_space->bas_fcts->get_real_vec;

  td->norm = 0.0;
  mesh_traverse(td->uh->fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS,
		H1_norm_uh_fct, td);

  return(sqrt(td->norm));
}

static void L2_norm_uh_fct(const EL_INFO *el_info, void *data)
{
  EVAL_TRAVERSE_DATA *ud = (EVAL_TRAVERSE_DATA *)data;
  REAL        det[MAX_N_QUAD_POINTS], normT;
  const REAL  *uh_loc, *uh_vec;
  int         iq;
  int         dim = el_info->mesh->dim;
  PARAMETRIC *parametric = el_info->mesh->parametric;

  if (parametric) {
    parametric->init_element(el_info, parametric);
    parametric->det(el_info, ud->quad_fast->quad, 0, nil, det);

    uh_loc = (*ud->get_real_vec)(el_info->el, ud->uh, nil);
    uh_vec = uh_at_qp(ud->quad_fast, uh_loc, nil);

    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      normT += det[iq] * ud->quad_fast->w[iq] * SQR(uh_vec[iq]);
    }
    ud->norm += normT;
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_det_1d(el_info);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_det_2d(el_info);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_det_3d(el_info);
#endif
#endif
    }
    
    uh_loc = (*ud->get_real_vec)(el_info->el, ud->uh, nil);
    uh_vec = uh_at_qp(ud->quad_fast, uh_loc, nil);

    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      normT += ud->quad_fast->w[iq]*SQR(uh_vec[iq]);
    }
    ud->norm += det[0] * normT;
  }

  return;
}

REAL L2_norm_uh(const QUAD *quad, const DOF_REAL_VEC *u_h)
{
  FUNCNAME("L2_norm_uh");
  EVAL_TRAVERSE_DATA td[1] = {{0}};
  int      deg;

  if (!(td->uh = u_h))
  {
    ERROR("no DOF vector u_h; returning 0.0\n");
    return(0.0);
  }
  if (!quad)
  {
    deg = 2*td->uh->fe_space->bas_fcts->degree;
    quad = get_quadrature(td->uh->fe_space->mesh->dim, deg);
  }
  td->quad_fast = get_quad_fast(td->uh->fe_space->bas_fcts, quad, INIT_PHI);
  td->get_real_vec = td->uh->fe_space->bas_fcts->get_real_vec;

  td->norm = 0.0;
  mesh_traverse(td->uh->fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS,
		L2_norm_uh_fct, td);

  return(sqrt(td->norm));
}

/*--------------------------------------------------------------------------*/
/* calculation H1 or L2 Norm of a vector valued finite element function     */
/*--------------------------------------------------------------------------*/

static void H1_norm_uh_d_fct(const EL_INFO *el_info, void *data)
{
  EVAL_TRAVERSE_DATA *ud = (EVAL_TRAVERSE_DATA *)data;
  REAL          det[MAX_N_QUAD_POINTS], norm2, normT;
  REAL_D        Lambda[MAX_N_QUAD_POINTS][N_LAMBDA];
  const REAL_D  *uh_loc;
  const REAL_DD *grduh_vec;
  PARAMETRIC    *parametric = el_info->mesh->parametric;
  int           iq, j, n;
  int           dim = el_info->mesh->dim;

  if(parametric) {
    parametric->init_element(el_info, parametric);
    parametric->grd_lambda(el_info, ud->quad_fast->quad, 0, nil, Lambda, det);

    uh_loc = (*ud->get_real_d_vec)(el_info->el, ud->uh_d, nil);
    grduh_vec = param_grd_uh_d_at_qp(ud->quad_fast, Lambda, uh_loc, nil);

    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = j = 0; j < DIM_OF_WORLD; j++)
	for (n = 0; n < DIM_OF_WORLD; n++)
	  norm2 += SQR(grduh_vec[iq][j][n]);
      
      normT += det[iq] * ud->quad_fast->w[iq] * norm2;
    }
    ud->norm += normT;
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_grd_lambda_1d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_grd_lambda_2d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_grd_lambda_3d(el_info, Lambda[0]);
#endif
#endif
    }

    uh_loc = (*ud->get_real_d_vec)(el_info->el, ud->uh_d, nil);
    grduh_vec = grd_uh_d_at_qp(ud->quad_fast, (const REAL_D *)Lambda[0],
			       uh_loc, nil);

    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = j = 0; j < DIM_OF_WORLD; j++)
	for (n = 0; n < DIM_OF_WORLD; n++)
	  norm2 += SQR(grduh_vec[iq][j][n]);
      
      normT += ud->quad_fast->w[iq]*norm2;
    }
    ud->norm += det[0]*normT;
  }

  return;
}


REAL H1_norm_uh_d(const QUAD *quad, const DOF_REAL_D_VEC *u_h)
{
  FUNCNAME("H1_norm_uh_d");
  EVAL_TRAVERSE_DATA td[1] = {{0}};
  int      deg;

  if (!(td->uh_d = u_h))
  {
    ERROR("no DOF vector u_h; returning 0.0\n");
    return(0.0);
  }
  if (!quad)
  {
    deg = 2*td->uh_d->fe_space->bas_fcts->degree-2;
    quad = get_quadrature(td->uh_d->fe_space->mesh->dim, deg);
  }

  td->quad_fast = get_quad_fast(td->uh_d->fe_space->bas_fcts, 
				quad, INIT_GRD_PHI);
  td->get_real_d_vec = td->uh_d->fe_space->bas_fcts->get_real_d_vec;

  td->norm = 0.0;
  mesh_traverse(td->uh_d->fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS, 
		H1_norm_uh_d_fct, td);

  return(sqrt(td->norm));
}

static void L2_norm_uh_d_fct(const EL_INFO *el_info, void *data)
{
  EVAL_TRAVERSE_DATA *ud = (EVAL_TRAVERSE_DATA *)data;
  REAL         det[MAX_N_QUAD_POINTS], normT, norm2;
  const REAL_D *uh_loc, *uh_vec;
  int          iq, n;
  int          dim = el_info->mesh->dim;
  PARAMETRIC   *parametric = el_info->mesh->parametric;

  if (parametric) {
    parametric->init_element(el_info, parametric);
    parametric->det(el_info, ud->quad_fast->quad, 0, nil, det);

    uh_loc = (*ud->get_real_d_vec)(el_info->el, ud->uh_d, nil);
    uh_vec = uh_d_at_qp(ud->quad_fast, uh_loc, nil);
    
    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = n = 0; n < DIM_OF_WORLD; n++)
	norm2 += SQR(uh_vec[iq][n]);
      normT += det[iq]*ud->quad_fast->w[iq]*norm2;
    }
    ud->norm += normT;
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_det_1d(el_info);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_det_2d(el_info);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_det_3d(el_info);
#endif
#endif
    }

    uh_loc = (*ud->get_real_d_vec)(el_info->el, ud->uh_d, nil);
    uh_vec = uh_d_at_qp(ud->quad_fast, uh_loc, nil);
    
    for (normT = iq = 0; iq < ud->quad_fast->n_points; iq++) {
      for (norm2 = n = 0; n < DIM_OF_WORLD; n++)
	norm2 += SQR(uh_vec[iq][n]);
      normT += ud->quad_fast->w[iq]*norm2;
    }
    ud->norm += det[0]*normT;
  }

  return;
}

REAL L2_norm_uh_d(const QUAD *quad, const DOF_REAL_D_VEC *u_h)
{
  FUNCNAME("L2_norm_uh_d");
  EVAL_TRAVERSE_DATA td[1] = {{0}};
  int      deg;

  if (!(td->uh_d = u_h))
  {
    ERROR("no DOF vector u_h; returning 0.0\n");
    return(0.0);
  }
  if (!quad)
  {
    deg = 2*td->uh_d->fe_space->bas_fcts->degree;
    quad = get_quadrature(td->uh_d->fe_space->mesh->dim, deg);
  }
  td->quad_fast = get_quad_fast(td->uh_d->fe_space->bas_fcts, quad, INIT_PHI);
  td->get_real_d_vec = td->uh_d->fe_space->bas_fcts->get_real_d_vec;

  td->norm = 0.0;
  mesh_traverse(td->uh_d->fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS, 
		L2_norm_uh_d_fct, td);

  return(sqrt(td->norm));
}

static const DOF  *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
static int        n_bas_fcts;
static const DOF_ADMIN  *admin;

/*  scalar problems                                                        */
static REAL       *inter_vec, (*inter_fct)(const REAL_D);
static const REAL *(*interpol_el)(const EL_INFO *, int, const int *, 
				  REAL (*)(const REAL_D),
				  REAL (*f_loc)(const EL_INFO *el_info,
						const REAL lambda[N_LAMBDA]),
				  REAL *);

/*  vector valued problems                                                 */
static REAL_D     *inter_vec_d;
static const REAL *(*inter_fct_d)(const REAL_D, REAL_D);
static const REAL_D *(*interpol_el_d)(const EL_INFO *, int, const int *b_no,
				      const REAL *(*)(const REAL_D, REAL_D), 
			     const REAL *(*f_loc)(const EL_INFO *el_info,
						  const REAL lambda[N_LAMBDA],
						       REAL_D val),
				      REAL_D *);

static void interpol_fct(const EL_INFO *el_info, void *data)
{
  int         i;
  const DOF   *dof = get_dof_indices(el_info->el, admin, nil);
  const REAL  *inter_val = interpol_el(el_info, 0, nil, inter_fct, nil, nil);

  for (i = 0; i < n_bas_fcts; i++)
    inter_vec[dof[i]] = inter_val[i];

  return;
}

void interpol(REAL (*fct)(const REAL_D), DOF_REAL_VEC *vec)
{
  FUNCNAME("interpol");
  const FE_SPACE   *fe_space;

  GET_DOF_VEC(inter_vec, vec);

  if (!(fe_space = vec->fe_space))
  {
    MSG("no dof admin in vec %s, skipping interpolation\n", NAME(vec));
    return;
  }

  if (!(admin = fe_space->admin))
  {
    MSG("no dof admin in fe_space %s, skipping interpolation\n",
	NAME(fe_space));
    return;
  }
  
  if (!fe_space->bas_fcts)
  {
    MSG("no basis functions in admin of vec %s, skipping interpolation\n", 
	NAME(vec));
    return;
  }
  n_bas_fcts = fe_space->bas_fcts->n_bas_fcts;

  if (!(inter_fct = fct))
  {
    MSG("function that should be interpolated only pointer to nil, ");
    print_msg("skipping interpolation\n");
    return;
  }

  if (!(interpol_el = fe_space->bas_fcts->interpol))
  {
    MSG("no function for interpolation on an element available\n");
    MSG("in basis functions of vec %s, skipping interpolation\n", NAME(vec));
    return;
  }
  if (!(get_dof_indices = fe_space->bas_fcts->get_dof_indices))
  {
    MSG("no function for getting dof's on an element available\n");
    MSG("in basis functions of vec %s, skipping interpolation\n", NAME(vec));
    return;
  }

  mesh_traverse(fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS, interpol_fct, nil);

  return;
}

static void interpol_fct_d(const EL_INFO *el_info, void *data)
{
  int          i, k;
  const DOF    *dof = get_dof_indices(el_info->el, admin, nil);
  const REAL_D *inter_val = interpol_el_d(el_info, 0, nil,
					  inter_fct_d, nil,nil);

  for (i = 0; i < n_bas_fcts; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      inter_vec_d[dof[i]][k] = inter_val[i][k];

  return;
}

void interpol_d(const REAL *(*fct)(const REAL_D, REAL_D), DOF_REAL_D_VEC *vec)
{
  FUNCNAME("interpol_d");
  const FE_SPACE   *fe_space;

  GET_DOF_VEC(inter_vec_d, vec);

  if (!(fe_space = vec->fe_space))
  {
    MSG("no dof admin in vec %s, skipping interpolation\n", NAME(vec));
    return;
  }

  if (!(admin = fe_space->admin))
  {
    MSG("no dof admin in fe_space %s, skipping interpolation\n",
	NAME(fe_space));
    return;
  }
  
  if (!fe_space->bas_fcts)
  {
    MSG("no basis functions in admin of vec %s, skipping interpolation\n", 
	NAME(vec));
    return;
  }
  n_bas_fcts = fe_space->bas_fcts->n_bas_fcts;

  if (!(inter_fct_d = fct))
  {
    MSG("function that should be interpolated only pointer to nil, ");
    print_msg("skipping interpolation\n");
    return;
  }

  if (!(interpol_el_d = fe_space->bas_fcts->interpol_d))
  {
    MSG("no function for interpolation on an element available\n");
    MSG("in basis functions of vec %s, skipping interpolation\n", NAME(vec));
    return;
  }
  if (!(get_dof_indices = fe_space->bas_fcts->get_dof_indices))
  {
    MSG("no function for getting dof's on an element available\n");
    MSG("in basis functions of vec %s, skipping interpolation\n", NAME(vec));
    return;
  }

  mesh_traverse(fe_space->mesh, -1, CALL_LEAF_EL|FILL_COORDS, 
		interpol_fct_d, nil);

  return;
}
