#  Common external function handlers

## jerryx_handler_assert

**Summary**

Assert for scripts. The routine calls `jerry_port_fatal` on assertion failure.

**Prototype**

```c
jerry_value_t
jerryx_handler_assert (const jerry_value_t func_obj_val, const jerry_value_t this_p,
                       const jerry_value_t args_p[], const jerry_length_t args_cnt);
```

- `func_obj_val` - the function object that was called (unused).
- `this_p` - the `this` value of the call (unused).
- `args_p` - the array of function arguments.
- `args_cnt` - the number of function arguments.
- return value - `jerry_value_t` representing boolean true, if only one argument
  was passed and that argument was a boolean true. Note that the function does
  not return otherwise.

**See also**

- [jerryx_handler_register_global](#jerryx_handler_register_global)


## jerryx_handler_gc

**Summary**

Expose garbage collector to scripts.

**Prototype**

```c
jerry_value_t
jerryx_handler_gc (const jerry_value_t func_obj_val, const jerry_value_t this_p,
                   const jerry_value_t args_p[], const jerry_length_t args_cnt);
```

- `func_obj_val` - the function object that was called (unused).
- `this_p` - the `this` value of the call (unused).
- `args_p` - the array of function arguments (unused).
- `args_cnt` - the number of function arguments (unused).
- return value - `jerry_value_t` representing `undefined`.

**See also**

- [jerryx_handler_register_global](#jerryx_handler_register_global)


## jerryx_handler_print

**Summary**

Provide a `print` implementation for scripts. The routine converts all of its
arguments to strings and outputs them char-by-char using
`jerryx_port_handler_print_char`. The NUL character is output as "\u0000",
other characters are output bytewise.

*Note*: This implementation does not use standard C `printf` to print its
output. This allows more flexibility but also extends the core JerryScript
engine port API. Applications that want to use `jerryx_handler_print` must
ensure that their port implementation also provides
`jerryx_port_handler_print_char`.

**Prototype**

```c
jerry_value_t
jerryx_handler_print (const jerry_value_t func_obj_val, const jerry_value_t this_p,
                      const jerry_value_t args_p[], const jerry_length_t args_cnt);
```

- `func_obj_val` - the function object that was called (unused).
- `this_p` - the `this` value of the call (unused).
- `args_p` - the array of function arguments.
- `args_cnt` - the number of function arguments.
- return value - `jerry_value_t` representing `undefined` if all arguments could
  be converted to strings, an `Error` otherwise.

**See also**

- [jerryx_handler_register_global](#jerryx_handler_register_global)
- [jerryx_port_handler_print_char](#jerryx_port_handler_print_char)


# Handler registration helper

## jerryx_handler_register_global

**Summary**

Register a JavaScript function in the global object.

*Note*: Returned value must be freed with `jerry_release_value`, when it is no
longer needed.

**Prototype**

```c
jerry_value_t
jerryx_handler_register_global (const jerry_char_t *name_p,
                                jerry_external_handler_t handler_p);
```

- `name_p` - the name of the function to be registered.
- `handler_p` - the address of the external function handler.
- return value - `jerry_value_t` representing boolean true, if the operation was
  successful, an `Error` otherwise.

**Example**

```c
#include "jerryscript.h"
#include "jerryscript-ext/handler.h"

static const struct {
  const char *name_p;
  jerry_external_handler_t handler_p;
} common_functions[] =
{
  { "assert", jerryx_handler_assert },
  { "gc", jerryx_handler_gc },
  { "print", jerryx_handler_print },
  { NULL, NULL }
};

static void register_common_functions ()
{
  jerry_value_t ret = jerry_create_undefined ();

  for (int i = 0; common_functions[i].name_p != NULL && !jerry_value_has_error_flag (ret); i++)
  {
    ret = jerryx_handler_register_global ((const jerry_char_t *) common_functions[i].name_p,
                                          common_functions[i].handler_p);
  }

  return ret;
}
```


# Port API extension

## jerryx_port_handler_print_char

**Summary**

Print a single character.

**Prototype**

```c
void
jerryx_port_handler_print_char (char c);
```

- `c` - the character to print.

**Example**

```c
/**
 * Print a character to stdout with printf.
 */
void
jerryx_port_handler_print_char (char c)
{
  printf ("%c", c);
} /* jerryx_port_handler_print_char */
```

**See also**

- [jerryx_handler_print](#jerryx_handler_print)
