/*
 * cpu8051.c
 *
 * Copyright (C) 1999 Jonathan St-André
 * Copyright (C) 1999 Hugo Villeneuve <hugo@hugovil.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* Define only here, for not having extern scope on local variables. */
#define CPU8051_M 1

#include <stdio.h>
#include <stdint.h>

#include "reg8051.h"
#include "cpu8051.h"
#include "memory.h"
#include "disasm.h"
#include "instructions_8051.h"

/* Check if the address is a breakpoint */
int
IsBreakpoint(unsigned int address)
{
	int k;

	for (k = 0; k < cpu8051.bp_count; k++) {
		if (cpu8051.bp[k] == address)
			return 1;
	}

	/* The address was not found in the list of breakpoints */
	return 0;
}

/* Show Breakpoints list */
void
ShowBreakpoints(void)
{
	int k;

	for (k = 0; k < cpu8051.bp_count; k++)
		printf("Breakpoint at address = %.4X\n", cpu8051.bp[k]);
}

/* Set Breakpoint at address at the end of the breakpoint list */
void
SetBreakpoint(unsigned int address)
{
	if (IsBreakpoint(address))
		return; /* Already a breakpoint */

	if (cpu8051.bp_count < MAXBP)
		cpu8051.bp[cpu8051.bp_count++] = address;
}

/* Clear Breakpoint at Address from list */
void
ClearBreakpoint(unsigned int address)
{
	int k;

	for (k = 0; k < cpu8051.bp_count; k++) {
		if (cpu8051.bp[k] == address) {
			/* Fill removed breakpoint slot with last entry */
			cpu8051.bp[k] = cpu8051.bp[cpu8051.bp_count - 1];
			cpu8051.bp_count--;
		}
	}
}

/* Toggle the breakpoint at Address. */
void
ToggleBreakpoint(unsigned int address)
{
	if (IsBreakpoint(address))
		ClearBreakpoint(address);
	else
		SetBreakpoint(address);
}

void
cpu8051_init(void)
{
	cpu8051.pc = 0;
	cpu8051.clock = 0;
	cpu8051.active_priority = -1;
	cpu8051.bp_count = 0;
}

/* Reset the registers and CPU state */
void
cpu8051_Reset(void)
{
	int i;

	cpu8051.pc = 0;
	cpu8051.clock = 0;
	cpu8051.active_priority = -1;

	/* Reset registers */

	for (i = 0; i < 256; i++) {
		/* Clear  IRAM nad SFR */
		memory_write8(INT_MEM_ID, i, 0);
	}

	memory_write8(INT_MEM_ID, _P0_, 0xFF);
	memory_write8(INT_MEM_ID, _P1_, 0xFF);
	memory_write8(INT_MEM_ID, _P2_, 0xFF);
	memory_write8(INT_MEM_ID, _P3_, 0xFF);
	memory_write8(INT_MEM_ID, _SP_, 0x07);
}

static void
cpu8051_convert_bit_address(uint8_t bit_address, uint8_t *byte_address,
			    uint8_t *bit_number)
{
	if (bit_address > 0x7F) {
		/* SFR 80-FF */
		*byte_address = bit_address & 0xF8;
		*bit_number = bit_address & 0x07;
	} else {
		/* 20-2F */
		*byte_address = (bit_address >> 3) + 0x20;
		*bit_number = bit_address & 0x07;
	}
}

/* Write with a direct addressing mode at Address the new Value */
void
cpu8051_WriteD(unsigned int Address, unsigned char Value)
{
	memory_write8(INT_MEM_ID, Address, Value);
}

/* Write with an indirect addressing mode at Address the new Value */
void
cpu8051_WriteI(unsigned int Address, unsigned char Value)
{
	if (Address > 0x7F) {
		memory_write8(EXT_MEM_ID, Address, Value);
		return;
	}

	memory_write8(INT_MEM_ID, Address, Value);
}

/* Write with a bit addressing mode at BitAddress the new Value */
void
cpu8051_WriteB(uint8_t bit_address, uint8_t value)
{
	uint8_t byte_address;
	uint8_t bit_number;
	unsigned char ByteValue, ByteMask;

	cpu8051_convert_bit_address(bit_address, &byte_address, &bit_number);

	ByteMask = ((1 << bit_number) ^ 0xFF);
	ByteValue = cpu8051_ReadD(byte_address) & ByteMask;
	ByteValue += value << bit_number;
	cpu8051_WriteD(byte_address, ByteValue);
}

/* Read with a direct addressing mode at Address */
unsigned char
cpu8051_ReadD(unsigned int Address)
{
	if (Address > 0xFF)
		return memory_read8(EXT_MEM_ID, Address);
	else
		return memory_read8(INT_MEM_ID, Address);
}

/* Read with a indirect addressing mode at Address */
unsigned char
cpu8051_ReadI(unsigned int Address)
{
	if (Address > 0x7F)
		return memory_read8(EXT_MEM_ID, Address);
	else
		return memory_read8(INT_MEM_ID, Address);
}

/* Read with a bit addressing mode at BitAddress */
unsigned char
cpu8051_ReadB(uint8_t bit_address)
{
	uint8_t byte_address;
	uint8_t bit_number;
	unsigned char BitValue;

	cpu8051_convert_bit_address(bit_address, &byte_address, &bit_number);

	BitValue = (cpu8051_ReadD(byte_address) >> bit_number);
	BitValue &= 1;
	return BitValue;
}

static void
cpu8051_process_interrupt(int pc, int pri)
{
	unsigned char SP;

	SP = cpu8051_ReadD(_SP_);
	cpu8051_WriteI(++SP, (cpu8051.pc & 0xFF));
	cpu8051_WriteI(++SP, (cpu8051.pc >> 8));
	cpu8051_WriteD(_SP_, SP);
	cpu8051.pc = 0x0B;
	cpu8051.active_priority = pri;
}


/* Check interrupts state and process them as needed */
static void
cpu8051_CheckInterrupts(void)
{
	int i;

	if ((cpu8051_ReadD(_IE_) & 0x80) == 0)
		return;

	for (i = 1; i >= 0; i--) {
		if (cpu8051.active_priority < i) {
			/* Interrupt timer 0 */
			if ((cpu8051_ReadD(_IE_) & 0x02) &&
			    ((cpu8051_ReadD(_IP_ & 0x02) ? i : !i) &&
			     (cpu8051_ReadD(_TCON_) & 0x20))) {
				cpu8051_WriteD(_TCON_,
					       cpu8051_ReadD(_TCON_) & 0xDF);
				cpu8051_process_interrupt(0x0B, i);
				return;
			}
			/* Interrupt timer 1 */
			if ((cpu8051_ReadD(_IE_) & 0x08) &&
			    ((cpu8051_ReadD(_IP_) & 0x08) ? i : !i) &&
			    (cpu8051_ReadD(_TCON_) & 0x80)) {
				cpu8051_WriteD(_TCON_,
					       cpu8051_ReadD(_TCON_) & 0x7F);
				cpu8051_process_interrupt(0x1B, i);
				return;
			}
			/* Serial Interrupts */
			if ((cpu8051_ReadD(_IE_) & 0x10) &&
			    ((cpu8051_ReadD(_IP_) & 0x10) ? i : !i) &&
			    (cpu8051_ReadD(_SCON_) & 0x03)) {
				cpu8051_process_interrupt(0x23, i);
				return;
			}
			/* Interrupt timer 2 */
			if ((cpu8051_ReadD(_IE_) & 0x20) &&
			    ((cpu8051_ReadD(_IP_) & 0x20) ? i : !i) &&
			    (cpu8051_ReadD(_T2CON_) & 0x80)) {
				cpu8051_process_interrupt(0x2B, i);
				return;
			}
		}
	}
}

static void
process_timer(uint8_t tl, uint8_t th, uint8_t tf_mask, uint8_t TR, uint8_t mode,
	      uint8_t GATE, uint32_t TimerCounter)
{
	unsigned int tmp;

	switch (mode) {
	case 0:
		/* Mode 0, 13-bits counter. */
		tmp = cpu8051_ReadD(th) * 0x100 + cpu8051_ReadD(tl);
		tmp++;
		tmp &= 0x1FFF; /* We keep only 13 bits */

		if (tmp == 0)  /* If overflow set TF0 */
			cpu8051_WriteD(_TCON_, cpu8051_ReadD(_TCON_) | tf_mask);
		cpu8051_WriteD(_TH0_, tmp / 0x100);
		cpu8051_WriteD(_TL0_, tmp & 0xFF);
		break;
	case 1:
		/* Mode 1, 16-bits counter */
		tmp = cpu8051_ReadD(th) * 0x100 + cpu8051_ReadD(tl);
		tmp++;
		tmp &= 0xFFFF; /* We keep only 16 bits */
		if (tmp == 0) /* If overflow set TF0 */
			cpu8051_WriteD(_TCON_, cpu8051_ReadD(_TCON_) | tf_mask);
		cpu8051_WriteD(_TH0_, (tmp / 0x100));
		cpu8051_WriteD(_TL0_, (tmp & 0xFF));
		break;
	case 2:
		/* Mode 2, 8-bits counter with Auto-Reload */
		tmp = cpu8051_ReadD(tl);
		tmp++;
		tmp &= 0xFF;
		if (tmp == 0) {
			/* If overflow -> reload and set TF0 */
			cpu8051_WriteD(_TCON_, cpu8051_ReadD(_TCON_) | tf_mask);
			cpu8051_WriteD(tl, cpu8051_ReadD(th));
		} else
			cpu8051_WriteD(tl, tmp);
		break;
	case 3:
		/* Mode 3 : inactive mode for timer 1 */
		if (tl == _TL0_) {
			/* TL0 and TH0 are 2 independents 8-bits timers. */
			if (TR && !GATE && !TimerCounter) {
				tmp = cpu8051_ReadD(tl);
				tmp++;
				tmp &= 0xFF;
				if (tmp == 0) /* If TL0 overflow set TF0 */
					cpu8051_WriteD(_TCON_,
						       cpu8051_ReadD(_TCON_) |
						       tf_mask);
				cpu8051_WriteD(tl, tmp);
			} /* TH0 utilise TR1 et TF1. */
			TR = cpu8051_ReadD(_TCON_) & 0x40;
			if (TR) {
				tmp = cpu8051_ReadD(th);
				tmp++;
				tmp &= 0xFF;
				if (tmp == 0) /* If TH0 overflow set TF1 */
					cpu8051_WriteD(_TCON_,
						       cpu8051_ReadD(_TCON_) |
						       0x80);
				cpu8051_WriteD(_TH0_, tmp);
			}
		}
		break;
	}
}

/* Run timers */
static void
cpu8051_DoTimers(void)
{
	unsigned int TR;
	unsigned int MODE;
	unsigned int GATE;
	unsigned int TimerCounter;

	/* Timer 0 */
	TR = cpu8051_ReadD(_TCON_) & 0x10;
	MODE = cpu8051_ReadD(_TMOD_) & 0x03;
	GATE = cpu8051_ReadD(_TMOD_) & 0x08;
	TimerCounter = cpu8051_ReadD(_TMOD_) & 0x04;

	if ((TR && !GATE && !TimerCounter) || (MODE == 3))
		process_timer(_TL0_, _TH0_, 0x20, TR, MODE, GATE, TimerCounter);

	/* Timer 1 */
	TR = cpu8051_ReadD(_TCON_) & 0x40;
	MODE = (cpu8051_ReadD(_TMOD_) & 0x30) >> 4 ;
	GATE = cpu8051_ReadD(_TMOD_) & 0x80;
	TimerCounter = cpu8051_ReadD(_TMOD_) & 0x40;

	if (TR && !GATE && !TimerCounter)
		process_timer(_TL1_, _TH1_, 0x80, TR, MODE, GATE, TimerCounter);
}

/* Execute at address cpu8051.pc from PGMMem */
void
cpu8051_Exec(void)
{
	int i;
	unsigned char opcode;
	int insttiming;

	opcode = memory_read8(PGM_MEM_ID, cpu8051.pc);
	cpu8051.pc++;
	insttiming = (*opcode_table[opcode])(); /* Function callback. */

	for (i = 0; i < insttiming; i++) {
		cpu8051_CheckInterrupts();
		cpu8051_DoTimers();
		cpu8051.clock++;
	}
}

/*
 * Addressing modes defined in the order as they appear in disasm.h
 * from table argstext[]
 */
#define ADDR11 0
#define ADDR16 1
#define DIRECT 3
#define BITADDR 14
#define RELADDR 15
#define DATAIMM 16
#define DATA16 22
#define CBITADDR 23

/*
 * SFR Memory map [80h - FFh]
 * ---------------------------------------------------------------
 * F8 |      |      |      |      |      |      |      |      | FF
 * F0 |   B  |      |      |      |      |      |      |      | F7
 * E8 |      |      |      |      |      |      |      |      | EF
 * E0 |  ACC |      |      |      |      |      |      |      | E7
 * D8 |      |      |      |      |      |      |      |      | DF
 * D0 |  PSW |      |      |      |      |      |      |      | D7
 * C8 | T2CON|      |RCAP2L|RCAP2H|  TL2 |  TH2 |      |      | CF
 * C0 |      |      |      |      |      |      |      |      | C7
 * B8 |  IP  |      |      |      |      |      |      |      | BF
 * B0 |  P3  |      |      |      |      |      |      |      | B7
 * A8 |  IE  |      |      |      |      |      |      |      | AF
 * A0 |  P2  |      |      |      |      |      |      |      | A7
 * 98 | SCON | SBUF |      |      |      |      |      |      | 9F
 * 90 |  P1  |      |      |      |      |      |      |      | 97
 * 88 | TCON | TMOD |  TL0 |  TL1 |  TH0 |  TH1 |      |      | 8F
 * 80 |  P0  |  SP  |  DPL |  DPH |      |      |      | PCON | 87
 * ---------------------------------------------------------------
 */

/* Return as Text the name of the SFR register at Address if any */
static int
cpu8051_SFRMemInfo(unsigned int Address, char *Text)
{
	switch (Address) {
	case 0x80: return sprintf(Text, "P0");
	case 0x81: return sprintf(Text, "SP");
	case 0x82: return sprintf(Text, "DPL");
	case 0x83: return sprintf(Text, "DPH");
	case 0x87: return sprintf(Text, "PCON");
	case 0x88: return sprintf(Text, "TCON");
	case 0x89: return sprintf(Text, "TMOD");
	case 0x8A: return sprintf(Text, "TL0");
	case 0x8B: return sprintf(Text, "TL1");
	case 0x8C: return sprintf(Text, "TH0");
	case 0x8D: return sprintf(Text, "TH1");
	case 0x90: return sprintf(Text, "P1");
	case 0x98: return sprintf(Text, "SCON");
	case 0x99: return sprintf(Text, "SBUF");
	case 0xA0: return sprintf(Text, "P2");
	case 0xA8: return sprintf(Text, "IE");
	case 0xB0: return sprintf(Text, "P3");
	case 0xB8: return sprintf(Text, "IP");
	case 0xC8: return sprintf(Text, "T2CON");
	case 0xCA: return sprintf(Text, "RCAP2L");
	case 0xCB: return sprintf(Text, "RCAP2H");
	case 0xCC: return sprintf(Text, "TL2");
	case 0xCD: return sprintf(Text, "TH2");
	case 0xD0: return sprintf(Text, "PSW");
	case 0xE0: return sprintf(Text, "ACC");
	case 0xF0: return sprintf(Text, "B");
	default: return sprintf(Text, "%.2XH", Address);
	}
}

/* Return as Text the decoded BitAddress */
static void
cpu8051_IntMemBitInfo(uint8_t bit_address, char *text)
{
	uint8_t byte_address;
	uint8_t bit_number;
	int len;

	cpu8051_convert_bit_address(bit_address, &byte_address, &bit_number);

	len = cpu8051_SFRMemInfo(byte_address, text);
	sprintf(&text[len], ".%X", bit_address);
}

/* Disasm one instruction at Address into a Text string */
int
cpu8051_Disasm(unsigned int Address, char *Text)
{
	int len = 0;
	char TextTmp[20];
	unsigned char OpCode;
	int ArgTblOfs;
	int InstSize;
	int i;

	OpCode = memory_read8(PGM_MEM_ID, Address);
	InstSize = InstSizesTbl[OpCode];

	len += sprintf(Text, " %.4X ", Address);

	for (i = 0; i < InstSize; i++)
		len += sprintf(&Text[len], " %.2X",
				      memory_read8(PGM_MEM_ID, Address + i));

	Address++;

	for (; len < 17;)
		len += sprintf(&Text[len], " ");

	len += sprintf(&Text[len], "%s ",
			      InstTextTbl[InstTypesTbl[OpCode]]);
	ArgTblOfs = OpCode << 2;

	for (; len < 25;)
		len += sprintf(&Text[len], " ");

	/*
	 * MOV direct, direct (OpCode 85h) is peculiar, the operands
	 * are inverted
	 */
	if (OpCode == 0x85) {
		cpu8051_SFRMemInfo(memory_read8(PGM_MEM_ID, Address + 1),
				   TextTmp);
		len += sprintf(&Text[len], "%s,", TextTmp);
		cpu8051_SFRMemInfo(memory_read8(PGM_MEM_ID, Address),
				   TextTmp);
		len += sprintf(&Text[len], "%s", TextTmp);
		Address += 2;
		return InstSize;
	}

	for (i = 1; i <= InstArgTbl[ArgTblOfs]; i++) {
		switch (InstArgTbl[ArgTblOfs + i]) {
		case ADDR11: {
			len += sprintf(&Text[len],
				       "%.4XH", ((OpCode << 3) & 0xF00) +
				       (memory_read8(PGM_MEM_ID, Address)));
			Address++;
			break;
		}
		case ADDR16: {
			len += sprintf(
				&Text[len], "%.4XH",
				((memory_read8(PGM_MEM_ID, Address) << 8) +
				 memory_read8(PGM_MEM_ID, Address + 1)));
			Address += 2;
			break;
		}
		case DIRECT: {
			cpu8051_SFRMemInfo(memory_read8(PGM_MEM_ID, Address),
					   TextTmp);
			len += sprintf(&Text[len], "%s", TextTmp);
			Address++;
			break;
		}
		case BITADDR: {
			cpu8051_IntMemBitInfo(
				(memory_read8(PGM_MEM_ID, Address) & 0xF8),
				TextTmp);
			len += sprintf(&Text[len], "%s.%X" , TextTmp,
				       (memory_read8(PGM_MEM_ID, Address) & 7));
			Address++;
			break;
		}
		case RELADDR: {
			Address++;
			len += sprintf(&Text[len], "%.4XH", (Address & 0xFF00) +
				       (((Address & 0xFF) +
					 memory_read8(PGM_MEM_ID,
						      Address - 1)) & 0xFF));
			break;
		}
		case DATAIMM: {
			len += sprintf(&Text[len], "#%.2XH",
				       memory_read8(PGM_MEM_ID, Address));
			Address++;
			break;
		}
		case DATA16: {
			len += sprintf(&Text[len], "#%.4XH",
				       ((memory_read8(PGM_MEM_ID,
						      Address) << 8) +
					memory_read8(PGM_MEM_ID, Address+1)));
			Address += 2;
			break;
		}
		case CBITADDR: {
			cpu8051_IntMemBitInfo((memory_read8(PGM_MEM_ID,
							    Address) & 0xF8),
					      TextTmp);
			len += sprintf(&Text[len], "/%s.%X", TextTmp,
				       (memory_read8(PGM_MEM_ID, Address) & 7));
			Address++;
			break;
		}
		default: {
			len += sprintf(&Text[len], "%s",
				       ArgsTextTbl[InstArgTbl[ArgTblOfs + i]]);
		}
		}
		if (i < InstArgTbl[ArgTblOfs])
			len += sprintf(&Text[len], ",");
	}

	return InstSize;
}
