#!/bin/bash

# Common helper functions for downloading archives based on versions in XML files

# Get the repository root directory
REPO_ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"

# Define the path to Versions.props once
PACKAGE_VERSIONS_PATH="$REPO_ROOT/eng/Versions.props"

# Helper to extract a property value from an XML file
function GetXmlPropertyValue {
  local propName="$1"
  local filePath="$2"
  local value=""
  local line pattern
  line=$(grep -m 1 "<$propName>" "$filePath" || :)
  pattern="<$propName>(.*)</$propName>"
  if [[ $line =~ $pattern ]]; then
    value="${BASH_REMATCH[1]}"
  fi
  echo "$value"
}

# Helper to download a file with retries
function DownloadWithRetries {
  local url="$1"
  local targetDir="$2"
  (
    cd "$targetDir" &&
    for i in {1..5}; do
      if curl -fL --retry 5 -O "$url"; then
        return 0
      else
        case $? in
          18)
            sleep 3
            ;;
          *)
            return 1
            ;;
        esac
      fi
    done
    return 1
  )
}

# Generic function to download an archive based on property name
function DownloadArchive {
  local label="$1"
  local propertyName="$2"
  local isRequired="$3"
  local artifactsRid="$4"
  local outputDir="$5"
  local destinationFilenamePrefix="${6:-}"

  local notFoundMessage="No $label found to download..."

  local archiveVersion
  archiveVersion=$(GetXmlPropertyValue "$propertyName" "$PACKAGE_VERSIONS_PATH")
  if [[ -z "$archiveVersion" ]]; then
    if [ "$isRequired" == true ]; then
      echo "  ERROR: $notFoundMessage"
      return 1
    else
      echo "  $notFoundMessage"
      return 0
    fi
  fi

  local archiveUrl
  local artifactsBaseFileName="Private.SourceBuilt.Artifacts"
  local prebuiltsBaseFileName="Private.SourceBuilt.Prebuilts"
  local defaultArtifactsRid='centos.10-x64'

  if [[ "$propertyName" == "MicrosoftNETSdkVersion" ]]; then
    archiveUrl="https://ci.dot.net/public/source-build/$artifactsBaseFileName.$archiveVersion.$artifactsRid.tar.gz"
  elif [[ "$propertyName" == *Prebuilts* ]]; then
    archiveUrl="https://builds.dotnet.microsoft.com/source-built-artifacts/assets/$prebuiltsBaseFileName.$archiveVersion.$defaultArtifactsRid.tar.gz"
  elif [[ "$propertyName" == *Artifacts* ]]; then
    archiveUrl="https://builds.dotnet.microsoft.com/source-built-artifacts/assets/$artifactsBaseFileName.$archiveVersion.$artifactsRid.tar.gz"
  elif [[ "$propertyName" == *Sdk* ]]; then
    archiveUrl="https://builds.dotnet.microsoft.com/source-built-artifacts/sdks/dotnet-sdk-$archiveVersion-$artifactsRid.tar.gz"
  else
    echo "  ERROR: Unknown archive property name: $propertyName"
    return 1
  fi

  echo "  Downloading $label from $archiveUrl..."
  if ! DownloadWithRetries "$archiveUrl" "$outputDir"; then
    echo "  ERROR: Failed to download $archiveUrl"
    return 1
  fi

  # Rename the file if a destination filename prefix is provided
  if [[ -n "$destinationFilenamePrefix" ]]; then
    local downloadedFilename
    downloadedFilename=$(basename "$archiveUrl")
    # Extract the suffix from the downloaded filename
    local suffix="${downloadedFilename#$artifactsBaseFileName}"
    local newFilename="$destinationFilenamePrefix$suffix"
    mv "$outputDir/$downloadedFilename" "$outputDir/$newFilename"
    echo "  Renamed $downloadedFilename to $newFilename"
  fi

  return 0
}
