/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#define _GNU_SOURCE /* getopt_long() */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <locale.h> /* setlocale() */
#include <sys/ioctl.h>
#include <getopt.h> /* getopt_long() */

/* open */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cdw_logging.h"
#include "gettext.h"
#include "cdw_drive.h"
#include "cdw_utils.h"
#include "cdw_debug.h"

#include "config_cdw.h"


static void cdw_utils_print_version(void);
static void cdw_utils_print_help(const char *program_name);




void after_event(const char *title, __attribute__((unused)) int adddic)
{
	cdw_logging_display_log_conditional(title);
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
#ifdef HAVE_LIBSQLITE
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
	/* disabled in version 0.4.0, ejecting after write operation is
	handled by conditional_eject_after_write(); even if this eject is
	required by other operation, they are unsupported in 0.4.0 */
	/*
	if (strcmp(config.eject, "1") == 0) {
		eject_tray(config.cdrw_device);
	}
	*/

	return;
}





cdw_arguments_t cdw_commandline_arguments;

static struct option cdw_getopt_long_options[] = {
	{ "version",          no_argument,       0, 0},
	{ "enable-dvd-rp-dl", no_argument,       0, 0},
	{ "escdelay",         required_argument, 0, 0},
	{ "help",             no_argument,       0, 0},
	{ 0,                  0,                 0, 0} };



/**
   \brief Process main(argc, argv[]) arguments

   \return -1 on errors - cdw has to close returning -1
   \return 0 on no errors - cdw has to close returning 0
   \return 1 on no errors - cdw should continue running
*/
int cdw_utils_process_commandline_arguments(int argc, char *argv[])
{
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp((char *) argv[1], "--catalog") == 0) || (strcmp((char *) argv[1], "-c") == 0)) {
		initscr();
		start_color();
		cbreak();
		noecho();
		keypad(stdscr, TRUE);
		/* colors */
		init_curses_colors();
		lines = LINES - 2;
		if ((LINES < 24) || (COLS < 79)) {
			/* 2TRANS: error message displayed when terminal requirements are not met */
			fprintf(stderr, _("Needed min 80x25 terminal!"));
			exit(EXIT_FAILURE);
		}
		cddb_window();
		exit(EXIT_SUCCESS);
	}
#endif

	cdw_commandline_arguments.support_dvd_rp_dl = false;
	cdw_commandline_arguments.escdelay_ms = -1;

	while (1) {
		int option_index = 0;
		int c = getopt_long(argc, argv, "hv", cdw_getopt_long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
		case 0:
			cdw_sdm ("INFO: option %s\n", cdw_getopt_long_options[option_index].name);
			if (optarg) {
			        cdw_sdm ("INFO:    with arg %s\n", optarg);
			}
			if (option_index == 0) {
				cdw_utils_print_version();
				return 0;  /* close cdw without signalling errors */
			} else if (option_index == 1) {
				cdw_commandline_arguments.support_dvd_rp_dl = true;
			} else if (option_index == 2) {
				char *rest = (char *) NULL;
				cdw_commandline_arguments.escdelay_ms = (int) strtol(optarg, &rest, 10);
				if (rest != (char *) NULL && strlen(rest)) {
					/* 2TRANS: this is error message printed to terminal,
					   first '%s' is name of option expecting a valid value,
					   second '%s' is invalid value of the option */
					fprintf(stderr, _("ERROR: incorrect value of option %s: \"%s\"\n"),
						cdw_getopt_long_options[option_index].name, optarg);
					return -1;
				}
				if (cdw_commandline_arguments.escdelay_ms < 0) {
					/* 2TRANS: this is error message printed to terminal,
					   "ESC delay" is number describing time between
					    pressing ESC key, and reaction to the key */
					fprintf(stderr, _("ERROR: ESC delay can't be negative\n"));
					return -1;
				}
			} else if (option_index == 3) {
				cdw_utils_print_help(argv[0]);
				return 0; /* close cdw without signalling errors */
			}
			break;
		case 'h':
			cdw_utils_print_help(argv[0]);
			return 0; /* close cdw without signalling errors */
		case 'v':
			cdw_utils_print_version();
			return 0; /* close cdw without signalling errors */
		case '?':
			/* 2TRANS: this is error message printed to terminal */
			fprintf(stderr, _("ERROR: unknown option in command line\n"));
			/* 2TRANS: this is error message printed to terminal */
			fprintf(stderr, _("use '-h' option for help message\n"));
			return -1;
		default:
			break;
		}
	}
	return 1; /* 1 = cdw can continue running */
}





void cdw_utils_print_version(void)
{
	/* 2TRANS: this is message printed in terminal,
	   first %s is program name, second %s is
	   program version */
	printf(_("%s %s\n"), PACKAGE, VERSION);

	return;
}





void cdw_utils_print_help(const char *program_name)
{
	/* 2TRANS: this is message printed in terminal,
	   first %s is program name, second %s is
	   program version */
	printf(_("%s %s\n"), PACKAGE, VERSION);
	/* 2TRANS: this is message printed in terminal,
	   first %s is program name, second %s is
	   program version */
	printf(_("Copyright (C) 2002 - 2003 Balazs Varkonyi\n"));
	/* 2TRANS: this is message printed in terminal,
	   first %s is program name, second %s is
	   program version */
	printf(_("Copyright (C) 2007 - 2014 Kamil Ignacak\n\n"));
	/* 2TRANS: this is message printed in terminal,
	   '2+' means "version 2, or later";
	   keep double newline character */
	printf(_("License: GNU General Public License, version 2+\n\n"));

	/* 2TRANS: this is help text displayed in console;
	   %s will be replaced by program name, don't change its
	   position */
	printf(_("Usage: %s [options]\n\n"), program_name);
	/* 2TRANS: this is help text displayed in console */
	printf(_("Options:\n\n"));
#ifdef HAVE_LIBMYSQLCLIENT
	/* 2TRANS: this is help text displayed in console;
	   don't localize anything before colon */
	printf(_("  -c | --catalog            : start in Disk Catalog mode\n"));
#endif
	/* 2TRANS: this is help text displayed in console;
	   don't localize anything before colon */
	printf(_("  -h | --help               : show this message\n"));
	/* 2TRANS: this is help text displayed in console;
	   don't localize anything before colon */
	printf(_("  -v | --version            : show version of this software\n"));
	/* 2TRANS: this is help text displayed in console;
	   don't localize anything before colon */
	printf(_("  --escdelay=X              : modify ESC key delay period;\n"));
	/* 2TRANS: this is help text displayed in console */
	printf(_("                              'X' is non-negative time in milliseconds\n"));
	/* 2TRANS: this is help text displayed in console;
	   don't localize anything before colon */
	printf(_("  --enable-dvd-rp-dl        : enable support for DVD+R DL;\n"));
	/* 2TRANS: this is help text displayed in console;
	   "buggy" refers to support of DVD+R DL discs */
	printf(_("                              (buggy, with dvd+rw-tools only)\n\n"));

	return;
}





/*
 * Simple wrapper for locale setting functions
 */
int cdw_locale_init(void)
{
	setlocale(LC_ALL, "");
	textdomain(PACKAGE);
	bindtextdomain(PACKAGE, LOCALEDIR);

	return 0;
}





#if 0
/* currently unused, disabling to suppress compiler warnings */
/**
 * Read raw CD sector using Linux ioctl(2) - just a non-portable tool
 *
 * \param sector - valid sector number of sector to be read
 * \param buffer - char table of size 2352
 *
 * \return 0
 */
int raw_read_sector(lsn_t sector, char *_buffer)
{
	int fd = open(config.cdrw_device, O_RDONLY | O_NONBLOCK);
	if (fd == -1) {
		/* debug code */
		/* fprintf(stderr, "\n\n raw_read failed\n\n\n"); */
		return -1;
	}


	union {
		struct cdrom_msf msf;   /* input WARNING: make sure which data structure is used as input, read comments for CDROMREAD* in ioctl.txt */
		char buffer[2352];      /* return */
	} arg;

	msf_t p_msf;
	cdio_lsn_to_msf(3000, &p_msf);
	arg.msf.cdmsf_min0 = p_msf.m;     /* start minute */
	arg.msf.cdmsf_sec0 = p_msf.s;     /* start second */
	arg.msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	int rv0 = ioctl(fd, CDROMREADRAW, &arg);

	/* debug code */
	/* perror("1 -- ");
	fprintf(stderr, "rv0 = %d\n", rv0);

	fprintf(stderr, ">>RAW1>>");
	int i = 0;
	for (i =  0; i< 40; i++) {
	fprintf(stderr, " %x ", arg.buffer[i]);
}
	fprintf(stderr, "<<\n");
	*/


	/* ************************************* */


	char internal_buffer[2352];
	struct cdrom_read arg2;
	arg2.cdread_lba = 3000; /* FIXME: LBA != lsn */
	arg2.cdread_bufaddr = (char *) malloc(2352);
	arg2.cdread_buflen = 2352;

	int rv = ioctl(fd, CDROMREADRAW, &arg2);

	/* debug code */
	/*
	perror("2 -- ");
	fprintf(stderr, "rv = %d\n", rv);

	fprintf(stderr, ">>RAW2>>");
	for (i =  0; i< 40; i++) {
	fprintf(stderr, " %x ", internal_buffer[i]);
}
	fprintf(stderr, "<<\n");
	*/

	close(fd);


	/* int disc_status = ioctl(fd, CDROM_DISC_STATUS, 0); */
	/* debug code */
	/*
	if (disc_status == CDS_NO_INFO) {
	fprintf(stderr, ">>>>>>>>> CDS_NO_INFO <<<<<<<<<<<\n");
} else if (disc_status == CDS_AUDIO) {
	fprintf(stderr, ">>>>>>>>> CDS_AUDIO <<<<<<<<<<<\n");
} else if (disc_status == CDS_MIXED) {
	fprintf(stderr, ">>>>>>>>> CDS_MIXED <<<<<<<<<<<\n");
} else if (disc_status == CDS_XA_2_2) {
	fprintf(stderr, ">>>>>>>>> CDS_XA_2_2 <<<<<<<<<<<\n");
} else if (disc_status == CDS_XA_2_1) {
	fprintf(stderr, ">>>>>>>>> CDS_XA_2_1 <<<<<<<<<<<\n");
} else if (disc_status == CDS_DATA_1) {
	fprintf(stderr, ">>>>>>>>> CDS_DATA_1 <<<<<<<<<<<\n");
} else {
	fprintf(stderr, ">>>>>>>>> ???? <<<<<<<<<<<\n");
}
	*/

	return 0;
}





int raw_read_sector2(lsn_t sector)
{
	struct cdrom_msf msf;
	char buffer[2352];

	msf_t p_msf;

	int fd = open(config.cdrw_device, O_RDWR | O_NONBLOCK);
	if (fd == -1) {
		/* debug code */
		/* fprintf(stderr, "bledny desk. pliku\n"); */
		return -1;

	}

	cdio_lsn_to_msf(sector, &p_msf);
	msf.cdmsf_min0 = p_msf.m;     /* start minute */
	msf.cdmsf_sec0 = p_msf.s;     /* start second */
	msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	msf.cdmsf_min0 = 1;     /* start minute */
	msf.cdmsf_sec0 = 1;     /* start second */
	msf.cdmsf_frame0 = 1;   /* start frame */

	memcpy(buffer, &msf, sizeof(msf));
	int rv = ioctl(fd, CDROMREADRAW, buffer);
	/* debug code */
	/*
	perror("UNO: -- ");
	fprintf(stderr, ": %d\n", rv);
	*/
	close(fd);

	return 0;
}
#endif





cdw_rv_t get_cds_mixed_with_ioctl(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	int fd = open(drive, O_RDONLY);
	if (fd != -1) {
		int ioctl_mode = ioctl(fd, CDROM_DISC_STATUS, 0);
		close(fd);
		if (ioctl_mode != -1) {
			if (ioctl_mode == CDS_MIXED) {
				return CDW_OK;
			} else {
				return CDW_NO;
			}
		} else {
			return CDW_ERROR;
		}
	} else {
		return CDW_ERROR;
	}
}







/**
   \brief Read one line of text, but no longer than 10k of chars

   Function ends the line with '\0';

   \param file - open file that you want to read from

   \return pointer to freshly allocated space with characters read
   \return NULL if reading failed
*/
char *my_readline_10k(FILE *file)
{
	char *buffer = NULL;
	size_t total_len = 0;
	while(1) {
		char fgets_buffer[100];
		int fgets_buffer_size = 100; /* reasonable initial value */

		char *r = fgets(fgets_buffer, fgets_buffer_size, file);
		if (r == fgets_buffer) { /* some content was read to buffer */

			size_t fgets_len = strlen(fgets_buffer);
			total_len += fgets_len;
			char *tmp = (char *) realloc(buffer, total_len + 1);

			if (tmp != buffer) { /* buffer was re-allocated AND moved */
				buffer = tmp;
			}

			char *end = buffer + total_len - fgets_len;
			/* +1 for ending '\0' that is always added by fgets(); */
			strncpy(end, fgets_buffer, fgets_len + 1);

			/* '\n' from file is also read by fgets() and put into string */
			if (*(buffer + total_len - 1) == '\n') {
				// *(buffer + total_len - 1) = '\0';
				/* this also means that last fragment of line was read */
				break;
			}

			if (total_len >= 10000) {
				*(buffer + total_len) = '\0';
				/* I can't imagine any option longer than 10k chars */
				break;
			}


		} else { /* r == NULL: EOF or error */
			break;
		}
	}

	return buffer;
}









/**
   \brief Function inspecting given table of id,label pairs and returning
   label matching given id.

   First argument is table of elements of cdw_id_label_t type.
   Last element of the table must be {-1, "unknown value"}, where -1 serves
   as guard and "unknown value" string that is displayed when there is no
   other match.

   Second argument is id / key of searched string.

   \param table - table of id,label pairs
   \param id -  id to be matched against values in table

   \return pointer to const string that matches given id,
   \return fall back error string if no id matches
*/
const char *cdw_utils_id_label_table_get_label(cdw_id_clabel_t *table, cdw_id_t id)
{
	cdw_assert (table != (cdw_id_clabel_t *) NULL, "ERROR: input table is NULL\n");

#ifndef NDEBUG
	int j = 0;
	/* this piece of code is used to detect potential incorrectly built
	   tables without final guard string */
	while (table[j].label != (char *) NULL) {
		__attribute__((unused)) size_t len = strlen(table[j].label);
		j++;
	}
#endif

	int i = 0;
	/* it turns out that there may be valid IDs == -1,
	   so I'm using (char *) NULL as guard */
	while (table[i].label != (char *) NULL
	       && table[i].id != id) {

		i++;
	}

	return table[i].label; /* may be (char *) NULL */
}





cdw_id_t cdw_utils_id_label_table_get_id(cdw_id_clabel_t *table, const char *label, cdw_id_t guard)
{
	cdw_assert (table, "ERROR: input table is NULL\n");

	if (!label) {
		cdw_vdm ("WARNING: looking for id of NULL string, returning guard \n");
		return guard;
	}

	int i = 0;
	while (table[i].id != guard) {
		if (!strcmp(table[i].label, label)) {
			return table[i].id;
		}

		i++;
	}

	return guard;
}





static cdw_id_clabel_t cdw_rv_t_table[] = {
	{ CDW_OK,             "CDW_OK"        },
	{ CDW_CANCEL,         "CDW_CANCEL"    },
	{ CDW_NO,             "CDW_NO"        },
	{ CDW_ERROR,          "CDW_ERROR"     },
	{ -1,                 (char *) NULL   }}; /* guard */


const char *cdw_utils_get_crv_label(cdw_rv_t crv)
{
	return cdw_utils_id_label_table_get_label(cdw_rv_t_table, crv);
}





cdw_id_clabel_t cdw_bool_type_labels[] = {
	{ CDW_UNKNOWN, "CDW_UNKNOWN" },
	{ CDW_TRUE,    "CDW_TRUE" },
	{ CDW_FALSE,   "CDW_FALSE" },
	{ 0,           (char *) NULL }};


const char *cdw_utils_get_cdw_bool_type_label(int cdw_bool)
{
	return cdw_utils_id_label_table_get_label(cdw_bool_type_labels, cdw_bool);
}





/* boolean, but with three values */
cdw_id_clabel_t cdw_bool_type_chars[] = {
	/* 2TRANS: this string means "unknown type or state";
	   no more than 2 chars */
	{ CDW_UNKNOWN, gettext_noop("?") }, /* also known as "init" */
	/* 2TRANS: "N" like in "No, the disc doesn't have this attribute";
	   no more than 2 chars */
	{ CDW_FALSE,   gettext_noop("N") },
	/* 2TRANS: "Y" like in "Yes, the disc has this attribute";
	   no more than 2 chars */
	{ CDW_TRUE,    gettext_noop("Y") },
	{ -1,          (char *) NULL     }}; /* guard */


const char *cdw_utils_get_cdw_bool_type_char(int cdw_bool)
{
	cdw_assert (cdw_bool == CDW_UNKNOWN || cdw_bool == CDW_FALSE || cdw_bool == CDW_TRUE,
		    "ERROR: invalid value of cdw_bool: %d\n", cdw_bool);

	return cdw_utils_id_label_table_get_label(cdw_bool_type_chars, cdw_bool);
}





/**
   \brief Helper function for qsort() - compare two integer values

   Function that is used as compar() argument to qsort() function.
   It returns -1, 0 or 1  if first argument is less than, equal or
   larger than second.

   Passing this function to qsort() results in table of ints sorted
   from lowest to highest.

   \param int1 - first integer to compare
   \param int2 - second integer to compare

   \return -1, 0 or 1  if first int is less than, equal or larger than second
 */
int cdw_utils_compare_ints(const void *int1, const void *int2)
{
	const int *i1 = (const int *) int1;
	const int *i2 = (const int *) int2;

	if (*i1 < *i2) {
		return -1;
	} else if (*i1 > *i2) {
		return 1;
	} else {
		return 0;
	}
}





/**
   \brief Helper function for qsort() - compare two integer values

   Function that is used as compar() argument to qsort() function.
   It returns -1, 0 or 1  if first argument is less than, equal or
   larger than second.

   Passing this function to qsort() results in table of ints sorted
   from lowest to highest.

   \param int1 - first integer to compare
   \param int2 - second integer to compare

   \return -1, 0 or 1  if first int is less than, equal or larger than second
 */
int cdw_utils_compare_ints_reverse(const void *int1, const void *int2)
{
	const int *i1 = (const int *) int1;
	const int *i2 = (const int *) int2;

	if (*i1 > *i2) {
		return -1;
	} else if (*i1 < *i2) {
		return 1;
	} else {
		return 0;
	}
}





/**
   \brief Calculate estimated time of accomplishment for a task, convert this time to string

   You can calculate \p eta like this:
    - eta = amount left / speed
    - speed = amount done / time from beginning of process
    - eta = (amount left / amount done) * time from beginning of process

    or:
    int eta = (int) (((1.0 * todo_size) / done_size) * time_from_start);

    \param eta_string - output string with ETA in hh:mm:ss format
    \param eta - input eta value (most probably in seconds ;) )
    \param n - size of output buffer (does not include space for ending '\0')
*/
void cdw_utils_eta_calculations(char *eta_string, int eta, size_t n)
{
	cdw_sdm ("ETA input value: %d\n", eta);
	if (eta < 0) { /* it may happen at the beginning of process */
		/* 2TRANS: ETA is Expected Time of Arrival, time to finish a task;
		   three "??" values are number of hours, minutes seconds left,
		   "??" are used when the time cannot be calculated.
		   The string will be displayed in progress dialog window. Keep short. */
		snprintf(eta_string, n + 1, _("ETA: ??:??:??"));
	} else {
		/* C99 truncates result of division towards zero (C: A Reference manual, 5th ed., 7.6.1.) */

		int eta_hour = (eta / 60) / 60;
		/* extract number of seconds in full hours from total number of seconds */
		int eta_min = (eta - ((eta_hour * 60) * 60)) / 60;
		/* extract number of seconds in full hours and in full minutes from total number of seconds */
		int eta_sec = eta - (((eta_hour * 60) * 60) + (eta_min * 60));

		/* 2TRANS: ETA is Expected Time of Arrival, time to finish a task;
		   three "%d" values are number of hours, minutes seconds left. The
		   string will be displayed in progress dialog window. Keep short. */
		snprintf(eta_string, n + 1, _("ETA: %d:%02d:%02d"), eta_hour, eta_min, eta_sec);
	}
	cdw_sdm ("ETA output value: %s\n", eta_string);

	return;
}





int cdw_utils_compress_table_of_ints(int *table, int n_max)
{
	table[0] = table[0];
	int j = 1;
	for (int i = 1; i < n_max; i++) {
		if (table[i] == table[j - 1]) {
			;
		} else {
			table[j] = table[i];
			j++;
		}
	}
	return j;
}




/* *********************** */
/* *** unit tests code *** */
/* *********************** */

//#define CDW_UNIT_TEST_CODE /* definition used during development of unit tests code */
#ifdef CDW_UNIT_TEST_CODE



static void test_cdw_utils_eta_calculations(void);
static void test_cdw_utils_compress_table_of_ints(void);
static void test_cdw_utils_id_label_table_get_label(void);
static void test_cdw_utils_id_label_table_get_id(void);

void cdw_utils_run_tests(void)
{
	fprintf(stderr, "testing cdw_utils.c\n");

	test_cdw_utils_eta_calculations();
	test_cdw_utils_compress_table_of_ints();
	test_cdw_utils_id_label_table_get_label();
	test_cdw_utils_id_label_table_get_id();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_utils_eta_calculations(void)
{
	fprintf(stderr, "\ttesting cdw_utils_eta_calculations()... ");

	char buf[50 + 1];

	struct {
		size_t buf_len; /* this test may use subbuffers of various lengths */
		int input_eta;
		const char *expected_result;
	} test_data[] = {
		{  20,  0, "ETA: 0:00:00" },
		{  20,  1, "ETA: 0:00:01" },
		{  20,  2, "ETA: 0:00:02" },
		{  20,  3, "ETA: 0:00:03" },
		{  20, 10, "ETA: 0:00:10" },
		{  20, 60, "ETA: 0:01:00" },
		{  20, 61, "ETA: 0:01:01" },
		{  20, 20 * 60 + 4,                    "ETA: 0:20:04" },
		{  20, (17 * 60 * 60) + (21 * 60) + 4, "ETA: 17:21:04" },
		{  10, (17 * 60 * 60) + (21 * 60) + 4, "ETA: 17:21" },

		{   0,  0, (char *) NULL }};

	int i = 0;
	while (test_data[i].expected_result != (char *) NULL) {
		cdw_utils_eta_calculations(buf, test_data[i].input_eta, test_data[i].buf_len);
		int rv = strcmp(test_data[i].expected_result, buf);
		cdw_assert_test (rv == 0, "ERROR: failed test #%d, eta = %d\n", i, test_data[i].input_eta);
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_utils_compress_table_of_ints(void)
{
	fprintf(stderr, "\ttesting cdw_utils_compress_table_of_ints()... ");

#define TABLE_SIZE 18
	struct {
		int table_in[TABLE_SIZE];
		int table_out[TABLE_SIZE];
		int expected_n;
	} test_data[] = {
		{ { 0, 1, 1, 2, 3, 4, 4, 4, 4,  6, 6, 6, 6, 7, 9, 9, 9, 10 },
		  { 0, 1, 2, 3, 4, 6, 7, 9, 10, 6, 6, 6, 6, 7, 9, 9, 9, 10 },
		  9 },

		{ { 0, 1, 2, 3, 3, 3, 4, 5,  5,  7,  8, 8, 8, 10, 10, 10, 11, 12 },
		  { 0, 1, 2, 3, 4, 5, 7, 8, 10, 11, 12, 8, 8, 10, 10, 10, 11, 12 },
		  11 },

		{ { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
		  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
		  1 },

		{ { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
		  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
		  -1 /* guard */ } };

	int i = 0;
	while (test_data[i].expected_n != -1) {
		int rv = cdw_utils_compress_table_of_ints(test_data[i].table_in, TABLE_SIZE);
		cdw_assert (rv == test_data[i].expected_n,
			    "ERROR: data set #%d: expected number of unique elements = %d, result is %d\n",
			    i, test_data[i].expected_n, rv);
		for (int j = 0; j < rv; j++) {
			cdw_assert (test_data[i].table_in[j] == test_data[i].table_out[j],
				    "ERROR: data set #%d: elements #%d do not match (%d != %d)\n",
				    i, j, test_data[i].table_in[j], test_data[i].table_out[j]);
		}
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_utils_id_label_table_get_label(void)
{
	fprintf(stderr, "\ttesting cdw_utils_id_label_table_get_label()... ");

	cdw_id_clabel_t test_table[] = {
		{    23,   "hello",               },
		{    76,   "milky",               },
		{   111,   "Tuesday",             },
		{   487,   "robotic",             },
		{ -5235,   "normalized",          },
		{   385,   "every day",           },
		{   244,   " so called solution " },
		{   444,   " orange",             },
		{    98,   (char *) NULL,         }};


	int i = 0;
	while (test_table[i].label) {
		const char *label = cdw_utils_id_label_table_get_label(test_table, test_table[i].id);
		cdw_assert (!strcmp(label, test_table[i].label),
			    "ERROR: data set #%d: expected label \"%s\", got label \"%s\" (id = %lld)\n",
			    i, test_table[i].label, label, test_table[i].id);
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_utils_id_label_table_get_id(void)
{
	fprintf(stderr, "\ttesting cdw_utils_id_label_table_get_id()... ");

	cdw_id_t the_guard = 495;

	cdw_id_clabel_t test_table[] = {
		{        23,   "hello",               },
		{        76,   "milky",               },
		{       111,   "Tuesday",             },
		{       487,   "robotic",             },
		{     -5235,   "normalized",          },
		{       385,   "every day",           },
		{       244,   " so called solution " },
		{       444,   " orange",             },
		{        98,   "swift ",              },
		{ the_guard,   "closed",              }};

	int i = 0;
	while (test_table[i].id != the_guard) {
		cdw_id_t id = cdw_utils_id_label_table_get_id(test_table, test_table[i].label, the_guard);
		cdw_assert (id == test_table[i].id,
			    "ERROR: data set #%d: expected id = %lld, got id = %lld (label = \"%s\")\n",
			    i, test_table[i].id, id, test_table[i].label);
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}




#endif /* #ifdef CDW_UNIT_TEST_CODE */
