/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _GNU_SOURCE /* needed by isblank(), see 'man isblank' */

#include <math.h> /* floor() */
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>

#include "cdw_string.h"
#include "cdw_debug.h"



/**
   \file cdw_string.c
   \brief Simple string utilities

   Few utility functions that handle (char *) strings.
   You will probably find some of them in a modern C library that
   tries to implement something more than ANSI C90.
*/



static size_t cdw_string_wrap_start(size_t width, size_t len, int align);
static size_t cdw_string_get_line_len(const char *string, size_t chars_max);
static size_t cdw_string_count_lines_new(const char *string, size_t width_max);


/**
   \brief Return pointer to first char in string not being space, "\t", "\r" or "\n"

   Returned pointer points to part of string given as an argument, so it
   cannot be free()d.

   Returned pointer may point to string of length = 0 - that means that
   original string consisted only of white characters.

   \p str has to be standard C string ended with "\0".

   \param str - string which should be trimmed from left side

   \return pointer to first char in argument string str, which is neither space, nor "\t", "\n" or "\r" character
   \return NULL if argument is NULL
   \return unchanged argument if argument is an empty string
*/
char *cdw_string_ltrim(char *str)
{
	if (str == NULL) {
		return NULL;
	}

	if (*str == '\0') {
		return (char *) str;
	}

	/* The  strspn()  function calculates the length of the
	   initial segment of first argument which consists entirely
	   of characters in second argument. */
	return str + strspn(str, " \t\r\n");
}





/**
   \brief Remove all spaces, "\t", "\r" or "\n" from tail of string

   Function removes all spaces, "\t", "\n" and "\r" chars from tail of string by
   inserting "\0" char after last char that is not space nor tab character.

   \param str - string to be trimmed at the end, it has to be ended with "\0"

   \return pointer to string trimmed from the right side,
   \return NULL if argument is NULL
   \return unchanged argument if argument is an empty string
 */
char *cdw_string_rtrim(char *str)
{
	if (str == NULL) {
		return NULL;
	}

	if (*str == '\0') {
		return str;
	}

	char *p;
	for (p = str; *p; p++)
		;

	p--;
	while ( (*p == ' ') || (*p == '\t') || (*p == '\r') || (*p == '\n') ) {
		if (p == str) {
			p--;
			break; /* read note below */
		} else {
			p--;
		}

	}

	/* (p == str) condition sooner or later becomes true in case of
	   'str' values being all blank characters; in a moment when this
	   condition is met, 'p--' will move 'p' pointer too far left and
	   will reach memory outside allocated space; checks of value of
	   such memory cells performed in 'while' loop ( (*p == ' ') etc)
	   are in fact invalid reads beyond 'our' memory; valgrind
	   complains about them, so we 'break' the loop to avoid such
	   reads and to avoid littering valgrind log; such reads seem to
	   be harmless on my box, but just to be on a safe side... */

	*++p = '\0';
	return str;
}





char *cdw_string_trim(const char *string)
{
	if (!string) {
		cdw_vdm ("WARNING: passed NULL string to the function\n");
		return (char *) NULL;
	}

	size_t length = strlen(string);
	size_t len = length;

	/* walk white chars starting from the end */
	int i = (int) len - 1; /* watch out for "" input strings, keep the 'i >= 0' condition below */
	while (isspace(string[i]) && i >= 0) {
		i--;
		length--;
	}

	/* walk white chars at the beginning */
        size_t start = 0;
	/* last part of a loop test is for 'length' value
	   flipping through zero to MAX */
	while (isspace(string[start]) && start < len && length <= len) {
		start++;
		length--;
	}

	length = length > len ? 0 : length;

	char *out = (char *) NULL;
	/* not sure how strndup() behaves with zero length */
	if (length) {
		out = strndup(string + start, length);
	} else {
		out = strdup("");
	}
	out[length] = '\0';
	return out;
}





bool cdw_string_is_all_spaces(const char *string)
{
	size_t len = strlen(string);
	for (size_t i = 0; i < len; i++) {
		if (!isspace(string[i])) {
			return false;
		}
	}
	return true;
}





/* GPL 2 (or later) code from
gcc-2.95.2/gcc-core-2.95.2.tar.gz/gcc-2.95.2/libiberty/concat.c
*/
char *cdw_string_concat(const char *first, ...)
{
	register size_t length;
	register char *newstr;
	register char *end;
	register const char *arg;
	va_list args;

	/* First compute the size of the result and get sufficient memory. */

	va_start (args, first);

	if (first == NULL) {
		length = 0;
	} else {
		length = strlen (first);
		while ((arg = va_arg (args, const char *)) != NULL) {
			length += strlen (arg);
		}
	}

	newstr = (char *) malloc (length + 1);
	va_end (args);

	/* Now copy the individual pieces to the result string. */

	if (newstr != NULL) {
		va_start (args, first);
		end = newstr;
		if (first != NULL) {
			arg = first;
			while (*arg) {
				*end++ = *arg++;
			}
			while ((arg = va_arg (args, const char *)) != NULL) {
				while (*arg) {
					*end++ = *arg++;
				}
			}
		}
		*end = '\0';
		va_end (args);
	}

	return (newstr);
}





/**
   \brief Append one string to another

   Append 'tail' string to 'head' string. 'head' is reallocated, that is why
   you have to pass a pointer to a string as a first arg.
   Result is a new 'head': pointer to concatenated 'head' and 'tail'.
   Old 'head' is freed, so make sure that 'head is dynamically allocated
   before passing it to append().

   \param head - the string that you want to append sth to
   \param tail - the string that you want to append to tail

   \return CDW_ERROR if some malloc/concat error occurred
   \return CDW_OK if success
 */
cdw_rv_t cdw_string_append(char **head, const char *tail)
{
	char *tmp = cdw_string_concat(*head, tail, NULL);
	if (tmp == NULL) {
		return CDW_ERROR;
	} else {
		free(*head);
		*head = tmp;
		return CDW_OK;
	}
}





/**
   \brief Check if given string doesn't contain any strange characters

   Make sure that string entered by user can't be used to do sth bad
   to/with application. Invalid characters are specified in cdw_string.h
   (see definition of CDW_STRING_UNSAFE_CHARS_STRING).

   Function accepts NULL pointers but then it returns CDW_GEN_ERROR.
   Empty strings ("") are treated as valid strings.

   \p invalid should be a string of length 2. If \p input has any invalid
   chars, first of them will be put into first char of \p invalid.
   \p invalid may be NULL, then function will not produce any output.

   \param string - string to be checked
   \param invalid - place where function will place first unsafe char from input_string

   \return CDW_NO if string contains any char that is not valid
   \return CDW_OK if string contains only valid characters
   \return CDW_GEN_ERROR if input_string is NULL
 */
cdw_rv_t cdw_string_security_parser(const char *string, char *invalid)
{
	if (string == (char *) NULL) {
		cdw_vdm ("ERROR: passing NULL string to the function\n");
		return CDW_ERROR;
	}

	char *c = strpbrk(string, CDW_STRING_UNSAFE_CHARS_STRING);
	if (c == (char *) NULL) {
		/* no unsafe chars */
		return CDW_OK;
	} else {
		if (invalid == (char *) NULL) {
			cdw_vdm ("WARNING: buffer for invalid char is NULL\n");
		} else {
			strncpy(invalid, c, 1);
			invalid[1] = '\0';
		}
		return CDW_NO;
	}
#if 0

	size_t l = strlen(string);
	for (size_t i = 0; i < l; i++) {
		int c = (int) string[i];
		if (isalnum(c) || isblank(c)
			|| string[i] == '-'
		        || string[i] == '~'
			|| string[i] == '.'
			|| string[i] == ','
			|| string[i] == '_'
			|| string[i] == '/'
			|| string[i] == '='
		        || string[i] == '+'
			|| string[i] == ':') {
			;
		} else {
			return CDW_NO;
		}
	}

	return CDW_OK;
#endif
}





/**
   \brief Replace content of target string with content of source string

   Copy string from source to target. If target space is not allocated,
   the function will do this. If target already points to some string,
   old target string will be overwritten, and target pointer may be modified.

   target pointer must be either empty pointer or pointer returned by
   malloc, realloc or calloc.

   *target is set to NULL if there are some (re)allocation errors.

   \param target - pointer to target string that you want to modify
   \param source - source string that you want to copy to target

   \return CDW_ERROR on errors
   \return CDW_OK on success
 */
cdw_rv_t cdw_string_set(char **target, const char *source)
{
	if (source == NULL) {
		return CDW_ERROR;
	}

	size_t len = strlen(source);

	char *tmp = (char *) realloc(*target, len + 1);
	if (tmp == NULL) {
		cdw_vdm ("ERROR: failed to realloc target string \"%s\" (source string = \"%s\")\n", *target, source);
		return CDW_ERROR;
	} else {
		*target = tmp;
	}

	strncpy(*target, source, len + 1);

	return CDW_OK;
}





/**
   \brief Check if given string starts with another string

   Silly wrapper for function that compares two strings. To be more precise
   the function compares only first n=strlen(substring) chars of both strings
   to check if first argument starts with given string (second argument).
   I could use strncmp(), but I want to avoid putting call to strlen() every
   time I need to check what a string starts with.

   You can use this function to check lines read from color configuration file.
   Make sure that 'line' starts with non-white chars. The search is
   case-insensitive (hence 'ci' in name).

   \param line - string that you want to check (to check if it starts with given substring)
   \param substring - field name that you are looking for in given line

   \return true if given line starts with field name,
   \return false otherwise
*/
bool cdw_string_starts_with_ci(const char *line, const char *substring)
{
	if (!strncasecmp(line, substring, strlen(substring))) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Free all strings in a given table

   Free list of strings that were malloc()ed. Free()d pointers are
   set to (char *) NULL. Table itself is not free()d.

   You can use this function e.g. to free labels of dropdowns.

   \param labels - table of labels that you want to free
   \param n_max - size of table (maximal number of strings to be free()d)
*/
void cdw_string_free_table_of_strings(char *labels[], int n_max)
{
	if (labels != (char **) NULL) {
		for (int i = 0; i < n_max; i++) {
			if (labels[i] != (char *) NULL) {
				free(labels[i]);
				labels[i] = (char *) NULL;
			}
		}
	}

	return;
}





/**
   \brief Create table of labels that represent given numbers

   Using int values from table of ints create table of (char *) strings
   (lables) that represent these ints. Inst should be no longer than
   3 digits.

   Last element of the list is set to (char *) NULL.
   \p labels is a pointer to already alocated table. Table size must be big
   enough to store all labels and one guard element at the end.

   \param labels - table where labels will be stored
   \param ints - input values (in a table) to be converted
   \param n - number of ints to convert, +1 for guard in result table

   \return CDW_ERROR on malloc() error
   \return CDW_OK on success
*/
cdw_rv_t cdw_string_ints2labels_3(char **labels, int ints[], size_t n)
{
	cdw_assert (labels != (char **) NULL, "you should alloc labels table\n");
	cdw_assert (n > 0, "incorect size of labels table: n = %d\n", (int) n);

	size_t i = 0;
	for (i = 0; i < n; i++) {
		labels[i] = (char *) malloc(3 + 1); /* 3 digits + 1 ending '\0' */
		if (labels[i] == NULL) {
			return CDW_ERROR;
		}
		memset(labels[i], ' ', 4);

		snprintf(labels[i], 3 + 1, "%d", ints[i]);

		cdw_sdm ("input int = %d, constructed labels[%d] = \"%s\"\n", ints[i], i, labels[i]);
	}
	/* obligatory ending element (guard) */
	labels[n] = (char *) NULL;

	return CDW_OK;
}





/**
   \brief Map several possible values of option to true/false values

   Function compares value in \p buffer with several defined strings to
   check if value in \p buffer should be treated as 'true' or 'false'.

   If string in \p buffer is recognized (either as having meaning of 'true'
   or 'false'), appropriate bool value is assigned to \p value and CDW_OK
   is returned. Otherwise value of \p value is not changed and CDW_ERROR
   is returned.

   There are several values of \p buffer that can be mapped into 'true',
   and several other values that can be mapped to 'false'. Case of string
   in \p buffer is not significant.

   If \p buffer is NULL function returns CDW_ERROR.

   Remember to remove leading and trailing white chars from \p buffer
   before passing it to the function.

   \param buffer - buffer that you want to check
   \param value - variable that you want to assign true/false to

   \return CDW_OK if value in first argument was recognized
   \return CDW_ERROR if value in first argument was not recognized or first argument is NULL
*/
cdw_rv_t cdw_string_get_bool_value(const char *buffer, bool *value)
{
	if (buffer == (char *) NULL) {
		return CDW_ERROR;
	}

	if (!strcasecmp(buffer, "1") || !strcasecmp(buffer, "yes") || !strcasecmp(buffer, "true")) {
		*value = true;
		return CDW_OK;
	} else if (!strcasecmp(buffer, "0") || !strcasecmp(buffer, "no") || !strcasecmp(buffer, "false")) {
		*value = false;
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Check if given string can be printed by ncurses, if not then
   produce printable representation

   Function uses mbstowsc() to check if \p string can be safely printed
   in ncurses window (i.e. if it contains printable characters only).

   If there are any non-printable characters, a pointer to freshly alloced
   printable representation of \p string is returned. Caller is responsible
   for free()ing such string. If \p string contain only printable characters,
   NULL is returned.

   \param string - string to be checked

   \return NULL if input string can be correctly displayed
   \return pointer to printable representation of input string if input string can't be correctly displayed
*/
char *cdw_string_get_printable_if_needed(const char *string)
{
	size_t len = strlen(string);
	wchar_t *wide_string = (wchar_t *) malloc((len + 1) * (sizeof(wchar_t)));
	/* mbstowcs() converts multibyte 'char *' strings into 'wchar_t' strings */
	size_t conv = mbstowcs(wide_string, string, len);
	free(wide_string);
	wide_string = (wchar_t *) NULL;

	if (conv == (size_t) -1) {
		/* there is some non-printable character in "string" */
		char *printable_string = strdup(string);
		if (printable_string == (char *) NULL) {
			cdw_assert (0, "ERROR: failed to strdup() string \"%s\"\n", string);
			return (char *) NULL;
		}
		for (size_t i = 0; i < len; i++) {
			if ((unsigned char) printable_string[i] > 127) {
				printable_string[i] = '?';
			}
		}
		return printable_string;
	} else {
		return (char *) NULL;
	}
}





/**
   \brief Get last char in a last word, in specified range

   Search in given \p string for last non-white character in last word.
   Range of the search is specified by \p chars_max: area of search starts
   from the beginning of string to p_chars - 1. There are some corner
   cases, hopefully they will be explained by examples below (just remember
   that chars_max is counted from one, and return value is counted from 0).

   "test string"/3 -> 2
   "test string"/6 -> 3
   "test string"/11 -> 10 (special case: limit == string length)
   "test string"/12 -> 10

   \param string - string to be searched
   \param chars_max - maximal number of characters to check for word's end

   \return index of last non-white character in a word, in given range
*/
size_t cdw_string_get_line_end(const char *string, size_t chars_max)
{
	size_t len = strlen(string);

	/* first decide how many chars from the start of string are
	   to be checked */
	size_t range = 0;
	if (len == 0) {
		range = 0;
		return 0;
	} else if (len < chars_max) {
		range = len;
	} else if (len == chars_max) {
		range = len;
	} else {
		range = chars_max;
	}

	/* '\n' is considered an end of line, so we need to search
	   for first occurrence of the char in string */
	for (size_t j = 0; j < range; j++) {
		if (string[j] == '\n') {
			return j == 0 ? 0 : j - 1;
		}
	}

	/* few special cases */
	if (range == 0) {
		return 0;
	} else if (len == chars_max) {
		return range - 1;
	} else if (len < chars_max) {
		return len - 1;
	} else {
		;
	}


	/* what is left is simple case: searching for a white char
	   from end till start of string */
	size_t i;
	size_t last_space = 0;
	for (i = range - 1; ; i--) {
		if (isspace(string[i])) {
			last_space = i;
		}
		if (last_space != 0 && !isspace(string[i]) && string[i] != '\0') {
			/* a char preceding last space in string,
			   this is what we are searching for */
			return i;
		}
		if (i == 0) {
			break;
		}

	}
	/* most probably zero, but caller will have to deal with it himself */
	return i == 0 ? range - 1 : i;
}





size_t cdw_string_get_line_len(const char *string, size_t chars_max)
{
	size_t len = strlen(string);
	if (len == 0) {
		cdw_sdm ("INFO: returning 0 for empty string\n");
		return 0;
	}

	/* first decide how many chars from the start of string are
	   to be checked */
	size_t range = 0;
	if (len <= chars_max) {
		range = len;
	} else {
		range = chars_max;
	}

	if (string[0] == '\n') {
		cdw_sdm ("INFO: returning 1 for '\n' string\n");
		return 1;
	}

	/* '\n' is considered an end of line, so we need to search
	   for first occurrence of the char in string */
	for (size_t j = 0; j < range; j++) {
		if (string[j] == '\n') {
			cdw_sdm ("INFO: returning %zd on '\n' (\"%s\")\n", j, string);
			return j;
		}
	}

	/* few special cases */
	if (range == 0) {
		cdw_vdm ("INFO: returning 0 on zero range\n");
		return 0;
	} else if (len == chars_max) {
		cdw_sdm ("INFO: returning %zd on len == chars_max (\"%s\")\n", len, string);
		return len;
	} else if (len < chars_max) {
		cdw_sdm ("INFO: returning %zd on len < chars_max (\"%s\")\n", len, string);
		return len;
	} else {
		;
	}

	if (len > chars_max && isspace(string[chars_max])) {
		cdw_sdm ("INFO: returning %zd on ending space (\"%s\")\n", chars_max, string);
		return chars_max;
	}

	/* searching for a white char from end towards start of string */
	size_t i;
	for (i = range - 1; ; i--) {
		if (string[i] == '\0' || isspace(string[i])) {
			break;
		}

		if (i == 0) {
			i = range;
			break;
		}
	}

	cdw_sdm ("INFO len of string \"%s\" is %zd (chars max = %zd)\n", string, i, chars_max);

	return i;
}




/**
   \brief Wrap and align a string

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function wraps given text using \p width_max as a constraint,
   and aligns it to the left, center or right.
   Wrapping occurs on word boundaries (if possible). Newline
   characters ("\n") embedded in input string are respected.

   Resulting string can be printed in a window or form field of
   given \p width_max. Text returned by this function will look
   like this (line breaks added for readability, location of NUL
   is approximate):

   "some text in first line     "
   "in second                   "
   "and some more text in third\0"

   Text will be aligned according to \p align (left, center, right).
   See cdw_string.h for symbolic constants.

   Function doesn't modify \p string, it returns modified copy of it.

   \param string - text buffer with text to be wrapped
   \param width_max - maximal width (length) of lines
   \param align - align of text in lines.

   \return modified string on success
   \return NULL pointer on failure
*/
char *cdw_string_wrap(const char *string, size_t width_max, int align)
{
	/* Input text buffer may have arbitrary length, but output text
	   buffer's length must be a multiple of width_max. */
	size_t n_lines = cdw_string_count_lines_new(string, width_max);
	size_t out_len = width_max * n_lines;
	char *retval = malloc(out_len + 1);
	memset(retval, ' ', out_len + 1);

	const char *input = string;
	char *output = retval;

	/* copy text from input to output */
	while (1) {
		/* function returns zero-based index */
		size_t line_len = cdw_string_get_line_len(input, width_max);
		if (line_len == 0) {
			break;
		} else if (line_len == 1 && input[0] == '\n') {
			input++;
			continue;
		} else {
			;
		}

		size_t begin = cdw_string_wrap_start(width_max, line_len, align);
		snprintf(output + begin, line_len + 1, "%s", input);

		output[begin + line_len] = ' '; /* overwrite trailing NUL, we don't want to end the string just yet */

		input = input + line_len;
		output = output + width_max;

		while (isspace(*input) && *input != '\0') {
			input++;
		}
	}

	retval[out_len] = '\0';

	cdw_sdm ("INFO: returning %zd lines in \"%s\"\n", n_lines, retval);

	return retval;
}





/**
   \brief Calculate starting point of text aligned in a line

   \date Function's top-level comment reviewed on 2012-01-06
   \date Function's body reviewed on 2012-01-06

   Helper function for cdw_string_wrap(), calculating position of start
   of text in logical line. The position (for constant line width and
   constant text length) may vary depending on \p align
   (left, center, right).
   Returned value is rounded down, not up (if needed).

   Function returns index/position from which an aligned text should start.

   Example:
   line: "               " -> width = 15;
   text: "moon river"      -> len = 10
   align = CDW_ALIGN_CENTER
   return value = 2 (rounded down; rounded up would be 3);

   \param width - width of a line in which an aligned text will be put
   \param len - length of text
   \param align - align requested (see cdw_string.h) for details

   \return starting position for aligned string
*/
size_t cdw_string_wrap_start(size_t width, size_t len, int align)
{
	if (align == CDW_ALIGN_LEFT) {
		return 0;
	} else if (align == CDW_ALIGN_CENTER) {
		/* using intermediate variable to avoid compiler's warning */
		double tmp = floor((double) (width - len) / 2);
		return (size_t) tmp;
	} else { /* align == CDW_ALIGN_RIGHT */
		return width - len;
	}
}





/**
   \brief Count how many lines would be in a string if it was wrapped

   Count how many lines would there be in a \p string, if it was wrapped
   with \p width_max as a constraint on line width.
   In other words: how many lines would there be if the \p string was
   passed to cdw_string_wrap().

   \param string - string in which to count lines
   \param width_max - intended maximal width of text after wrapping

   \return number of lines in text-to-be-wrapped
*/
size_t cdw_string_count_lines(const char *string, size_t width_max)
{
	cdw_assert (width_max > 5, "ERROR: maximal width is far too small\n");
	cdw_assert (string != (char *) NULL, "ERROR: text to wrap is NULL\n");

	const char *start = string;
	size_t len = strlen(start);
	size_t n_lines = 0;

	for (size_t i = 0; i < len; ) {
		size_t n = cdw_string_get_line_end(start + i, width_max);
		for (size_t a = 0; a <= n; a++) {
			if (*(start + i + a) == '\n') {
				n = a;
				n_lines++;
				break;
			}
		}

		i += n + 1;
		n_lines++;

		/* there may be white char that followed word's end */
		if (isspace(*(start + i)) && *(start + i) != '\0') {
			i++;
		}
		if (n == 0) {
			break;
		}
	}

	cdw_sdm ("INFO: returning %zd for \"%s\"\n", n_lines == 0 ? 1 : n_lines, string);
	/* there should be at least one line, always;
	   but we shouldn't do here 'return ++n_lines' */
	return n_lines == 0 ? 1 : n_lines;
}





size_t cdw_string_count_lines_new(const char *string, size_t width_max)
{
	cdw_assert (width_max > 5, "ERROR: maximal width is far too small\n");
	cdw_assert (string, "ERROR: text to wrap is NULL\n");

	size_t n_lines = 0;

	size_t n = 0;
	size_t i = 0;
	while ((n = cdw_string_get_line_len(string + i, width_max))) {
		if (n == 1 && *(string + i) == '\n') {
			i++;
			continue;
		}
		i += n;
		n_lines++;
	}

	cdw_sdm ("INFO: returning %zd for \"%s\"\n", n_lines == 0 ? 1 : n_lines, string);
	return n_lines;
}





/**
   \brief Predicate function comparing two strings

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   This is just a wrapper around strcmp(), suitable for passing
   as a predicate function to cdw_dll_append() and cdw_dll_is_member().

   Don't pass NULL strings to the function.

   \param string1 - first string for comparison
   \param string2 - second string for comparison

   \return true if the two strings are equal
   \return false otherwise

*/
bool cdw_string_equal(const void *string1, const void *string2)
{
	cdw_assert (string1, "ERROR: string1 is NULL\n");
	cdw_assert (string2, "ERROR: string2 is NULL\n");
	return !strcmp(string1, string2);
}





/**
   \brief Delete a C string

   Deallocate a string pointed to by \p string. Set the string pointer
   (but not the pointer to pointer) to NULL.

   The function is intended to replace the following pattern:

   if (string) {
           free(string);
	   string = (char *) NULL;
   }

   There are so many places where I deallocate string this way that I
   finally decided to make a wrapper for it.

   \param string - pointer to string to free
*/
void cdw_string_delete(char **string)
{
	cdw_assert (string, "ERROR: function argument is NULL\n");

	if (*string) {
		free(*string);
		*string = (char *) NULL;
	}
	return;
}





//#define CDW_UNIT_TEST_CODE
#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void test_cdw_string_set(void);
static void test_cdw_string_security_parser(void);
static void test_cdw_string_starts_with_ci(void);
static void test_cdw_string_free_table_of_strings(void);
static void test_cdw_string_rtrim(void);
static void test_cdw_string_ltrim(void);
static void test_cdw_string_trim(void);
static void test_cdw_string_is_all_spaces(void);
static void test_cdw_string_concat(void);
static void test_cdw_string_append(void);
static void test_cdw_string_get_bool_value(void);
//static void test_cdw_string_get_words_end(void);
static void test_cdw_string_get_line_end(void);
static void test_cdw_string_get_line_len(void);
static void test_cdw_string_wrap(void);
static void test_cdw_string_equal(void);
static void test_cdw_string_count_lines_new(void);

void cdw_string_run_tests(void)
{
	fprintf(stderr, "testing cdw_string.c\n");

	test_cdw_string_set();

	test_cdw_string_security_parser();
	test_cdw_string_starts_with_ci();

	test_cdw_string_rtrim();
	test_cdw_string_ltrim();
	test_cdw_string_trim();
	test_cdw_string_is_all_spaces();

	test_cdw_string_free_table_of_strings();

	test_cdw_string_concat();
	test_cdw_string_append();

	test_cdw_string_get_bool_value();

	test_cdw_string_get_line_end();
	test_cdw_string_get_line_len();

	test_cdw_string_wrap();

	test_cdw_string_equal();

	test_cdw_string_count_lines_new();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_string_set(void)
{
	fprintf(stderr, "\ttesting cdw_string_set()... ");

	int d = 0;

	/* initialization of data */

	/* note that second string is longer than first,
	   and third shorter than second */
	const char *buffer1 = "my small test string, but not too short";
	const char *buffer2 = "Beautiful world, isn't it? A beautiful world behind the firewall. I should go out more often. But not too far from my terminal.";
	const char *buffer3 = "short string";

	char *test_string1 = strdup(buffer1);
	assert(test_string1 != (char *) NULL);

	char *test_string2 = NULL;

	cdw_rv_t crv = CDW_NO;

	/* first test with source string == NULL */
	crv = cdw_string_set(&test_string1, (char *)NULL);
	assert(crv == CDW_ERROR);
	/* in this case first argument should be intact */
	d = strcmp(test_string1, buffer1);
	assert(d == 0);


	/* test with target string already having some content */
	crv = cdw_string_set(&test_string1, buffer2);
	assert(crv == CDW_OK);
	d = strcmp(test_string1, buffer2);
	assert(d == 0);


	/* test with the same target string, but this time set it to much shorter string */
	crv = cdw_string_set(&test_string1, buffer3);
	assert(crv == CDW_OK);
	d = strcmp(test_string1, buffer3);
	assert(d == 0);


	/* set null target to some value */
	crv = cdw_string_set(&test_string2, buffer1);
	assert(crv == CDW_OK);
	d = strcmp(test_string2, buffer1);
	assert(d == 0);


	free(test_string1);
	test_string1 = NULL;

	free(test_string2);
	test_string2 = NULL;

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_security_parser(void)
{
	fprintf(stderr, "\ttesting cdw_string_security_parser()... ");

	struct {
		const char *input;
		const char *output;
		cdw_rv_t crv;
	} test_data[] = { { (char *) NULL, (char *) NULL, CDW_ERROR },
			  { "test*test",   "*",   CDW_NO },
			  { "*testtest",   "*",   CDW_NO },
			  { "testtest*",   "*",   CDW_NO },
			  { "test&test",   "&",   CDW_NO },
			  { "&testtest",   "&",   CDW_NO },
			  { "testtest&",   "&",   CDW_NO },
			  { "test!test",   "!",   CDW_NO },
			  { "!testtest",   "!",   CDW_NO },
			  { "testtest!",   "!",   CDW_NO },
			  { "test;test",   ";",   CDW_NO },
			  { ";testtest",   ";",   CDW_NO },
			  { "testtest;",   ";",   CDW_NO },
			  { "test`test",   "`",   CDW_NO },
			  { "`testtest",   "`",   CDW_NO },
			  { "testtest`",   "`",   CDW_NO },
			  { "test\"test",  "\"",  CDW_NO },
			  { "\"testtest",  "\"",  CDW_NO },
			  { "testtest\"",  "\"",  CDW_NO },
			  { "test|test",   "|",   CDW_NO },
			  { "testtest|",   "|",   CDW_NO },
			  { "|testtest",   "|",   CDW_NO },
			  { "test$test",   "$",   CDW_NO },
			  { "testtest$",   "$",   CDW_NO },
			  { "$testtest",   "$",   CDW_NO },
			  /* many unusual chars, but this string is safe */
			  { "test tes t-.,_/ =:598,_/fduf98-. _/  no:", (char *) NULL, CDW_OK },
			  { (char *) NULL, (char *) NULL, CDW_CANCEL }};

	int i = 0;
	while (test_data[i].crv != CDW_CANCEL) {
		char output[2];
		cdw_rv_t crv = cdw_string_security_parser(test_data[i].input, output);
		cdw_assert(crv == test_data[i].crv, "ERROR: failed input test #%d\n", i);
		if (test_data[i].output != (char *) NULL) {
			cdw_assert (!strcmp(test_data[i].output, output), "ERROR: failed output test #%d\n", i);
		}
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_starts_with_ci(void)
{
	fprintf(stderr, "\ttesting cdw_string_starts_with_ci()... ");

	bool result;

	result = cdw_string_starts_with_ci("test string one", "test");
	assert(result == true);

	result = cdw_string_starts_with_ci("Test string one", "test");
	assert(result == true);

	result = cdw_string_starts_with_ci("Test string one", "Test");
	assert(result == true);

	result = cdw_string_starts_with_ci("test string one", "Test");
	assert(result == true);


	result = cdw_string_starts_with_ci("ttest string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci(" test string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("1test string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("t est string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("one string test", "test");
	assert(result == false);

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_rtrim(void)
{
	fprintf(stderr, "\ttesting cdw_string_rtrim()... ");

	char *result;

	const char *test_strings[] = {
		"test string \t\r\n ",
		"test string \n\t\r ",
		"test string \r\n\t ",

		"test string \t\r\n",
		"test string \n\t\r",
		"test string \r\n\t",

		"test string \t\r \n ",
		"test string \n\t \r ",
		"test string \r\n \t ",

		"test string\t \r\n ",
		"test string\n \t\r ",
		"test string\r \n\t ",

		(char *) NULL };

	char test_buffer[20];

	int i = 0;
	while (test_strings[i] != (char *) NULL) {
		strcpy(test_buffer, test_strings[i]);
		result = cdw_string_rtrim(test_buffer);
		int r = strcmp(result, "test string");
		assert(r == 0);
		size_t l = strlen(result);
		assert(l == 11);
		assert(*(result + 11) == '\0');

		i++;
	}

	result = cdw_string_rtrim(NULL);
	assert(result == NULL);

	char empty[] = "";
	result = cdw_string_rtrim(empty);
	assert(*result == '\0');

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_ltrim(void)
{
	fprintf(stderr, "\ttesting cdw_string_ltrim()... ");

	char *result;

	const char *test_strings[] = {
		" \t\r\n test string",
		" \n\t\r test string",
		" \r\n\t test string",

		" \t\r\ntest string",
		" \n\t\rtest string",
		" \r\n\ttest string",

		" \t\r \n test string",
		" \n\t \r test string",
		" \r\n \t test string",

		"\t \r\n test string",
		"\n \t\r test string",
		"\r \n\t test string",
		(char *) NULL };


	int i = 0;
	while (test_strings[i] != (char *) NULL) {
		result = cdw_string_ltrim(test_strings[i]);
		int r = strncmp(result, "test string", strlen("test string"));
		assert(r == 0);
		size_t l = strlen(result);
		assert(l == 11);
		assert(*(result + 11) == '\0');

		i++;
	}

	result = cdw_string_ltrim(NULL);
	assert(result == NULL);

	char empty_string[] = "";
	result = cdw_string_ltrim(empty_string);
	assert(*result == '\0');

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_trim(void)
{
	fprintf(stderr, "\ttesting cdw_string_trim()... ");

	const char *test_strings[] = {
		" \t\r\n test string",
		" \n\t\r test string",
		" \r\n\t test string",

		"test string \t\r\n",
		"test string \n\t\r",
		"test string \r\n\t",

		" \t\r \n test string",
		" \n\t \r test string",
		" \r\n \t test string",

		"test string \t \r\n ",
		"test string \n \t\r ",
		"test string \r \n\t ",

		"   \t\r \n     \r    \n             test string                \n \t ",
		(char *) NULL };


	int i = 0;
	while (test_strings[i]) {
		char *result = cdw_string_trim(test_strings[i]);
		cdw_assert (!strcmp(result, "test string"), "ERROR: failed at test #%d / \"%s\"\n", i, test_strings[i]);
		free(result);
		result = (char *) NULL;

		i++;
	}

	char *result = cdw_string_trim((char *) NULL);
	cdw_assert (!result, "ERROR: failed at NULL test\n");

	result = cdw_string_trim("");
	cdw_assert (*result == '\0', "ERROR: failed at \"empty string\" test\n");
	free(result);
	result = (char *) NULL;

	result = cdw_string_trim(" \t   \t   \n   \t  ");
	cdw_assert (!strcmp(result, ""), "ERROR: failed at \"empty string 3\" test\n");
	free(result);
	result = (char *) NULL;

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_is_all_spaces(void)
{
	fprintf(stderr, "\ttesting cdw_string_is_all_spaces()... ");

	struct {
		const char *string;
		bool expected_result;
	} input_data[] = {
		{ "    test string \t \n ", false },
		{ "abcdefghijklmnopqrstuv", false },
		{ "                      ", true  },
		{ "           a          ", false },
		{ "",                       true  },
		{ " \t                   ", true  },
		{ "                    \n", true  },
		{ " \r                 \t", true  },
		{ (char *) NULL,            true  }}; /* guard */

	int i = 0;
	while (input_data[i].string) {
		bool result = cdw_string_is_all_spaces(input_data[i].string);
		cdw_assert (result == input_data[i].expected_result, "ERROR: failed at test #%d\n", i);

		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_concat(void)
{
	fprintf(stderr, "\ttesting cdw_string_concat()... ");

	const char *substring1 = "four \t\r\n ";

	char *result1 = cdw_string_concat("one ", "two ", "three ", substring1, "five", (char *) NULL);
	assert(result1 != NULL);
	int r1 = strcmp(result1, "one two three four \t\r\n five");
	assert(r1 == 0);
	assert(*(result1 + 27) == '\0');

	char *result2 = cdw_string_concat("one ", "", "three ", substring1, "five", (char *) NULL);
	assert(result2 != NULL);
	int r2 = strcmp(result2, "one three four \t\r\n five");
	assert(r2 == 0);
	assert(*(result2 + 23) == '\0');

	/* this testcase may not be supported by your compiler if it does not
	support strings longer than 509 chars */
	const char *long_string1 = "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd";
	const char *long_string2 = "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee";
	const char *long_string3 = "fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";
	const char *sum_of_long_strings = "ddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeefffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";
	char *result3 = cdw_string_concat(long_string1, long_string2, long_string3, (char *) NULL);
	assert(result3 != NULL);
	int r3 = strcmp(result3, sum_of_long_strings);
	assert(r3 == 0);
	assert(*(result3 + strlen(sum_of_long_strings)) == '\0');

	free(result1);
	free(result2);
	free(result3);

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_append(void)
{
	fprintf(stderr, "\ttesting cdw_string_append()... ");

	char *head = (char *) malloc(strlen("test_string_one") + 1);
	assert(head != NULL);
	strcpy(head, "test_string_one");

	cdw_rv_t crv = cdw_string_append(&head, "I_LOVE_TESTING");
	assert(crv == CDW_OK);
	int r = strcmp(head, "test_string_oneI_LOVE_TESTING");
	assert(r == 0);

	crv = cdw_string_append(&head, "");
	assert(crv == CDW_OK);
	r = strcmp(head, "test_string_oneI_LOVE_TESTING");
	assert(r == 0);

	free(head);

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_free_table_of_strings(void)
{
	fprintf(stderr, "\ttesting cdw_string_free_table_of_strings()... ");

#define N_STRINGS_MAX 9
	const char *strings[N_STRINGS_MAX] = {"one", "two", "three", "four", "five", "six", "seven", "eight", "nine"};
	char **table = (char **) malloc((N_STRINGS_MAX + 1) * sizeof (char *)); /* +1 for guard */
	assert(table != (char **) NULL);

	table[N_STRINGS_MAX] = (char *) NULL;
	for (int i = 0; i < N_STRINGS_MAX; i++) {
		table[i] = (char *) NULL;
		table[i] = strdup(*(strings + i));
		assert(table[i] != (char *) NULL);

		cdw_sdm ("INFO: table[%d] = \"%s\"\n", i, table[i]);
	}

	cdw_string_free_table_of_strings(table, N_STRINGS_MAX);

	for (int i = 0; i < N_STRINGS_MAX + 1; i++) { /* +1 - check guard too */
		assert(table[i] == (char *) NULL);
	}
	free(table);
	table = (char **) NULL;

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_string_get_bool_value(void)
{
	fprintf(stderr, "\ttesting cdw_string_get_bool_value()... ");

	/* remember that cdw_string_get_bool_value() is called with
	   values that are already trimmed */

	/* also remember that value of second arg is not changed when
	   value in first buffer is not recognized */

	bool result = false;
	cdw_rv_t crv = cdw_string_get_bool_value((char *) NULL, &result);
	cdw_assert_test (crv == CDW_ERROR, "ERROR: failed crv test #-1\n");
	cdw_assert_test (result == false, "ERROR: failed bool test #-1\n");

	struct {
	        const char *input_string;
		bool input_bool;
		bool expected_bool;
		cdw_rv_t expected_crv;
	} test_data[] = {
		/* tests with incorrect input */
		{ "0 ",    true,  true,  CDW_ERROR },
		{ "foo",   true,  true,  CDW_ERROR },
		{ "bar",   false, false, CDW_ERROR },
		{ " true", false, false, CDW_ERROR },
		{ "",      true,  true,  CDW_ERROR },

		/* test with correct input meaning 'true' */
		{ "1",     false, true, CDW_OK },
		{ "yes",   false, true, CDW_OK },
		{ "YES",   false, true, CDW_OK },
		{ "yEs",   false, true, CDW_OK },
		{ "true",  false, true, CDW_OK },
		{ "TrUe",  false, true, CDW_OK },
		{ "TRUE",  false, true, CDW_OK },

		/* tests with correct input meaning 'false' */
		{ "0",      true, false, CDW_OK },
		{ "no",     true, false, CDW_OK },
		{ "NO",     true, false, CDW_OK },
		{ "nO",     true, false, CDW_OK },
		{ "false",  true, false, CDW_OK },
		{ "FaLse",  true, false, CDW_OK },
		{ "FALSE",  true, false, CDW_OK },

		{ (char *) NULL, true, true, CDW_OK }};

	int i = 0;
	while (test_data[i].input_string != (char *) NULL) {
		result = test_data[i].input_bool;
		crv = cdw_string_get_bool_value(test_data[i].input_string, &result);
		cdw_assert_test (crv == test_data[i].expected_crv, "ERROR: failed crv test #%d\n", i);
		cdw_assert_test (result == test_data[i].expected_bool, "ERROR: failed bool test #%d\n", i);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





static void test_cdw_string_get_line_end(void)
{
	fprintf(stderr, "\ttesting cdw_string_get_line_end()... ");
	struct {
		const char *string;
		size_t width_max;
		size_t expected_value;
	} test_data[] = {
		{ "",             1,  0 },
		{ "",            10,  0 },
		{ "a",            1,  0 },
		{ "a",           10,  0 },
		{ "ab",           1,  0 },
		{ "ab",          10,  1 },
		{ "hello world",  1,  0 },
		{ "hello world",  5,  4 },
		{ "hello world",  6,  4 },
		{ "hello world",  7,  4 },
		{ "hello world", 11, 10 },
		{ "hello world", 12, 10 },

		{ "hello\nworld",  1,  0 },
		{ "hello\nworld",  5,  4 },
		{ "hello\nworld",  6,  4 },
		{ "hello\nworld",  7,  4 },
		{ "hello\nworld", 11,  4 },
		{ "hello\nworld, this is me ", 27, 4 },


		{ "12345678901234567890", 1, 0 },
		{ "12345678901234567890", 10, 9 },
		{ "12345678901234567890", 20, 19 },
		{ "12345678901234567890", 21, 19 },
		{ "12345678901234567890", 22, 19 },
		{ "12345678901234567890", 30, 19 },
		{ "12345678901234567890 12345678901234567890", 30, 19 },
		{ "12345678901234567890 12345678901234567890", 41, 40 },

		{ (char *) NULL, 0, 0 },

	};

	int i = 0;
	while (test_data[i].width_max != 0) {
		size_t result = cdw_string_get_line_end(test_data[i].string, test_data[i].width_max);
		cdw_assert (result == test_data[i].expected_value,
			    "ERROR: test #%d: string = \"%s\", expected value = %zd, result = %zd\n",
			    i, test_data[i].string, test_data[i].expected_value, result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





static void test_cdw_string_get_line_len(void)
{
	fprintf(stderr, "\ttesting cdw_string_get_line_len()... ");
	struct {
		const char *string;
		size_t width_max;
		size_t expected_value;
	} test_data[] = {
		{ "",             1,  0 },
		{ "",            10,  0 },
		{ "a",            1,  1 },
		{ "a",           10,  1 },
		{ "ab",           1,  1 },

		{ "ab",          10,  2 },
		{ "ab",           2,  2 },
		{ "hello world",  1,  1 },
		{ "hello world",  5,  5 },
		{ "hello world",  6,  5 },

		{ "hello world",  7,  5 },
		{ "hello world", 11, 11 },
		{ "hello world", 12, 11 },

		{ "hello\nworld",  1,  1 },
		{ "hello\nworld",  5,  5 },
		{ "hello\nworld",  6,  5 },
		{ "hello\nworld",  7,  5 },
		{ "hello\nworld", 11,  5 },
		{ "hello\nworld, this is me ", 27, 5 },


		{ "12345678901234567890", 1,  1 },
		{ "12345678901234567890", 10, 10 },
		{ "12345678901234567890", 20, 20 },
		{ "12345678901234567890", 21, 20 },
		{ "12345678901234567890", 22, 20 },
		{ "12345678901234567890", 30, 20 },
		{ "12345678901234567890 12345678901234567890", 30, 20 },
		{ "12345678901234567890 12345678901234567890", 41, 41 },

		{ (char *) NULL, 0, 0 },

	};

	int i = 0;
	while (test_data[i].width_max != 0) {
		size_t result = cdw_string_get_line_len(test_data[i].string, test_data[i].width_max);
		cdw_assert (result == test_data[i].expected_value,
			    "ERROR: test #%d: string = \"%s\", expected value = %zd, result = %zd\n",
			    i, test_data[i].string, test_data[i].expected_value, result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_string_wrap(void)
{
	fprintf(stderr, "\ttesting cdw_string_wrap()... ");

	/* input data */
	struct test_data {
		const char *string;     /* string to be wrapped */
		size_t max_width;          /* maximal width of logical line, a given data */
		const char *expected_result;  /* expected return value */
	} test_data[] = {
		{
			"this is a test string that is supposed to test cdw string wrap test function",
			40,
			/* 1234567890123456789012345678901234567890 */
			  "this is a test string that is supposed  "
			  "to test cdw string wrap test function   "
		},
#if 0
		/* FIXME: this testcase would fail */
		{
			"this is one test string that is supposed to test cdw string wrap test function",
			40,
			/* 1234567890123456789012345678901234567890 */
			  "this is one test string that is supposed"
			  "to test cdw string wrap test function   "
		},
#endif

		{
			"Another test string\n"
			"but this time with some line breaks in the middle of\n"
			"the string",
			40,
			/* 1234567890123456789012345678901234567890 */
			  "Another test string                     "
			  "but this time with some line breaks in  "
			  "the middle of                           "
			  "the string                              ",
		},

		{

			"This is yet another string that serves\n"
			"as input data for unit tests\n"
			"of cdw string wrap function from cdw string module I feel that it's kind of hard to come up with text for this long string, you know,\n"
			"so...\n"
			"perhaps\n"
			"I will stop after this test line. Yeah, that's a good idea",
			40,
			/* 1234567890123456789012345678901234567890 */
			  "This is yet another string that serves  "
			  "as input data for unit tests            "
			  "of cdw string wrap function from cdw    "
			  "string module I feel that it's kind of  "
			  "hard to come up with text for this long "
			  "string, you know,                       "
			  "so...                                   "
			  "perhaps                                 "
			  "I will stop after this test line. Yeah, "
			  "that's a good idea                      "

		},

		{
			/* guard */
			(char *) NULL,
			0,
			(char *) NULL
		}
	};

	int i = 0;
	while (test_data[i].string) {
		char *output = cdw_string_wrap(test_data[i].string, test_data[i].max_width, CDW_ALIGN_LEFT);
		int rv = strcmp(output, test_data[i].expected_result);
		if (rv) {
			fprintf(stderr, "ERROR: failed at testcase #%d\n", i);
			for (unsigned int k = 0; k < test_data[i].max_width; k++) {
				fprintf(stderr, "o");
			}
			fprintf(stderr, "\n");


			bool run = true;
			for (size_t j = 0; run; j++) {
				for (size_t k = 0; k < test_data[i].max_width; k++) {
					if (output[(j * test_data[i].max_width) + k] == '\0') {
						run = false;
						break;
					}
					fprintf(stderr, "%c", output[(j * test_data[i].max_width) + k]);
				}
				fprintf(stderr, "\"\n");


				for (size_t k = 0; k < test_data[i].max_width; k++) {
					if (test_data[i].expected_result[(j * test_data[i].max_width) + k] == '\0') {
						run = false;
						break;
					}
					fprintf(stderr, "%c", test_data[i].expected_result[(j * test_data[i].max_width) + k]);
				}
				fprintf(stderr, "\"\n");

			}

			free(output);
			output = (char *) NULL;

			assert(0);
		}
		free(output);
		output = (char *) NULL;


		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_equal(void)
{
	fprintf(stderr, "\ttesting cdw_string_equal()... ");

	struct {
		const char *string1;
		const char *string2;
		bool expected_result;
	} test_data[] = {
		{ "", "", true },
		{ "", "a", false },
		{ "a", "a", true },
		{ "test string", "test", false },
		{ "test string", "test string", true },
		{ (char *) NULL, (char *) NULL, false } }; /* guard */

	int i = 0;
	while (test_data[i].string1) {
		bool rv = cdw_string_equal((const void *) test_data[i].string1,
					   (const void *) test_data[i].string2);
		cdw_assert (rv == test_data[i].expected_result,
			    "ERROR: failed in test #%d ('%s' - '%s')\n",
			    i, test_data[i].string1, test_data[i].string2);
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_count_lines_new(void)
{

	fprintf(stderr, "\ttesting cdw_string_count_lines_new()... ");

	struct {
		const char *string;
		size_t width_max;
		size_t expected_value;
	} input_data[] = {
		{ "",            10, 0 },

		{ "hello",       10, 1 },
		{ "1234567890",   10, 1 },
		{ "12345678901",  10, 2 },

		/*
		  "animated"
		  " show with"
		  " lots of"
		  " images" */
		{ "animated show with lots of images", 10, 4 },

		/* "animated"
		   " show"
		   "with lots"
		   "of images" */
		{ "animated show\nwith lots of images", 10, 4 },

		/* "animated"
		   " show"
		   "with lots"
		   "of images" */
		{ "animated show\n\nwith lots of images", 10, 4 },

		/* "naturalist"
		   "ic show"
		   " with lots"
		   " of images" */
		{ "naturalistic show with lots of images", 10, 4 },

		/* "naturalist"
		   "ic show"
		   " with"
		   " loads of"
		   " images" */
		{ "naturalistic show with loads of images", 10, 5 },

		/* "naturalist"
		   "ic"
		   "show with"
		   " loads of"
		   " images" */
		{ "naturalistic\nshow with loads of images", 10, 5 },

		{ (char *) NULL, 10, 0 } }; /* guard */

	int i = 0;
	while (input_data[i].string) {
		size_t rv = cdw_string_count_lines_new(input_data[i].string, input_data[i].width_max);
		cdw_assert (rv == input_data[i].expected_value, "ERROR: failed at test #%d (rv = %zd, expected %zd)\n", i, rv, input_data[i].expected_value);

		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}

#endif /* #ifdef CDW_UNIT_TEST_CODE */
