#include "AviReadStream.h"
#include "AviReadHandler.h"
#include "formats.h"
#include "avm_output.h"
#include "avm_cpuinfo.h"
#include "utils.h"

#include <string.h>
#include <stdio.h>

AVM_BEGIN_NAMESPACE;

AviReadStream::AviReadStream(AviReadHandler* handler,
			     const AVIStreamHeader& hdr,
			     uint_t id, const void* format, size_t fsize)
    :m_pHandler(handler), m_iId(id), m_uiChunk(0), m_uiPosition(0),
    m_Header(hdr), m_uiFormatSize(fsize),
    m_uiStart(0), m_uiStreamSize(0), m_uiAlignedSize(0),
    m_uiKeyChunks(0), m_uiKeySize(0),  m_uiKeyMaxSize(0), m_uiKeyMinSize(~0U),
    m_uiDeltaSize(0), m_uiDeltaMaxSize(0), m_uiDeltaMinSize(~0U)
{
    if ((m_pcFormat = new char[m_uiFormatSize]))
	memcpy(m_pcFormat, format, m_uiFormatSize);

    if (GetType() == IStream::Audio && m_Header.dwSampleSize)
    {
	m_dFrameRate = m_WAVEFORMATEX->nAvgBytesPerSec / m_Header.dwSampleSize;
	m_Positions.reserve(16384);
	m_Positions.push_back(0);
    }
    else
    {
	m_dFrameRate = (m_Header.dwScale) ? m_Header.dwRate / (double) m_Header.dwScale : 1.;
	if (GetType() == IStream::Audio) {
	    // VBR audio has dwSampleSize == 0
	    // use tranlation table for time - see note in addChunk().
	    m_Positions.reserve(16384);
	    m_Positions.push_back(0);
	    if (m_WAVEFORMATEX->nBlockAlign < 32)
		AVM_WRITE("AVI reader", "WARNING: WaveFormat::BlockAlign=%d is too small for VBR audio stream!\n", m_WAVEFORMATEX->nBlockAlign);
	}
    }
    m_Offsets.reserve(16384);
}

AviReadStream::~AviReadStream()
{
    delete[] m_pcFormat;
}

double AviReadStream::CacheSize() const
{
    return m_pHandler->m_Input.cacheSize();
}

void AviReadStream::ClearCache()
{
    m_pHandler->m_Input.clear();
}

size_t AviReadStream::GetHeader(void* header, size_t size) const
{

    if (header && size >= sizeof(m_Header))
    {
	memset(header, 0, size);
	memcpy(header, &m_Header, sizeof(m_Header));
    }
    return sizeof(m_Header);
}

size_t AviReadStream::GetFormat(void *format, size_t size) const
{
    if (format)
	memcpy(format, m_pcFormat, (size < m_uiFormatSize) ? size : m_uiFormatSize);

    return m_uiFormatSize;
}

framepos_t AviReadStream::GetLength() const
{
    return m_Header.dwStart + m_Header.dwLength;
}

double AviReadStream::GetLengthTime() const
{
    return GetTime(GetLength());
}

framepos_t AviReadStream::GetNearestKeyFrame(framepos_t pos) const
{
    pos = getFramePos(pos);

    framepos_t rpos = m_Header.dwStart + pos;

    if (m_Header.dwSampleSize
	|| (pos < m_Offsets.size() && (m_Offsets[pos] & 1)))
	return rpos; // already keyframe

    framepos_t prev = GetPrevKeyFrame(rpos);
    framepos_t next = GetNextKeyFrame(rpos);

    return ((rpos - prev) < (next - rpos)) ? prev : next;
}

framepos_t AviReadStream::GetNextKeyFrame(framepos_t pos) const
{
    //printf("GETNEXT   %d   %d   %d\n", m_iId, pos, m_uiPosition);
    pos = getFramePos(pos);

    if (m_Header.dwSampleSize == 0)
    {
	do
	{
	    if (pos >= m_Offsets.size())
		return ERR;

	} while (!(m_Offsets[pos] & 1) && ++pos);
    }

    //printf("GETNEXT   return %d\n", pos + m_Header.dwStart);
    return m_Header.dwStart + pos;
}

framepos_t AviReadStream::GetPrevKeyFrame(framepos_t pos) const
{
    //printf("GETPREV   %d   %d   %d\n", m_iId, pos, m_uiPosition);
    pos = getFramePos(pos);

    if (pos > 0)
	pos--; // go at least one frame back

    if (m_Header.dwSampleSize == 0) {
	if (pos >= m_Offsets.size())
	    pos = (framepos_t)m_Offsets.size() - 1;

	while (pos > 0 && !(m_Offsets[--pos] & 1))
	    ;
    }

    //printf("GETPREV   return %d\n", pos + m_Header.dwStart);
    return m_Header.dwStart + pos;
}

StreamInfo* AviReadStream::GetStreamInfo() const
{
    if (m_StreamInfo.m_p->m_dLengthTime == 0.)
    {
	m_StreamInfo.m_p->setKfFrames(m_uiKeyMaxSize,
				      (m_uiKeyMinSize > m_uiKeyMaxSize)
				      ? m_uiKeyMaxSize : m_uiKeyMinSize,
				      m_uiKeyChunks, m_uiKeySize);

	m_StreamInfo.m_p->setFrames(m_uiDeltaMaxSize,
				    (m_uiDeltaMinSize > m_uiDeltaMaxSize)
                                    // usually no delta frames
				    ? m_uiDeltaMaxSize : m_uiDeltaMinSize,
				    (uint_t)(m_Offsets.size() - m_uiKeyChunks),
				    m_uiStreamSize - m_uiKeySize);

	m_StreamInfo.m_p->m_dLengthTime = GetLengthTime();
	m_StreamInfo.m_p->m_iQuality = m_Header.dwQuality;
	m_StreamInfo.m_p->m_iSampleSize = m_Header.dwSampleSize;

	if (m_pcFormat)
	    switch (GetType())
	    {
	    case IStream::Video:
		m_StreamInfo.m_p->m_Type = StreamInfo::Video;
		m_StreamInfo.m_p->m_uiFormat = m_BITMAPINFOHEADER->biCompression;
		m_StreamInfo.m_p->setVideo(m_BITMAPINFOHEADER->biWidth,
					   m_BITMAPINFOHEADER->biHeight);
		break;
	    case IStream::Audio:
		m_StreamInfo.m_p->m_Type = StreamInfo::Audio;
		m_StreamInfo.m_p->m_uiFormat = m_WAVEFORMATEX->wFormatTag;
		m_StreamInfo.m_p->setAudio(m_WAVEFORMATEX->nChannels,
					   m_WAVEFORMATEX->nSamplesPerSec,
					   m_WAVEFORMATEX->wBitsPerSample);
		break;
	    default:
                ; // FIXME
	    }
    }

    return new StreamInfo(m_StreamInfo);
}

double AviReadStream::GetTime(framepos_t pos) const
{
    pos = getFramePos(pos);

    if (m_Header.dwSampleSize || !m_Positions.size()) {
	pos += m_Header.dwStart;
	framepos_t l = GetLength();
	if (pos > l)
	    pos = l;
    } else
	pos = m_Header.dwStart + ((pos < m_Positions.size()) ? m_Positions[pos] : m_Positions.back());

    //printf("AviGetTime: %d    %f   (%f,  %d,   %.4s   s: %d/%" PRIsz ")  ss:%d\n",
    //       pos, pos / m_dFrameRate, m_dFrameRate, m_Header.dwStart,
    //       (const char*)&m_Header.fccType, m_uiPosition, m_Positions.size(), m_Header.dwSampleSize);
    return pos / m_dFrameRate;
}

IStream::StreamType AviReadStream::GetType() const
{
    switch (m_Header.fccType)
    {
    case streamtypeAUDIO: return IStream::Audio;
    case streamtypeIAVS:
    case streamtypeVIDEO: return IStream::Video;
    default: return IStream::Other;
    }
}

bool AviReadStream::IsKeyFrame(framepos_t pos) const
{
    if (m_Header.dwSampleSize || pos < m_Header.dwStart)
	return true; //audio

    pos = getFramePos(pos);
    return (pos < m_Offsets.size()) ? m_Offsets[pos] & 1 : true;
}

StreamPacket* AviReadStream::ReadPacket()
{
    StreamPacket* p = m_pHandler->m_Input.readPacket(m_iId, m_uiChunk);
    //printf("ReadPacket: %p  id:%d  chunk:%d  pos:%d %f  ss:%d\n", p, m_iId, m_uiChunk, m_uiPosition, GetTime(), m_Header.dwSampleSize);
    if (p)
    {
	p->SetPos(m_uiPosition);
	p->SetTime(GetTime(m_uiPosition));

	//if (m_Header.dwSampleSize) printf("readpacketpos  %d  %d   id:%d  %lld   %p   pos:%lld\n", m_uiPosition, p->size, m_iId, p->timestamp, p, m_pHandler->m_Input.pos());
        m_uiChunk++;
	m_uiPosition = m_Header.dwStart;
	if (m_Header.dwSampleSize)
	    m_uiPosition += (m_uiChunk < m_Positions.size())
		? m_Positions[m_uiChunk] : m_Positions.back();
	else
	    m_uiPosition += m_uiChunk;

	//printf("SETNEW RP %d/%d ck:%d/%d  ss:%d   ts: %" PRId64 "\n", m_uiPosition, p->position, m_uiChunk, (int)m_Positions.size(), m_Header.dwSampleSize, p->timestamp);
    }
    return p;
}

int AviReadStream::Seek(framepos_t framepos)
{
    if ((m_uiChunk = find(framepos)) == ERR)
	m_uiPosition = m_uiChunk = 0;
    else
	m_uiPosition = m_Header.dwStart + ((m_Header.dwSampleSize) ? m_Positions[m_uiChunk] : m_uiChunk);

    AVM_WRITE("AVI reader", 3, "AviReadStream::Seek(%u) -> %d  (%d)\n", framepos, m_uiChunk, m_iId);
    return 0;
}

int AviReadStream::SeekTime(double timepos)
{
    AVM_WRITE("AVI reader", 3, "AviReadStream::SeekTime(%f)\n", timepos);
    return Seek((framepos_t)(timepos * m_dFrameRate));
}

// takes normal pos
// returns chunk or pos without dwStart!
framepos_t AviReadStream::find(framepos_t pos) const
{
    if (pos < m_Header.dwStart)
	return ERR;

    framepos_t len = GetLength();
    //printf("FIND %d  %d   %d\n", pos, len, (int)m_Positions.size());
    if (pos > len)
	pos = len;

    pos -= m_Header.dwStart;
    // when we don't have translation table -> position is found
    if (!m_Positions.size())
        return pos;

    framepos_t low_limit = 0;
    framepos_t high_limit = (framepos_t)m_Positions.size() - 1;
    while (low_limit < high_limit)
    {
        // http://googleresearch.blogspot.com/2006/06/extra-extra-read-all-about-it-nearly.html
	framepos_t middle = low_limit + (high_limit - low_limit) / 2;
	if (pos >= m_Positions[middle])
	{
            // this fix is necessary to avoid deadlock
	    if (middle == low_limit)
                break;

	    low_limit = middle;
	    if (pos < m_Positions[middle + 1])
		break;
	}
	else
	    high_limit = middle;
    }

    //printf("FOUNDPOS  %d  %d   \n", low_limit, m_Positions[low_limit]);
    return low_limit;
}

void AviReadStream::addChunk(off_t coffset, size_t clen, bool iskf)
{
    if (iskf)
    {
	m_uiKeyChunks++;
	m_uiKeySize += clen;
	if (clen > m_uiKeyMaxSize)
	    m_uiKeyMaxSize = clen;
	if (clen < m_uiKeyMinSize)
	    m_uiKeyMinSize = clen;
	// AVI chunks begins on 'even' stream position
	// so use this the lowest bit as the keyframe flag
        // saving space
	coffset |= 1;
    }
    else
    {
	m_uiDeltaSize += clen;
	if (clen > m_uiDeltaMaxSize)
	    m_uiDeltaMaxSize = clen;
	if (clen < m_uiDeltaMinSize)
	    m_uiDeltaMinSize = clen;
    }

    m_uiStreamSize += clen;
    m_Offsets.push_back((uint32_t)coffset);

    if (m_Header.dwSampleSize)
    {
	// position in bytes for chunk stream is not need, it is always 1
	// - so use them only for wave audio frame position
	m_Positions.push_back((framepos_t)(m_uiStreamSize / m_Header.dwSampleSize));
    }
    else if (GetType() == IStream::Audio && m_Header.dwScale)
    {
	// VBR audio magic - we have to recalculate chunk position
	// if the chunk is bigger then nBlockAlign - it represents
	// more then one chunk in that case - rounding applies here!
        // http://www.virtualdub.org/blog/pivot/entry.php?id=27

	// mencoder produces nonstandard VBR files with nBlockAlign == 1
	// probably take it as one block per VBR audio chunk
        framepos_t balign = m_WAVEFORMATEX->nBlockAlign;
	m_uiAlignedSize += (balign < 32) ? 1 : (framepos_t)(clen + balign - 1) / balign;
	m_Positions.push_back(m_uiAlignedSize);
    }

#if 0
    if (m_iId > 0) printf("Id:%d/%5d  offset:%10d  sp:%10d  len:%6d  p:%5d (%d) -> %.3f\n",
			  m_iId, (int)m_Offsets.size(), (int)coffset & ~1U, (int)m_uiStreamSize,
			  (int)clen, (int)m_Positions.back(), (int)m_Positions.size(),
			  m_Positions.back() * m_Header.dwScale / (double)m_Header.dwRate);
    else printf("Id:%d/%5d  offset:%10d  sp:%10d  len:%6d  p:%5d -> %.3f\n", m_iId, (int)m_Offsets.size(),
		(int)coffset & ~1U, (int)m_uiStreamSize, (int)clen, (int)m_Offsets.size(),
		(double)m_Offsets.size() * m_Header.dwScale /  (double)m_Header.dwRate);
#endif
}

void AviReadStream::fixHeader()
{
    uint32_t n = (m_Header.dwSampleSize) ? (uint32_t)(m_uiStreamSize / m_Header.dwSampleSize) : (uint32_t)m_Offsets.size();
    if (n != m_Header.dwLength)
    {
	AVM_WRITE("AVI reader", "WARNING: stream header has incorrect dwLength (%d -> %d)\n", m_Header.dwLength, n);
	m_Header.dwLength = n;
    }
}

int AviReadStream::FixAvgBytes(uint_t bps)
{
    if (bps > 8000)
    {
	m_dFrameRate = bps;
	return 0;
    }
    // ok some broken files with vbr stream and nonvbr headers
    // are weird and we can't easily recognize them - they usually
    // have 32kbps blocks at the beginning
    // also this is hack to make playable several broken
    // sample files I have, thus it might fail for others...
    return -1;
}

AVM_END_NAMESPACE;
