/**
 * @brief Low level backlight driver for SysFD controlled backlights.
 *
 * This module contains the low level driver to control a LCD backlight
 * via the SysFS kernel interface. The low level driver overloads
 * methods of class backlight.
 
 * This driver uses the new kernel backlight interface introduced in
 * kernel version 2.6.18. It won't work on earlier kernels.
 *
 * It is also recommended to compile the kernel *without* the kernel
 * option CONFIG_PMAC_BACKLIGHT_LEGACY to prevent conflicts with ancient
 * kernel features controlling the backlight.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/driver_backlight_sysfs.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "class_backlight.h"
#include "driver_backlight_sysfs.h"
#include "debug.h"

/**
 * @brief Path to the backlight device in SysFS
 *
 * This string contains the directory name in SysFS where the 
 * backlight device is located. The directory name derives
 * from the graphics card name and it is expensive to retrieve
 * it every time again so it is saved here.
 */
static GString *sysfs_backlight_path;

/**
 * @brief Exit function of the driver - cleans up all ressources
 *
 * This function will be called when the work is done and the driver
 * should be unloaded. It frees all allocated ressources.
 */
void
driver_backlight_sysfs_exit ()
{
	g_string_free (sysfs_backlight_path, TRUE);
}

/**
 * @brief Get the current brightness level from the device
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the current brightness level of the LCD backlight.
 *
 * @return  current brightness level or -1 on error
 */
int
sysfsbl_get_brightness ()
{
	GString *var;
	FILE *fd;
	int brightness = -1;
	char buffer[5];

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/brightness");

	if ((fd = fopen(var->str, "r"))) {
		fgets(buffer, sizeof(buffer), fd);
		buffer[4] = 0;  /* to be safe, terminate the buffer */
		brightness = atoi (buffer);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
	return brightness;
}

/**
 * @brief Get the maximum brightness level the device supports
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the maximum brightness level the LCD backlight
 * device supports.
 *
 * @return  maximum supported brightness level of the device.
 */
int
sysfsbl_get_brightness_max ()
{
	GString *var;
	FILE *fd;
	int max_brightness = -1;
	char buffer[5];

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/max_brightness");

	if ((fd = fopen(var->str, "r"))) {
		fgets(buffer, sizeof(buffer), fd);
		buffer[4] = 0;  /* to be safe, terminate the buffer */
		max_brightness = atoi (buffer);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
	return max_brightness;
}

/**
 * @brief Change the current brightness level.
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * This function sets a new brightness level. The given level must be
 * valid for this device, that means it must be lower than the level
 * sysfsbl_get_brightness_max() returns. No further check is done on
 * that.
 *
 * @param  val   new brightness level
 */
void
sysfsbl_set_brightness (int val)
{
	GString *var;
	FILE *fd;
	char buffer[5];

	if (val > 255) return;

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/brightness");
	sprintf(buffer, "%d\n", val);

	if ((fd = fopen(var->str, "w"))) {
		fputs(buffer, fd);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
}

static struct driver_backlight driver_backlight_sysfs = {
	.name               = N_("SysFS Backlight Driver"),
	.get_brightness     = sysfsbl_get_brightness,
	.get_brightness_max = sysfsbl_get_brightness_max,
	.set_brightness     = sysfsbl_set_brightness,
	.driver_exit        = driver_backlight_sysfs_exit
};

/**
 * @brief Constructor of a SysFS backlight driver object
 *
 * This function probes for a backlight controller in the sysfs.
 * If sysfs is not mounted or a backlight controller could not
 * be found NULL is returned. Otherwise a driver object is
 * initialized and returned.
 *
 * @return  Initializes backlight driver object or NULL
 */
struct driver_backlight *
driver_backlight_sysfs_init ()
{
	DIR *dh;
	struct dirent *dir;

	sysfs_backlight_path = g_string_new ("/sys/class/backlight");

	if ((dh = opendir(sysfs_backlight_path->str))) {
		while ((dir = readdir(dh))) {
			if (dir->d_name[0] == '.') continue;
			if (g_str_has_suffix(dir->d_name, "bl0") ||
				g_str_has_suffix(dir->d_name, "bl")) {
				g_string_append_printf (sysfs_backlight_path, "/%s", dir->d_name);
#if defined(DEBUG)
				print_msg (PBB_INFO, "DBG: SysFS Backlight Path: %s\n", sysfs_backlight_path->str);
#endif
				return &driver_backlight_sysfs;
			}
		}
	}

	return NULL;
}

