static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2014 Dimitris Papavasiliou <dpapavas@google.com>                 \n"
" */                                                                           \n"
"                                                                              \n"
"/* This plug-in generates cellular noise textures based on the                \n"
" * function described in the paper                                            \n"
" *                                                                            \n"
" *    Steven Worley. 1996. A cellular texture basis function.                 \n"
" *    In Proceedings of the 23rd annual conference on Computer                \n"
" *    graphics and interactive techniques (SIGGRAPH '96).                     \n"
" *                                                                            \n"
" * Comments and improvements for this code are welcome.                       \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#define MAX_RANK 3                                                            \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double  (scale, _(\"Scale\"), 1.0)                                   \n"
"    description  (_(\"The scale of the noise function\"))                     \n"
"    value_range  (0, 20.0)                                                    \n"
"                                                                              \n"
"property_double  (shape, _(\"Shape\"), 2.0)                                   \n"
"    description  (_(\"Interpolate between Manhattan and Euclidean distance.\"))\n"
"    value_range  (1.0, 2.0)                                                   \n"
"                                                                              \n"
"property_int     (rank, _(\"Rank\"), 1)                                       \n"
"    description  (_(\"Select the n-th closest point\"))                       \n"
"    value_range  (1, MAX_RANK)                                                \n"
"                                                                              \n"
"property_int     (iterations, _(\"Iterations\"), 1)                           \n"
"    description  (_(\"The number of noise octaves.\"))                        \n"
"    value_range  (1, 20)                                                      \n"
"                                                                              \n"
"property_boolean (palettize, _(\"Palettize\"), FALSE)                         \n"
"    description  (_(\"Fill each cell with a random color\"))                  \n"
"                                                                              \n"
"property_seed    (seed, _(\"Random seed\"), rand)                             \n"
"    description  (_(\"The random seed for the noise function\"))              \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_POINT_RENDER                                                  \n"
"#define GEGL_OP_C_SOURCE noise-cell.c                                         \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <gegl-buffer-cl-iterator.h>                                          \n"
"#include <gegl-debug.h>                                                       \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#include \"opencl/noise-cell.cl.h\"                                           \n"
"                                                                              \n"
"/* Random feature counts following the Poisson distribution with              \n"
"   lambda equal to 7. */                                                      \n"
"                                                                              \n"
"static const gint poisson[256] = {                                            \n"
"  7, 9, 12, 12, 8, 7, 5, 5, 6, 7, 8, 6, 10, 7, 6, 2, 8, 3, 9, 5, 13, 10, 9,   \n"
"  8, 8, 9, 3, 8, 9, 6, 8, 7, 4, 9, 6, 3, 10, 7, 7, 7, 6, 7, 4, 14, 7, 6, 11,  \n"
"  7, 7, 7, 12, 7, 10, 6, 8, 11, 3, 5, 7, 7, 8, 7, 9, 8, 5, 8, 11, 3, 4, 5, 8, \n"
"  8, 7, 8, 9, 2, 7, 8, 12, 4, 8, 2, 11, 8, 14, 7, 8, 2, 3, 10, 4, 6, 9, 5, 8, \n"
"  7, 10, 10, 10, 14, 5, 7, 6, 4, 5, 6, 11, 8, 7, 3, 11, 5, 5, 2, 9, 7, 7, 7,  \n"
"  9, 2, 7, 6, 9, 7, 6, 5, 12, 5, 3, 11, 9, 12, 8, 6, 8, 6, 8, 5, 5, 7, 5, 2,  \n"
"  9, 5, 5, 8, 11, 8, 8, 10, 6, 4, 7, 14, 7, 3, 10, 7, 7, 4, 9, 10, 10, 9, 8,  \n"
"  8, 7, 6, 5, 10, 10, 5, 10, 7, 7, 10, 7, 4, 9, 9, 6, 8, 5, 10, 7, 3, 9, 9,   \n"
"  7, 8, 9, 7, 5, 7, 6, 5, 5, 12, 4, 7, 5, 5, 4, 5, 7, 10, 8, 7, 9, 4, 6, 11,  \n"
"  6, 3, 7, 8, 9, 5, 8, 6, 7, 8, 7, 7, 3, 7, 7, 9, 4, 5, 5, 6, 9, 7, 6, 12, 4, \n"
"  9, 10, 8, 8, 6, 4, 9, 9, 8, 11, 6, 8, 13, 8, 9, 12, 6, 9, 8                 \n"
"};                                                                            \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble  shape;                                                             \n"
"  gdouble  closest[MAX_RANK];                                                 \n"
"  guint    feature, rank, seed;                                               \n"
"  gboolean palettize;                                                         \n"
"} Context;                                                                    \n"
"                                                                              \n"
"static GeglClRunData *cl_data = NULL;                                         \n"
"                                                                              \n"
"static inline guint                                                           \n"
"philox (guint s,                                                              \n"
"        guint t,                                                              \n"
"        guint k)                                                              \n"
"{                                                                             \n"
"  guint64 p;                                                                  \n"
"  gint    i;                                                                  \n"
"                                                                              \n"
"  /* For details regarding this hash function consult:                        \n"
"                                                                              \n"
"     Salmon, J.K.; Moraes, M.A.; Dror, R.O.; Shaw, D.E., \"Parallel           \n"
"     random numbers: As easy as 1, 2, 3,\" High Performance Computing,        \n"
"     Networking, Storage and Analysis (SC), 2011 International                \n"
"     Conference for , vol., no., pp.1,12, 12-18 Nov. 2011 */\\                \n"
"                                                                              \n"
"  for (i = 0 ; i < 3 ; i += 1)                                                \n"
"    {                                                                         \n"
"      p = s * (guint64)0xcd9e8d57;                                            \n"
"                                                                              \n"
"      s = ((guint)(p >> 32)) ^ t ^ k;                                         \n"
"      t = (guint)p;                                                           \n"
"                                                                              \n"
"      k += 0x9e3779b9;                                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  return s;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static inline gdouble                                                         \n"
"lcg (guint *hash)                                                             \n"
"{                                                                             \n"
"  return (*hash = *hash * (guint)1664525 + (guint)1013904223) / (gdouble)4294967296.0;\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"search_box (gint     s,                                                       \n"
"            gint     t,                                                       \n"
"            gdouble  x,                                                       \n"
"            gdouble  y,                                                       \n"
"            Context *context)                                                 \n"
"{                                                                             \n"
"  guint hash;                                                                 \n"
"  gint i, n;                                                                  \n"
"                                                                              \n"
"  hash = philox ((guint)s, (guint)t, context->seed);                          \n"
"  n = poisson[hash >> 24];                                                    \n"
"                                                                              \n"
"  for (i = 0 ; i < n ; i += 1)                                                \n"
"    {                                                                         \n"
"      gdouble delta_x, delta_y, d;                                            \n"
"      gint    j, k;                                                           \n"
"                                                                              \n"
"      /* Calculate the distance to each feature point. */                     \n"
"                                                                              \n"
"      delta_x = s + lcg (&hash) - x;                                          \n"
"      delta_y = t + lcg (&hash) - y;                                          \n"
"                                                                              \n"
"      if (context->shape == 2)                                                \n"
"        d = delta_x * delta_x + delta_y * delta_y;                            \n"
"      else if (context->shape == 1)                                           \n"
"        d = fabs (delta_x) + fabs (delta_y);                                  \n"
"      else                                                                    \n"
"        d = pow (fabs (delta_x), context->shape) +                            \n"
"          pow (fabs (delta_y), context->shape);                               \n"
"                                                                              \n"
"      /* Insert it into the list of n closest distances if needed. */         \n"
"                                                                              \n"
"      for (j = 0 ; j < context->rank && d > context->closest[j] ; j += 1);    \n"
"                                                                              \n"
"      if (j < context->rank)                                                  \n"
"        {                                                                     \n"
"          for (k = context->rank - 1 ; k > j ; k -= 1)                        \n"
"            {                                                                 \n"
"              context->closest[k] = context->closest[k - 1];                  \n"
"            }                                                                 \n"
"                                                                              \n"
"          context->closest[j] = d;                                            \n"
"                                                                              \n"
"          if (j == context->rank - 1)                                         \n"
"            context->feature = hash;                                          \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static double                                                                 \n"
"noise2 (double x,                                                             \n"
"        double y,                                                             \n"
"        Context *context)                                                     \n"
"{                                                                             \n"
"  gdouble d_l, d_r, d_t, d_b, *d_0;                                           \n"
"  gint s, t, i;                                                               \n"
"                                                                              \n"
"  for (i = 0 ; i < context->rank ; context->closest[i] = INFINITY, i += 1);   \n"
"                                                                              \n"
"  s = (gint)floor(x);                                                         \n"
"  t = (gint)floor(y);                                                         \n"
"                                                                              \n"
"  /* Search the box the point is in. */                                       \n"
"                                                                              \n"
"  search_box (s, t, x, y, context);                                           \n"
"                                                                              \n"
"  d_0 = &context->closest[context->rank - 1];                                 \n"
"  d_l = x - s; d_l *= d_l;                                                    \n"
"  d_r = 1.0 - x + s; d_r *= d_r;                                              \n"
"  d_b = y - t; d_b *= d_b;                                                    \n"
"  d_t = 1.0 - y + t; d_t *= d_t;                                              \n"
"                                                                              \n"
"  /* Search adjacent boxes if it is possible for them to contain a            \n"
"   * nearby feature point. */                                                 \n"
"                                                                              \n"
"  if (d_l < *d_0)                                                             \n"
"    {                                                                         \n"
"      if (d_l + d_b < *d_0)                                                   \n"
"        search_box (s - 1, t - 1, x, y, context);                             \n"
"                                                                              \n"
"      search_box (s - 1, t, x, y, context);                                   \n"
"                                                                              \n"
"      if (d_l + d_t < *d_0)                                                   \n"
"        search_box (s - 1, t + 1, x, y, context);                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (d_b < *d_0)                                                             \n"
"    search_box (s, t - 1, x, y, context);                                     \n"
"                                                                              \n"
"  if (d_t < *d_0)                                                             \n"
"    search_box (s, t + 1, x, y, context);                                     \n"
"                                                                              \n"
"  if (d_r < *d_0)                                                             \n"
"    {                                                                         \n"
"      if (d_r + d_b < *d_0)                                                   \n"
"        search_box (s + 1, t - 1, x, y, context);                             \n"
"                                                                              \n"
"      search_box (s + 1, t, x, y, context);                                   \n"
"                                                                              \n"
"      if (d_r + d_t < *d_0)                                                   \n"
"        search_box (s + 1, t + 1, x, y, context);                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* If palettized output is requested return the normalized hash of          \n"
"   * the closest feature point, otherwise return the closest                  \n"
"   * distance. */                                                             \n"
"                                                                              \n"
"  if (context->palettize)                                                     \n"
"    return (gdouble)context->feature / 4294967295.0;                          \n"
"  else                                                                        \n"
"    return pow (context->closest[context->rank - 1], 1 / context->shape);     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  gegl_operation_set_format (operation, \"output\", babl_format (\"Y float\"));\n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  return gegl_rectangle_infinite_plane ();                                    \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"cl_process (GeglOperation       *operation,                                   \n"
"            cl_mem               out_tex,                                     \n"
"            const GeglRectangle *roi)                                         \n"
"{                                                                             \n"
"  GeglProperties *o        = GEGL_PROPERTIES (operation);                     \n"
"  const size_t  gbl_size[] = {roi->width, roi->height};                       \n"
"  size_t        work_group_size;                                              \n"
"  cl_uint       cl_iterations   = o->iterations;                              \n"
"  cl_int        cl_err          = 0;                                          \n"
"  cl_int        cl_x_0          = roi->x;                                     \n"
"  cl_int        cl_y_0          = roi->y;                                     \n"
"  cl_float      cl_scale        = o->scale / 50.0;                            \n"
"  cl_float      cl_shape        = o->shape;                                   \n"
"  cl_uint       cl_rank         = o->rank;                                    \n"
"  cl_uint       cl_seed         = o->seed;                                    \n"
"  cl_int        cl_palettize    = (cl_int)o->palettize;                       \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"  {                                                                           \n"
"    const char *kernel_name[] = {\"kernel_noise\", NULL};                     \n"
"    cl_data = gegl_cl_compile_and_build (noise_cell_cl_source, kernel_name);  \n"
"                                                                              \n"
"    if (!cl_data)                                                             \n"
"      return TRUE;                                                            \n"
"  }                                                                           \n"
"                                                                              \n"
"  cl_err = gegl_cl_set_kernel_args (cl_data->kernel[0],                       \n"
"                                    sizeof(cl_mem), &out_tex,                 \n"
"                                    sizeof(cl_int), &cl_x_0,                  \n"
"                                    sizeof(cl_int), &cl_y_0,                  \n"
"                                    sizeof(cl_uint), &cl_iterations,          \n"
"                                    sizeof(cl_float), &cl_scale,              \n"
"                                    sizeof(cl_float), &cl_shape,              \n"
"                                    sizeof(cl_uint), &cl_rank,                \n"
"                                    sizeof(cl_uint), &cl_seed,                \n"
"                                    sizeof(cl_int), &cl_palettize,            \n"
"                                    NULL);                                    \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clGetKernelWorkGroupInfo (cl_data->kernel[0],                 \n"
"                                          gegl_cl_get_device (),              \n"
"                                          CL_KERNEL_WORK_GROUP_SIZE,          \n"
"                                          sizeof (size_t), &work_group_size,  \n"
"                                          NULL);                              \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[0], 2,                \n"
"                                        NULL, gbl_size, NULL,                 \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clFinish (gegl_cl_get_command_queue ());                      \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"                                                                              \n"
"error:                                                                        \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"c_process (GeglOperation       *operation,                                    \n"
"           void                *out_buf,                                      \n"
"           glong                n_pixels,                                     \n"
"           const GeglRectangle *roi,                                          \n"
"           gint                 level)                                        \n"
"{                                                                             \n"
"  gint factor = (1 << level);                                                 \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"  Context     context;                                                        \n"
"  gfloat     *pixel = out_buf;                                                \n"
"                                                                              \n"
"  gint x = roi->x;                                                            \n"
"  gint y = roi->y;                                                            \n"
"                                                                              \n"
"  context.seed = o->seed;                                                     \n"
"  context.rank = o->rank;                                                     \n"
"  context.shape = o->shape;                                                   \n"
"  context.palettize = o->palettize;                                           \n"
"                                                                              \n"
"  while (n_pixels --)                                                         \n"
"  {                                                                           \n"
"    gint    i;                                                                \n"
"    gdouble c, d;                                                             \n"
"                                                                              \n"
"    /* Pile up noise octaves onto the output value. */                        \n"
"                                                                              \n"
"    for (i = 0, c = 1, d = o->scale / 50.0, *pixel = 0;                       \n"
"         i < o->iterations;                                                   \n"
"         c *= 2, d *= 2, i += 1) {                                            \n"
"      *pixel += noise2 ((double) (x * d * factor),                            \n"
"                        (double) (y * d * factor),                            \n"
"                        &context) / c;                                        \n"
"    }                                                                         \n"
"    pixel += 1;                                                               \n"
"                                                                              \n"
"    x++;                                                                      \n"
"    if (x>=roi->x + roi->width)                                               \n"
"      {                                                                       \n"
"        x=roi->x;                                                             \n"
"        y++;                                                                  \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *out_buf,                                        \n"
"         const GeglRectangle *roi,                                            \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"  const Babl         *out_format = gegl_operation_get_format (operation,      \n"
"                                                              \"output\");    \n"
"                                                                              \n"
"  g_assert(babl_format_get_n_components (out_format) == 1 &&                  \n"
"           babl_format_get_type (out_format, 0) == babl_type (\"float\"));    \n"
"                                                                              \n"
"  if (gegl_operation_use_opencl (operation))                                  \n"
"    {                                                                         \n"
"      GeglBufferClIterator *cl_iter;                                          \n"
"      gboolean              err;                                              \n"
"                                                                              \n"
"      GEGL_NOTE (GEGL_DEBUG_OPENCL, \"GEGL_OPERATION_POINT_RENDER: %s\", GEGL_OPERATION_GET_CLASS (operation)->name);\n"
"                                                                              \n"
"      cl_iter = gegl_buffer_cl_iterator_new (out_buf, roi, out_format, GEGL_CL_BUFFER_WRITE);\n"
"                                                                              \n"
"      while (gegl_buffer_cl_iterator_next (cl_iter, &err) && !err)            \n"
"        {                                                                     \n"
"          err = cl_process (operation, cl_iter->tex[0], cl_iter->roi);        \n"
"                                                                              \n"
"          if (err)                                                            \n"
"            {                                                                 \n"
"              gegl_buffer_cl_iterator_stop (cl_iter);                         \n"
"              break;                                                          \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (err)                                                                \n"
"        GEGL_NOTE (GEGL_DEBUG_OPENCL, \"Error: %s\", GEGL_OPERATION_GET_CLASS (operation)->name);\n"
"      else                                                                    \n"
"        return TRUE;                                                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (out_buf, roi, level, out_format,           \n"
"                                   GEGL_ACCESS_WRITE, GEGL_ABYSS_NONE);       \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"    c_process (operation, iter->data[0], iter->length, &iter->roi[0], level); \n"
"                                                                              \n"
"  return  TRUE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationSourceClass *source_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  source_class = GEGL_OPERATION_SOURCE_CLASS (klass);                         \n"
"                                                                              \n"
"  source_class->process = process;                                            \n"
"  operation_class->get_bounding_box = get_bounding_box;                       \n"
"  operation_class->prepare = prepare;                                         \n"
"  operation_class->opencl_support = TRUE;                                     \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",               \"gegl:cell-noise\",                              \n"
"    \"title\",              _(\"Cell Noise\"),                                \n"
"    \"categories\",         \"render\",                                       \n"
"    \"position-dependent\", \"true\",                                         \n"
"    \"description\", _(\"Generates a cellular texture.\"),                    \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
