package org.snpeff.snpEffect.testCases.unity;

import org.junit.jupiter.api.Test;
import org.snpeff.probablility.Binomial;
import org.snpeff.util.Gpr;
import org.snpeff.util.Log;

import java.util.Random;

/**
 * Test for Binomial distribution
 *
 * @author pcingola
 */
public class TestCasesBinomial {

    int numTests = 100;
    int MAX = 1000;
    Random rand;

    public TestCasesBinomial() {
        super();
        initRand();
    }

    /**
     * Compare pdf result
     */
    void compare(double p, int k, int n, double result) {
        double prob = Binomial.get().pdf(p, k, n);

        double abs = Math.abs(prob - result);
        double diff = abs / Math.min(prob, result);
        if ((abs > 1E-300) && (diff > 0.00000000001))
            throw new RuntimeException("Relative difference:" + diff + "\t\t" + prob + " != " + result);
    }

    /**
     * Compare cdf result
     */
    void compareCdf(double p, int k, int n, double result) {
        double prob = Binomial.get().cdf(p, k, n);

        double abs = Math.abs(prob - result);
        double diff = abs / Math.min(prob, result);
        if ((abs > 1E-300) && (diff > 0.00000000001))
            throw new RuntimeException("Relative difference:" + diff + "\t\t" + prob + " != " + result);
    }

    /**
     * Compare cdf (upper tail) result
     */
    void compareCdfUp(double p, int k, int n, double result) {
        double prob = Binomial.get().cdfUp(p, k, n);

        double abs = Math.abs(prob - result);
        double diff = abs / Math.min(prob, result);
        if ((abs > 1E-300) && (diff > 0.00000000001))
            throw new RuntimeException("Relative difference:" + diff + "\t\t" + prob + " != " + result);
    }

    /**
     * Create new tests
     */
    public void generate_test() {
        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < numTests; i++) {
            int n = rand.nextInt(MAX) + 1;
            int k = rand.nextInt(n);
            double p = rand.nextDouble();
            sb.append("\t print ( paste( 'compare( " + p + ", " + k + ", " + n + ", ', " + Binomial.get().toR(p, k, n) + " , ');' ) );\n");
        }

        String fileName = "binom_test.r";
        Gpr.toFile(fileName, sb);
        System.out.println("Output written to file " + fileName + "\n" + sb);
    }

    void initRand() {
        rand = new Random(20130323);
    }

    @Test
    public void test_01() {
        Log.debug("Test");
        compare(0.2880550055924883, 220, 254, 2.27799916244268e-82);
        compare(0.48868610851489847, 25, 31, 0.000221268194173791);
        compare(0.6629132547126394, 293, 559, 3.86472550479299e-12);
        compare(0.6322928232741156, 65, 66, 2.78440529984676e-12);
        compare(0.3803266972895648, 34, 679, 1.314814699959e-91);
        compare(0.6610024065808093, 343, 618, 1.12493666577642e-08);
        compare(0.9669210109265903, 408, 477, 1.68593574434389e-24);
        compare(0.6525402314581084, 44, 58, 0.0263966083661205);
        compare(0.9075852212570233, 156, 160, 0.000516731716666214);
        compare(0.008028963977602555, 116, 655, 3.15299546965815e-114);
        compare(0.4592882778407693, 343, 673, 0.0010011403382239);
        compare(0.6136164195134856, 381, 551, 2.58984861452828e-05);
        compare(0.3694187143900348, 522, 870, 1.92238491209693e-43);
        compare(0.125818858402313, 562, 644, 0);
        compare(0.20583096399837508, 174, 779, 0.0167344676926336);
        compare(0.11364245975796972, 455, 968, 2.79251390205488e-168);
        compare(0.8466025575004988, 41, 88, 1.25753768651977e-16);
        compare(0.8344910588156836, 347, 497, 2.12940585057056e-14);
        compare(0.17457663652363564, 645, 960, 1.31446122273001e-253);
        compare(0.6210632375421182, 222, 916, 2.46443531298369e-120);
        compare(0.1750847221641334, 31, 85, 1.57856924943353e-05);
        compare(0.21288801046684036, 918, 954, 0);
        compare(0.4448145716632864, 186, 319, 2.16211514395949e-07);
        compare(0.8788423546071854, 278, 607, 1.36461056192434e-137);
        compare(0.585608629673351, 662, 796, 1.42250500263234e-50);
        compare(0.8459406411592152, 402, 552, 5.79579006257795e-13);
        compare(0.0790489201111968, 2, 734, 1.11358896014852e-23);
        compare(0.3748800344755139, 248, 898, 9.9917072417052e-11);
        compare(0.01600419939074571, 66, 170, 7.29206363172316e-72);
        compare(0.8042720779314733, 96, 712, 0);
        compare(0.48988575568058135, 89, 461, 3.72855855797368e-40);
        compare(0.02318557422959655, 213, 355, 5.51250427427537e-248);
        compare(0.5531787211976205, 74, 415, 7.47634070245125e-56);
        compare(0.12557144155829625, 83, 113, 6.14391725015515e-50);
        compare(0.3838854352646155, 435, 537, 4.91259312692835e-91);
        compare(0.45033526491247167, 532, 920, 1.98389261154614e-15);
        compare(0.7118386391143148, 611, 784, 3.53635761635209e-06);
        compare(0.8172689111872286, 722, 993, 8.02121063350573e-13);
        compare(0.6144116421469201, 0, 497, 2.01179908340889e-206);
        compare(0.5317474217631377, 69, 87, 2.66985628502938e-07);
        compare(0.5069736498187624, 350, 589, 4.00297626208328e-06);
        compare(0.6500433296588933, 343, 447, 2.37867791420717e-08);
        compare(0.5255082947335764, 205, 316, 2.58668405401221e-06);
        compare(0.07230530495862175, 387, 912, 6.32237906177793e-191);
        compare(0.5271340615790059, 455, 515, 1.5593929135531e-67);
        compare(0.26169350457388063, 307, 346, 7.43702217204797e-133);
        compare(0.6188353276896835, 77, 496, 1.32661443079689e-100);
        compare(0.7758596391316109, 408, 872, 6.63712049117496e-87);
        compare(0.4655330780964223, 176, 898, 4.0879780160382e-64);
        compare(0.34090457251116146, 520, 744, 3.88907413337851e-88);
        compare(0.6256950714445055, 203, 755, 2.98530111376034e-88);
        compare(0.8981417731056028, 444, 455, 9.06056111764425e-11);
        compare(0.8160048708189973, 504, 767, 6.0793153709495e-26);
        compare(0.9104406932200412, 151, 583, 3.67583185385974e-316);
        compare(0.3161800209813844, 383, 628, 9.05690511647557e-52);
        compare(0.22761853991597336, 8, 122, 1.13257632473261e-06);
        compare(0.6908499179568875, 164, 602, 1.07758923866282e-98);
        compare(0.9792951357166998, 338, 917, 0);
        compare(0.6274735485131088, 726, 810, 7.70906608949334e-68);
        compare(0.28882041735246056, 286, 816, 2.02839319183285e-05);
        compare(0.9675988984427046, 330, 713, 0);
        compare(0.08332925407533187, 39, 107, 5.32406414922273e-16);
        compare(0.9433345819269151, 321, 363, 6.36353486911951e-06);
        compare(0.9126741757513345, 845, 912, 0.0159088843107101);
        compare(0.22172604537405216, 99, 322, 8.52266968196308e-05);
        compare(0.4839313626908692, 438, 445, 5.49208493337041e-126);
        compare(0.6067977365445869, 324, 558, 0.0154639674543825);
        compare(0.5683708756652726, 162, 648, 5.32719836203598e-61);
        compare(0.15234380597429398, 211, 611, 2.61860838130585e-32);
        compare(0.0442270460562022, 159, 659, 3.46750658509665e-69);
        compare(0.9981199891370522, 56, 544, 0);
        compare(0.10130103543856117, 7, 55, 0.131862759852073);
        compare(0.4579093177283158, 182, 276, 7.12584136371065e-12);
        compare(0.05237177126985082, 21, 58, 5.76419840217601e-13);
        compare(0.7600051209686236, 342, 953, 1.38158748043615e-151);
        compare(0.7715878536673834, 540, 659, 0.00044108912011156);
        compare(0.3689746594879658, 420, 435, 3.11942135006798e-158);
        compare(0.5114575394826792, 13, 114, 2.69460464670426e-19);
        compare(0.002137354086410359, 54, 759, 3.08462545559331e-62);
        compare(0.7377497775815932, 20, 43, 9.357811008202e-05);
        compare(0.295011677776578, 4, 38, 0.00385200500010442);
        compare(0.6655642284327655, 31, 57, 0.017303053709838);
        compare(0.5670552362666915, 335, 793, 9.2313826046733e-17);
        compare(0.3800237121096681, 386, 756, 8.98422538361038e-14);
        compare(0.29234274359023893, 423, 676, 4.91063634572408e-72);
        compare(0.7814875783438583, 299, 885, 1.58589939584726e-175);
        compare(0.6893970870750705, 407, 525, 2.84363131681983e-06);
        compare(0.10788027242194886, 619, 799, 0);
        compare(0.3102087391140995, 53, 155, 0.047127674304891);
        compare(0.4556743019731797, 248, 302, 2.8074617748569e-39);
        compare(0.6305493957283269, 20, 49, 0.000803449830003041);
        compare(0.5462619947342483, 350, 392, 2.86441558661771e-50);
        compare(0.4220056222008116, 334, 642, 1.14728462359207e-07);
        compare(0.9658963381392297, 459, 672, 2.31366962689554e-139);
        compare(0.12072919264214377, 299, 348, 8.63337820034743e-218);
        compare(0.24507597522216396, 42, 136, 0.0178998632472746);
        compare(0.49872044334629895, 402, 975, 1.13907816860804e-08);
        compare(0.43719484127406083, 195, 300, 5.72041875212051e-14);
        compare(0.6905317630008837, 675, 885, 3.76251802489755e-07);
        compare(0.30647238081939765, 65, 579, 9.08728110815288e-29);
    }

    @Test
    public void test_02() {
        Log.debug("Test");
        compareCdf(0.5, 10, 20, 0.588098526000976);
        compareCdf(0.5, 9, 20, 0.411901473999024);

        compareCdf(0.0315985578907112, 15, 49, 0.999999999998777);
        compareCdf(0.8416256059557135, 4, 39, 4.10879653660464e-24);
        compareCdf(0.22397504460855688, 459, 498, 1);
        compareCdf(0.5744394449336488, 140, 241, 0.604611000272961);
        compareCdf(0.061072990136159855, 141, 822, 1);
        compareCdf(0.10133694364837431, 766, 848, 1);
        compareCdf(0.3150716238717388, 156, 161, 1);
        compareCdf(0.2812469655480908, 77, 89, 1);
        compareCdf(0.4189870588097322, 44, 806, 4.54480908251841e-124);
        compareCdf(0.814511285922853, 764, 821, 1);
        compareCdf(0.25996015646132087, 506, 706, 1);
        compareCdf(0.2952690901053501, 296, 407, 1);
        compareCdf(0.7411420922813126, 66, 613, 1.08736702090574e-240);
        compareCdf(0.373701970969197, 476, 541, 1);
        compareCdf(0.7666268210300436, 406, 433, 1);
        compareCdf(0.3647176074298739, 43, 85, 0.997176056794048);
        compareCdf(0.9572006649618728, 464, 515, 4.03078544763385e-08);
        compareCdf(0.4960960629909432, 24, 63, 0.0439245898183373);
        compareCdf(0.7142484909716679, 549, 687, 0.999999862164622);
        compareCdf(0.0018173581000535677, 309, 382, 1);
        compareCdf(0.214977429160921, 178, 807, 0.668644110274903);
        compareCdf(0.435234951865472, 215, 360, 0.999999999752134);
        compareCdf(0.544188672182988, 769, 810, 1);
        compareCdf(0.9019546035463769, 80, 251, 8.55245884435596e-110);
        compareCdf(0.44614813029096223, 88, 158, 0.997962715267609);
        compareCdf(0.3937234858024129, 22, 136, 3.74619882334722e-09);
        compareCdf(0.11467767574770327, 178, 192, 1);
        compareCdf(0.04232992803407487, 100, 130, 1);
        compareCdf(0.030408425212244516, 21, 111, 0.999999999997077);
        compareCdf(0.9064592577246974, 145, 421, 1.51684634822297e-174);
        compareCdf(0.10120788723883922, 349, 415, 1);
        compareCdf(0.8546466553675671, 157, 873, 0);
        compareCdf(0.26511369510267313, 13, 322, 8.33649417925935e-27);
        compareCdf(0.21391316318265863, 108, 389, 0.998805400095655);
        compareCdf(0.27467581762557036, 82, 398, 0.000996384194407595);
        compareCdf(0.11636631722226876, 38, 454, 0.014957460397543);
        compareCdf(0.5974310330619085, 40, 99, 7.98186181241761e-05);
        compareCdf(0.8949923960403554, 329, 927, 0);
        compareCdf(0.23930189433868565, 792, 875, 1);
        compareCdf(0.4550620729846766, 3, 633, 3.15731554738885e-160);
        compareCdf(0.82603467979092, 163, 203, 0.216819155207953);
        compareCdf(0.2790172754160042, 41, 274, 2.99115579512493e-07);
        compareCdf(0.4623593273032588, 245, 366, 1);
        compareCdf(0.6104016850615213, 184, 439, 4.5536497584827e-16);
        compareCdf(0.7293761546112356, 465, 814, 2.23200166590086e-22);
        compareCdf(0.40634312226975855, 1, 36, 1.80390893340454e-07);
        compareCdf(0.6777433586700519, 587, 900, 0.0552039785215086);
        compareCdf(0.8482911060010974, 144, 271, 6.75066693158663e-35);
        compareCdf(0.8275603775168999, 262, 325, 0.171040328103452);
        compareCdf(0.557158318350401, 341, 410, 1);
        compareCdf(0.9954120298885412, 142, 485, 0);
        compareCdf(0.1717098865067872, 263, 584, 1);
        compareCdf(0.23485896246815763, 163, 236, 1);
        compareCdf(0.6179899223563772, 61, 525, 1.04203243813297e-126);
        compareCdf(0.29872990151296175, 447, 582, 1);
        compareCdf(0.22276978106085277, 64, 888, 6.25870571148122e-34);
        compareCdf(0.8098920446750989, 755, 972, 0.00541669530502942);
        compareCdf(0.805448072880287, 685, 779, 0.999999986342511);
        compareCdf(0.7617456582185831, 527, 797, 9.79774115902645e-11);
        compareCdf(0.02052775278793373, 312, 926, 1);
        compareCdf(0.06050969542041362, 123, 906, 1);
        compareCdf(0.5167404710358755, 177, 201, 1);
        compareCdf(0.5684352504864514, 533, 787, 0.999999999840666);
        compareCdf(0.9290846574187787, 30, 160, 1.21518177391261e-118);
        compareCdf(0.5104043393235549, 336, 932, 2.34634047039485e-20);
        compareCdf(0.3264879264674764, 172, 382, 0.999999817022784);
        compareCdf(0.16594143837858466, 234, 556, 1);
        compareCdf(0.5005006962249351, 363, 501, 1);
        compareCdf(0.48448553306926434, 167, 530, 1.86569408197832e-15);
        compareCdf(0.3993726396205646, 7, 154, 1.76367165284307e-24);
        compareCdf(0.6338413516089524, 225, 928, 3.25432967083597e-130);
        compareCdf(0.0870591636540885, 487, 923, 1);
        compareCdf(0.5200478723022472, 0, 66, 9.10044870681567e-22);
        compareCdf(0.8381502263251115, 680, 858, 0.0002587454067231);
        compareCdf(0.0014729983874522956, 162, 485, 1);
        compareCdf(0.36740946777806616, 256, 681, 0.692477158938492);
        compareCdf(0.8607639147021706, 683, 978, 4.198987445229e-39);
        compareCdf(0.7365565540733691, 255, 701, 8.62911896601198e-95);
        compareCdf(0.595352076471896, 127, 338, 3.40672530985903e-16);
        compareCdf(0.18112094183261407, 219, 356, 1);
        compareCdf(0.7405329529960012, 113, 777, 6.07540928086231e-266);
        compareCdf(0.9566411634535804, 789, 961, 1.27938834028367e-55);
        compareCdf(0.7487105478964354, 192, 380, 1.88356430897173e-24);
        compareCdf(0.3161782983010937, 237, 261, 1);
        compareCdf(0.4142169373259864, 162, 381, 0.68767873989128);
        compareCdf(0.9000579343393089, 142, 181, 3.38423292446903e-06);
        compareCdf(0.45518783162254484, 187, 740, 4.44240951280495e-30);
        compareCdf(0.14395053439385086, 210, 904, 0.999999999999414);
        compareCdf(0.1422948948203564, 3, 253, 1.78545895498266e-13);
        compareCdf(0.7928447032406584, 262, 719, 3.20803332022473e-136);
        compareCdf(0.7987062767466652, 485, 958, 1.83662404461171e-90);
        compareCdf(0.2461640435178536, 392, 533, 1);
        compareCdf(0.5864670787298277, 42, 327, 1.93661998237421e-66);
        compareCdf(0.25322278198972004, 333, 375, 1);
        compareCdf(0.24156170345539985, 309, 738, 1);
        compareCdf(0.42841136160874294, 564, 611, 1);
        compareCdf(0.32324009811943566, 94, 130, 1);
        compareCdf(0.16194783829735748, 330, 338, 1);
        compareCdf(0.10276092023152972, 15, 205, 0.0960752204197838);
        compareCdf(0.7647269362192434, 83, 535, 1.76845388058446e-195);
    }

    @Test
    public void test_03() {
        Log.debug("Test");
        compareCdfUp(0.5, 10, 20, 0.411901473999024);

        compareCdfUp(0.4325241487200683, 510, 684, 1.2630255422394e-62);
        compareCdfUp(0.10474320011237703, 147, 951, 8.56076782543366e-07);
        compareCdfUp(0.8086366394232849, 465, 624, 0.999939420844905);
        compareCdfUp(0.5677608480471762, 532, 544, 2.63794441691319e-113);
        compareCdfUp(0.36827382689722865, 30, 108, 0.969694377382543);
        compareCdfUp(0.9063470666719343, 98, 116, 0.977207483415814);
        compareCdfUp(0.7477717965376918, 333, 519, 0.999999947441912);
        compareCdfUp(0.429082222974312, 577, 741, 1.36728621707045e-84);
        compareCdfUp(0.5682035504838486, 256, 288, 1.87999383813503e-33);
        compareCdfUp(0.025704546613193657, 470, 952, 0);
        compareCdfUp(0.8122640455477184, 108, 218, 1);
        compareCdfUp(0.7241428992208556, 67, 468, 1);
        compareCdfUp(0.5096089566759726, 193, 343, 0.021545750024841);
        compareCdfUp(0.831836162497284, 276, 306, 0.000173907908449383);
        compareCdfUp(0.9573618767804927, 227, 343, 1);
        compareCdfUp(0.22643336335828113, 1, 404, 1);
        compareCdfUp(0.1683256360142621, 78, 320, 0.000224296344035126);
        compareCdfUp(0.02991374956614834, 169, 247, 1.53748274794725e-195);
        compareCdfUp(0.3615035394582511, 487, 548, 4.46541479506534e-147);
        compareCdfUp(0.725361542137053, 374, 530, 0.833490788403436);
        compareCdfUp(0.0878560883624, 287, 848, 9.59175959148473e-93);
        compareCdfUp(0.7570751571153317, 179, 503, 1);
        compareCdfUp(0.018790148404063456, 575, 761, 0);
        compareCdfUp(0.7923862180027913, 9, 55, 1);
        compareCdfUp(0.5156965951814722, 380, 654, 0.000347264649417081);
        compareCdfUp(0.11750898577710955, 185, 200, 2.23520671252059e-153);
        compareCdfUp(0.8459239707194798, 145, 267, 1);
        compareCdfUp(0.08993588622541504, 28, 32, 1.74202005555296e-27);
        compareCdfUp(0.31389490146198595, 12, 258, 1);
        compareCdfUp(0.17634056793132502, 308, 456, 7.5998827300982e-123);
        compareCdfUp(0.635643368803691, 112, 665, 1);
        compareCdfUp(0.8676446948549519, 316, 875, 1);
        compareCdfUp(0.600870245150411, 331, 736, 1);
        compareCdfUp(0.6818505213958855, 75, 190, 1);
        compareCdfUp(0.4386856711928322, 81, 681, 1);
        compareCdfUp(0.5404363226814495, 26, 96, 0.999999923144423);
        compareCdfUp(0.33152669448712757, 444, 960, 1.60852259034255e-17);
        compareCdfUp(0.035556956954415075, 800, 873, 0);
        compareCdfUp(0.20630888879547438, 368, 395, 9.92398668908747e-216);
        compareCdfUp(0.49298748856674457, 89, 172, 0.236454393516582);
        compareCdfUp(0.27269133854324756, 132, 241, 7.27370892379435e-20);
        compareCdfUp(0.27969285613753025, 236, 342, 1.78015601818654e-56);
        compareCdfUp(0.7754040620929956, 433, 748, 1);
        compareCdfUp(0.505101880270297, 53, 54, 9.60428858078393e-17);
        compareCdfUp(0.08090825412029035, 334, 1000, 1.79780810360313e-115);
        compareCdfUp(0.12160955825420361, 105, 885, 0.581634639406709);
        compareCdfUp(0.5282288365059128, 408, 955, 0.999999999747775);
        compareCdfUp(0.44252618868626303, 220, 261, 1.13065507833527e-41);
        compareCdfUp(0.43961996566353356, 529, 853, 1.20569218602341e-26);
        compareCdfUp(0.26463801465071235, 95, 116, 1.21915452548861e-36);
        compareCdfUp(0.3087942973640214, 173, 327, 4.61683805211065e-17);
        compareCdfUp(0.10773901326227697, 93, 215, 7.02619927591954e-35);
        compareCdfUp(0.46276697117573207, 205, 638, 0.999999999999751);
        compareCdfUp(0.5649384139175128, 41, 625, 1);
        compareCdfUp(0.427935490318074, 677, 942, 2.13304998763009e-73);
        compareCdfUp(0.3952445256827892, 193, 331, 1.89975032960559e-12);
        compareCdfUp(0.8218373562558291, 80, 128, 0.99999988897769);
        compareCdfUp(0.7838702756057898, 710, 979, 0.999991415233207);
        compareCdfUp(0.8390974574651676, 181, 582, 1);
        compareCdfUp(0.14590142661943084, 12, 935, 1);
        compareCdfUp(0.3033971214034189, 711, 924, 3.90215913829478e-188);
        compareCdfUp(0.9554134895141544, 315, 752, 1);
        compareCdfUp(0.8168679228307268, 312, 343, 4.87115865650546e-07);
        compareCdfUp(0.344819015576029, 0, 2, 0.570737877649236);
        compareCdfUp(0.7541327902281663, 116, 231, 1);
        compareCdfUp(0.6381719883253041, 151, 240, 0.590553902958393);
        compareCdfUp(0.7202876602671293, 53, 54, 2.01986633580599e-08);
        compareCdfUp(0.5573513482149981, 681, 946, 2.61153937469008e-25);
        compareCdfUp(0.620437482498488, 197, 695, 1);
        compareCdfUp(0.5714912384582769, 83, 280, 1);
        compareCdfUp(0.6729699828856657, 62, 626, 1);
        compareCdfUp(0.7477206976957012, 581, 645, 2.21951585437424e-23);
        compareCdfUp(0.5395717290960593, 298, 700, 0.999999998987963);
        compareCdfUp(0.5052206632214071, 361, 438, 1.02827846996761e-44);
        compareCdfUp(0.24206067765452277, 151, 985, 0.99999999999439);
        compareCdfUp(0.8748018681062145, 65, 171, 1);
        compareCdfUp(0.6894301755575459, 512, 636, 1.80857848528337e-11);
        compareCdfUp(0.2226523395938067, 183, 219, 6.5031411592594e-84);
        compareCdfUp(0.4480693399917436, 624, 771, 3.625172232218e-95);
        compareCdfUp(0.9095981965394785, 96, 329, 1);
        compareCdfUp(0.8539454587207053, 177, 445, 1);
        compareCdfUp(0.026167358391962825, 171, 179, 6.17339828295183e-261);
        compareCdfUp(0.6876037225584974, 188, 453, 1);
        compareCdfUp(0.16894073974120305, 378, 559, 9.91613371934661e-157);
        compareCdfUp(0.41116072353122457, 556, 886, 6.81159615543622e-39);
        compareCdfUp(0.205120704861534, 167, 173, 1.01894368704286e-107);
        compareCdfUp(0.40103795622406346, 340, 998, 0.999950518024976);
        compareCdfUp(0.10994204067405633, 416, 818, 3.60464271246009e-176);
        compareCdfUp(0.7010016547455058, 25, 61, 0.99999773658803);
        compareCdfUp(0.2554963549247443, 18, 409, 1);
        compareCdfUp(0.567146398968895, 589, 621, 5.99423628111629e-105);
        compareCdfUp(0.9797352184168645, 885, 937, 0.999999999817531);
        compareCdfUp(0.47977948313518437, 0, 1, 0.479779483135184);
        compareCdfUp(0.8708636452891308, 269, 883, 1);
        compareCdfUp(0.35673369502568386, 188, 199, 6.58767706572659e-71);
        compareCdfUp(0.9650302610310897, 58, 323, 1);
        compareCdfUp(0.4184563848209746, 428, 432, 1.28570814575099e-156);
        compareCdfUp(0.9741659644475322, 21, 28, 0.999994363875675);
        compareCdfUp(0.5272930458939531, 266, 685, 0.999999999999802);
        compareCdfUp(0.7360182275604109, 996, 998, 5.12649095529328e-131);
    }
}
