// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package f3

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/path"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
)

type RepositoryDriverInterface interface {
	GetRepositoryURL() string
	GetRepositoryInternalRef() string

	GetPullRequestBranch(*f3.PullRequestBranch) *f3.PullRequestBranch
	CreatePullRequestBranch(*f3.PullRequestBranch)
	DeletePullRequestBranch(*f3.PullRequestBranch)
}

type RepositoryNodeDriverProxyInterface interface {
	RepositoryDriverInterface
}

type RepositoryNodeInterface interface {
	generic.NodeInterface
	RepositoryNodeDriverProxyInterface
}

type repositoryNode struct {
	generic.Node
}

func (o *repositoryNode) GetRepositoryURL() string {
	return o.GetDriver().(RepositoryDriverInterface).GetRepositoryURL()
}

func (o *repositoryNode) GetRepositoryInternalRef() string {
	return o.GetDriver().(RepositoryDriverInterface).GetRepositoryInternalRef()
}

func newRepositoryNode(ctx context.Context, tree generic.TreeInterface) generic.NodeInterface {
	node := &repositoryNode{}
	return node.Init(node)
}

func NewRepositoryPath[T, U any](owners string, owner T, project U) path.Path {
	return generic.NewPathFromString(NewRepositoryPathString(owners, owner, project))
}

func NewRepositoryPathString[T, U any](owners string, owner T, project U) string {
	return fmt.Sprintf("%s/%v/projects/%v/repositories/vcs", owners, owner, project)
}

func NewRepositoryReference[T, U any](owners string, owner T, project U) *f3.Reference {
	return f3.NewReference(NewRepositoryPathString(owners, owner, project))
}

func NewPullRequestSameRepositoryReference() *f3.Reference {
	return f3.NewReference("../../repositories/vcs")
}
