// Author: James Hester
// Licence: GNU General Public License ver. 2+

#define BUILDING_LIBFITYK
#include "f_fcjasym.h"

using namespace std;

namespace fityk {

/* Gauss-Legendre coefficients for numerical integration */
const double FuncFCJAsymm::w100[] = {
0.0312554234538633569476425, 0.0312248842548493577323765, 0.0311638356962099067838183,
0.0310723374275665165878102, 0.0309504788504909882340635, 0.0307983790311525904277139,
0.0306161865839804484964594, 0.0304040795264548200165079, 0.0301622651051691449190687,
0.0298909795933328309168368, 0.0295904880599126425117545, 0.0292610841106382766201190, 0.0289030896011252031348762,
0.0285168543223950979909368, 0.0281027556591011733176483, 0.0276611982207923882942042, 0.0271926134465768801364916,
0.0266974591835709626603847, 0.0261762192395456763423087, 0.0256294029102081160756420, 0.0250575444815795897037642,
0.0244612027079570527199750, 0.0238409602659682059625604, 0.0231974231852541216224889, 0.0225312202563362727017970,
0.0218430024162473863139537, 0.0211334421125276415426723, 0.0204032326462094327668389, 0.0196530874944353058653815,
0.0188837396133749045529412, 0.0180959407221281166643908, 0.0172904605683235824393442, 0.0164680861761452126431050,
0.0156296210775460027239369, 0.0147758845274413017688800, 0.0139077107037187726879541, 0.0130259478929715422855586,
0.0121314576629794974077448, 0.0112251140231859771172216, 0.0103078025748689695857821, 0.0093804196536944579514182,
0.0084438714696689714026208, 0.0074990732554647115788287, 0.0065469484508453227641521, 0.0055884280038655151572119,
0.0046244500634221193510958, 0.0036559612013263751823425, 0.0026839253715534824194396, 0.0017093926535181052395294,
0.0007346344905056717304063};
const double FuncFCJAsymm::x100[] = {
0.0156289844215430828722167, 0.0468716824215916316149239, 0.0780685828134366366948174,
0.1091892035800611150034260, 0.1402031372361139732075146, 0.1710800805386032748875324, 0.2017898640957359972360489,
0.2323024818449739696495100, 0.2625881203715034791689293, 0.2926171880384719647375559, 0.3223603439005291517224766,
0.3517885263724217209723438, 0.3808729816246299567633625, 0.4095852916783015425288684, 0.4378974021720315131089780,
0.4657816497733580422492166, 0.4932107892081909335693088, 0.5201580198817630566468157, 0.5465970120650941674679943,
0.5725019326213811913168704, 0.5978474702471787212648065, 0.6226088602037077716041908, 0.6467619085141292798326303,
0.6702830156031410158025870, 0.6931491993558019659486479, 0.7153381175730564464599671, 0.7368280898020207055124277,
0.7575981185197071760356680, 0.7776279096494954756275514, 0.7968978923903144763895729, 0.8153892383391762543939888,
0.8330838798884008235429158, 0.8499645278795912842933626, 0.8660146884971646234107400, 0.8812186793850184155733168,
0.8955616449707269866985210, 0.9090295709825296904671263, 0.9216092981453339526669513, 0.9332885350430795459243337,
0.9440558701362559779627747, 0.9539007829254917428493369, 0.9628136542558155272936593, 0.9707857757637063319308979,
0.9778093584869182885537811, 0.9838775407060570154961002, 0.9889843952429917480044187, 0.9931249370374434596520099,
0.9962951347331251491861317, 0.9984919506395958184001634, 0.9997137267734412336782285};

const double FuncFCJAsymm::x1024[] = {
0.0015332313560626384065387, 0.0045996796509132604743248,
0.0076660846940754867627839, 0.0107324176515422803327458, 0.0137986496899844401539048, 0.0168647519770217265449962,
0.0199306956814939776907024, 0.0229964519737322146859283, 0.0260619920258297325581921, 0.0291272870119131747190088,
0.0321923081084135882953009, 0.0352570264943374577920498, 0.0383214133515377145376052, 0.0413854398649847193632977,
0.0444490772230372159692514, 0.0475122966177132524285687, 0.0505750692449610682823599, 0.0536373663049299446784129,
0.0566991590022410150066456, 0.0597604185462580334848567, 0.0628211161513580991486838, 0.0658812230372023327000985,
0.0689407104290065036692117, 0.0719995495578116053446277, 0.0750577116607543749280791, 0.0781151679813377563695878,
0.0811718897697013033399379, 0.0842278482828915197978074, 0.0872830147851321356094940, 0.0903373605480943146797811,
0.0933908568511667930531222, 0.0964434749817259444449839, 0.0994951862354057706638682, 0.1025459619163678143852404,
0.1055957733375709917393206, 0.1086445918210413421754502, 0.1116923886981416930665228, 0.1147391353098412365177689,
0.1177848030069850158450139, 0.1208293631505633191883714, 0.1238727871119809777282145, 0.1269150462733265659711591,
0.1299561120276415015747167, 0.1329959557791890421802183, 0.1360345489437231767245806, 0.1390718629487574087024745,
0.1421078692338334288514767, 0.1451425392507896747338214, 0.1481758444640297746894331, 0.1512077563507908736360111,
0.1542382464014118381930443, 0.1572672861196013386077717, 0.1602948470227058049622614, 0.1633209006419772551419632,
0.1663454185228409920472972, 0.1693683722251631675310675, 0.1723897333235182105457458, 0.1754094734074561169859457,
0.1784275640817695987127083, 0.1814439769667610892475458, 0.1844586836985096036255346, 0.1874716559291374498981239,
0.1904828653270767897777182, 0.1934922835773360459175133, 0.1964998823817661533215037, 0.1995056334593266523810493,
0.2025095085463516210358758, 0.2055114793968154435588961, 0.2085115177825984134657778, 0.2115095954937521680517391,
0.2145056843387649520596422, 0.2174997561448267079850562, 0.2204917827580939905255947, 0.2234817360439547026834844,
0.2264695878872926510320010, 0.2294553101927519176581055, 0.2324388748850010462953415, 0.2354202539089970401627982,
0.2383994192302491690277166, 0.2413763428350825830111093, 0.2443509967309017306575811, 0.2473233529464535787923793,
0.2502933835320906316905658, 0.2532610605600337470850902, 0.2562263561246347465424530, 0.2591892423426388177365829,
0.2621496913534467061535080, 0.2651076753193766937613805, 0.2680631664259263621824189, 0.2710161368820341379053566,
0.2739665589203406170790369, 0.2769144047974496674298651, 0.2798596467941893048479266, 0.2828022572158723421886958,
0.2857422083925568078394062, 0.2886794726793061316013119, 0.2916140224564490954412652, 0.2945458301298395466682397,
0.2974748681311158710926665, 0.3004011089179602237287060, 0.3033245249743575146018584, 0.3062450888108541472266190,
0.3091627729648165073212094, 0.3120775500006891993287636, 0.3149893925102530283167230, 0.3178982731128827248285835,
0.3208041644558044102645582, 0.3237070392143528003701590, 0.3266068700922281444141618, 0.3295036298217528976399056,
0.3323972911641281245763845, 0.3352878269096896307981228, 0.3381752098781638207253743, 0.3410594129189232790587667,
0.3439404089112420734451077, 0.3468181707645507759736923, 0.3496926714186912011050938, 0.3525638838441708576370887,
0.3554317810424171123150528, 0.3582963360460310626968790, 0.3611575219190411168852009, 0.3640153117571562777424605,
0.3668696786880191292071420, 0.3697205958714585223322883, 0.3725680364997419586702471, 0.3754119737978276686304337,
0.3782523810236163824397703, 0.3810892314682027913383487, 0.3839224984561266966457784, 0.3867521553456238443366159,
0.3895781755288764427662286, 0.3924005324322633611914264, 0.3952191995166100067331951, 0.3980341502774378774318886,
0.4008453582452137890482864, 0.4036527969855987732669841, 0.4064564400996966449616823, 0.4092562612243022361850445,
0.4120522340321492945489319, 0.4148443322321580436639788, 0.4176325295696824033106488, 0.4204167998267568670171117,
0.4231971168223430347225035, 0.4259734544125757982073747, 0.4287457864910091769763965, 0.4315140869888618022816824,
0.4342783298752620469783905, 0.4370384891574927989076034, 0.4397945388812358755048319, 0.4425464531308160773358662,
0.4452942060294448782650898, 0.4480377717394637499647905, 0.4507771244625871184774399, 0.4535122384401449505463744,
0.4562430879533249674337895, 0.4589696473234144839484647, 0.4616918909120418704091584, 0.4644097931214176352731591,
0.4671233283945751261630457, 0.4698324712156108470282980, 0.4725371961099243891820077, 0.4752374776444579739565725,
0.4779332904279356047259052, 0.4806246091111018260453658, 0.4833114083869600876643171, 0.4859936629910107111699206,
0.4886713477014884570245255, 0.4913444373395996897627612, 0.4940129067697591391182235, 0.4966767308998262548534419,
0.4993358846813411530706387, 0.5019903431097601517846292, 0.5046400812246908935430768, 0.5072850741101270528831987,
0.5099252968946826264179220, 0.5125607247518258033484145, 0.5151913329001124142038603, 0.5178170966034189556133159,
0.5204379911711751889184691, 0.5230539919585963104401304, 0.5256650743669146912153147, 0.5282712138436111840258187,
0.5308723858826459955432696, 0.5334685660246891214197081, 0.5360597298573503421568799, 0.5386458530154087775915395,
0.5412269111810419978382210, 0.5438028800840546885350993, 0.5463737355021068682427603, 0.5489394532609416558499039,
0.5515000092346125858442412, 0.5540553793457104693110943, 0.5566055395655897985264809, 0.5591504659145946930157566,
0.5616901344622843849532002, 0.5642245213276582417822586, 0.5667536026793803239405196, 0.5692773547360034755778519,
0.5717957537661929461605442, 0.5743087760889495408586850, 0.5768163980738322976184566, 0.5793185961411806888254667,
0.5818153467623363454697137, 0.5843066264598643017272666, 0.5867924118077737578782574, 0.5892726794317383594853053,
0.5917474060093159907610475, 0.5942165682701680800580147, 0.5966801429962784154186793, 0.5991381070221714681281111,
0.6015904372351302222163013, 0.6040371105754135078618616, 0.6064781040364728366534687, 0.6089133946651687366701116,
0.6113429595619865853458987, 0.6137667758812519380899084, 0.6161848208313453506363029, 0.6185970716749166931046915,
0.6210035057290989537555048, 0.6234041003657215304299416, 0.6257988330115230076688675, 0.6281876811483634175098794,
0.6305706223134359819666081, 0.6329476340994783351992008, 0.6353186941549832233898213, 0.6376837801844086803419153,
0.6400428699483876768269192, 0.6423959412639372417070377, 0.6447429720046670528676835, 0.6470839401009874959981582,
0.6494188235403171892641570, 0.6517476003672899719207013, 0.6540702486839613549191454, 0.6563867466500144315669620,
0.6586970724829652463040876, 0.6610012044583676196647058, 0.6632991209100174274984589, 0.6655908002301563325302097,
0.6678762208696749663426270, 0.6701553613383155598710345, 0.6724282002048740205051479, 0.6746947160974014538975312,
0.6769548877034051285838219, 0.6792086937700488815250166, 0.6814561131043529626873631, 0.6836971245733933167806834,
0.6859317071045003002812397, 0.6881598396854568318705713, 0.6903815013646959744270519, 0.6925966712514979467122689,
0.6948053285161865628996815, 0.6970074523903250980984011, 0.6992030221669115780303307, 0.7013920172005734910243170,
0.7035744169077619204963997, 0.7057502007669450960906928, 0.7079193483188013616608982, 0.7100818391664115582779368,
0.7122376529754508204546805, 0.7143867694743797837842896, 0.7165291684546352021941915, 0.7186648297708199730232898,
0.7207937333408925681355609, 0.7229158591463558692887801, 0.7250311872324454059827217, 0.7271396977083169940167956,
0.7292413707472337729927181, 0.7313361865867526410034676, 0.7334241255289100847554419, 0.7355051679404074033764222,
0.7375792942527953241676460, 0.7396464849626580085640129, 0.7417067206317964465721772, 0.7437599818874112379620360,
0.7458062494222847584928838, 0.7478455039949627094612890, 0.7498777264299350488635483, 0.7519028976178163024713854,
0.7539209985155252531253957, 0.7559320101464640065565832, 0.7579359136006964320521972, 0.7599326900351259762879594,
0.7619223206736728486546595, 0.7639047868074505764130149, 0.7658800697949419280166093, 0.7678481510621742029486694,
0.7698090121028938864243967, 0.7717626344787406673165402, 0.7737089998194208176678866, 0.7756480898228799321603470,
0.7775798862554750259163361, 0.7795043709521459890141759, 0.7814215258165863961053031, 0.7833313328214136695271245,
0.7852337740083385943114429, 0.7871288314883341834944720, 0.7890164874418038921405657, 0.7908967241187491784979139,
0.7927695238389364107105941, 0.7946348689920631175175217, 0.7964927420379235813750136, 0.7983431255065737724458586,
0.8001860019984956219039900, 0.8020213541847606330100649, 0.8038491648071928284194859, 0.8056694166785310321906380,
0.8074820926825904849673728, 0.8092871757744237908160400, 0.8110846489804811942036542, 0.8128744953987701856100790,
0.8146566981990144342734272, 0.8164312406228120465742028, 0.8181981059837931485700490, 0.8199572776677767911993239,
0.8217087391329271766780945, 0.8234524739099092046215225, 0.8251884656020433364270094, 0.8269166978854597764628854,
0.8286371545092519686128428, 0.8303498192956294067327593, 0.8320546761400697575830038, 0.8337517090114702948057846,
0.8354409019522986425235764, 0.8371222390787428271411563, 0.8387957045808606359402829, 0.8404612827227282810625704,
0.8421189578425883674826439, 0.8437687143529971635802028, 0.8454105367409711729261812, 0.8470444095681330059047621,
0.8486703174708565497995875, 0.8502882451604114359791023, 0.8518981774231068028225812, 0.8535000991204343530350070,
0.8550939951892107040056078, 0.8566798506417190298715048, 0.8582576505658499939545848, 0.8598273801252419702463831,
0.8613890245594205526224495, 0.8629425691839373504743648, 0.8644879993905080694542896, 0.8660253006471498760336444,
0.8675544584983180445842596, 0.8690754585650418856970762, 0.8705882865450599544602407, 0.8720929282129545374252050,
0.8735893694202854169962281, 0.8750775960957229119854680, 0.8765575942451801930826613, 0.8780293499519448719952049,
0.8794928493768098630212838, 0.8809480787582035158255322, 0.8823950244123190181935674, 0.8838336727332430675485994,
0.8852640101930838100201983, 0.8866860233420980458621863, 0.8880996988088177000235219, 0.8895050233001755566829532,
0.8909019836016302565651375, 0.8922905665772905558628607, 0.8936707591700388455969280, 0.8950425484016539302522575,
0.8964059213729330645356690, 0.8977608652638132471078410, 0.8991073673334917701488930, 0.9004454149205460236240486,
0.9017749954430525531228459, 0.9030960963987053701523781, 0.9044087053649335137720782, 0.9057128099990178624646022,
0.9070083980382071951444166, 0.9082954572998335002127549, 0.9095739756814265315746820, 0.9108439411608276105410847,
0.9121053417963026725455006, 0.9133581657266545576127977, 0.9146024011713345435238301, 0.9158380364305531206273175,
0.9170650598853900072573273, 0.9182834599979034047218800, 0.9194932253112384908353520, 0.9206943444497351509745089,
0.9218868061190349456451742, 0.9230705991061873135537215, 0.9242457122797550091847637, 0.9254121345899187738936182,
0.9265698550685812395293315, 0.9277188628294700636112689, 0.9288591470682402950895005, 0.9299906970625759697264543,
0.9311135021722909341445515, 0.9322275518394288975917975, 0.9333328355883627104845635, 0.9344293430258928687940732,
0.9355170638413452433503852, 0.9365959878066680331449597, 0.9376661047765279417201973, 0.9387274046884055757416456,
0.9397798775626900648558921, 0.9408235135027729019444869, 0.9418583026951410028915762, 0.9428842354094689849902736,
0.9439013019987106631201510, 0.9449094928991897628355911, 0.9459087986306898495121205, 0.9468992097965434727052183,
0.9478807170837205248834878, 0.9488533112629158137054760, 0.9498169831886358470168335, 0.9507717237992848297519245,
0.9517175241172498719314184, 0.9526543752489854069548347, 0.9535822683850968193944507, 0.9545011948004232815044368,
0.9554111458541197976665483, 0.9563121129897384560011695, 0.9572040877353088863799924, 0.9580870617034179240840996,
0.9589610265912884783587268, 0.9598259741808576051234879, 0.9606818963388537831043733, 0.9615287850168733926613630,
0.9623666322514563965930439, 0.9631954301641612222071790, 0.9640151709616388439537466, 0.9648258469357060659245549,
0.9656274504634180035311332, 0.9664199740071397636802195, 0.9672034101146173227737943, 0.9679777514190476018682591,
0.9687429906391477383350273, 0.9694991205792235533724866, 0.9702461341292372147270016, 0.9709840242648740939883669,
0.9717127840476088178328839, 0.9724324066247705125950353, 0.9731428852296072415565604, 0.9738442131813496343496072,
0.9745363838852737078785517, 0.9752193908327628781730396, 0.9758932276013691625928266, 0.9765578878548735718130775,
0.9772133653433456910269459, 0.9778596539032024498104955, 0.9784967474572660801033674, 0.9791246400148212617670490,
0.9797433256716714551911835, 0.9803527986101944204270933, 0.9809530530993969223366037, 0.9815440834949686212533729,
0.9821258842393351486632952, 0.9826984498617103674201996, 0.9832617749781478160230522, 0.9838158542915913364912672,
0.9843606825919248853856025, 0.9848962547560215275335618, 0.9854225657477916120303537, 0.9859396106182301300994116,
0.9864473845054632544104222, 0.9869458826347940594679517, 0.9874351003187474227003598, 0.9879150329571141058970610,
0.9883856760369940166627304, 0.9888470251328386495802522, 0.9892990759064927068006818, 0.9897418241072348978090276,
0.9901752655718179181502248, 0.9905993962245076069415402, 0.9910142120771212830473891, 0.9914197092290652598522332,
0.9918158838673715386394944, 0.9922027322667336806727008, 0.9925802507895418581838653, 0.9929484358859170846092543,
0.9933072840937446245820355, 0.9936567920387065844051246, 0.9939969564343136839997662, 0.9943277740819362116746914,
0.9946492418708341635125525, 0.9949613567781865697596566, 0.9952641158691200113800912, 0.9955575162967363309635588,
0.9958415553021395435525955, 0.9961162302144619548145649, 0.9963815384508894965215124, 0.9966374775166862927999356,
0.9968840450052184754903082, 0.9971212385979772738362093, 0.9973490560646014135491635, 0.9975674952628988745188845,
0.9977765541388680773265018, 0.9979762307267185998745420, 0.9981665231488915727109186, 0.9983474296160799746514418,
0.9985189484272491654281575, 0.9986810779696581776171579, 0.9988338167188825964389443, 0.9989771632388403756649803,
0.9991111161818228462260355, 0.9992356742885348165163858, 0.9993508363881507486653971, 0.9994566013984000492749057,
0.9995529683257070064969677, 0.9996399362654382464576482, 0.9997175044023747284307007, 0.9997856720116889628341744,
0.9998444384611711916084367, 0.9998938032169419878731474, 0.9999337658606177711221103, 0.9999643261538894550943330,
0.9999854843850284447675914, 0.9999972450545584403516182};
const double FuncFCJAsymm::w1024[]=
{0.0030664603092439082115513, 0.0030664314747171934849726, 0.0030663738059349007324470, 0.0030662873034393008056861,
0.0030661719680437936084028, 0.0030660278008329004477528, 0.0030658548031622538363679, 0.0030656529766585847450783,
0.0030654223232197073064431, 0.0030651628450145009692318, 0.0030648745444828901040266, 0.0030645574243358210601357,
0.0030642114875552366740338, 0.0030638367373940482295700, 0.0030634331773761048702058, 0.0030630008112961604635720,
0.0030625396432198379186545, 0.0030620496774835909559465, 0.0030615309186946633309249, 0.0030609833717310455112352,
0.0030604070417414288079918, 0.0030598019341451569616257, 0.0030591680546321751827342, 0.0030585054091629766484119,
0.0030578140039685464545661, 0.0030570938455503030247440, 0.0030563449406800369760227, 0.0030555672963998474425352,
0.0030547609200220758572342, 0.0030539258191292371925135, 0.0030530620015739486603347, 0.0030521694754788558725307,
0.0030512482492365564619779, 0.0030502983315095211653578, 0.0030493197312300123682482, 0.0030483124576000001133114,
0.0030472765200910755723677, 0.0030462119284443619831693, 0.0030451186926704230517109, 0.0030439968230491688209395,
0.0030428463301297590067471, 0.0030416672247305038021562, 0.0030404595179387621506312, 0.0030392232211108374894710,
0.0030379583458718709642643, 0.0030366649041157321154111, 0.0030353429080049070377385, 0.0030339923699703840142628,
0.0030326133027115366251721, 0.0030312057191960043331307, 0.0030297696326595705460252, 0.0030283050566060381583022,
0.0030268120048071025720655, 0.0030252904913022221991274, 0.0030237405303984864452325, 0.0030221621366704811776946,
0.0030205553249601516777118, 0.0030189201103766630786495, 0.0030172565082962582916016, 0.0030155645343621134195681,
0.0030138442044841906616068, 0.0030120955348390887083441, 0.0030103185418698906302495, 0.0030085132422860092601062,
0.0030066796530630300711306, 0.0030048177914425515522176, 0.0030029276749320230818149, 0.0030010093213045803019478,
0.0029990627485988779939449, 0.0029970879751189204574353, 0.0029950850194338893942123, 0.0029930539003779692985814,
0.0029909946370501703558363, 0.0029889072488141488505262, 0.0029867917552980250862041, 0.0029846481763941988183689,
0.0029824765322591622023349, 0.0029802768433133102577897, 0.0029780491302407488518214, 0.0029757934139891002022209,
0.0029735097157693059028890, 0.0029711980570554274731990, 0.0029688584595844444331918, 0.0029664909453560499065010,
0.0029640955366324437529314, 0.0029616722559381232326340, 0.0029592211260596712038487, 0.0029567421700455418562030,
0.0029542354112058439815854, 0.0029517008731121217846274, 0.0029491385795971332348581, 0.0029465485547546259626151,
0.0029439308229391107008170, 0.0029412854087656322747309, 0.0029386123371095381418860, 0.0029359116331062444843108,
0.0029331833221509998552933, 0.0029304274298986463828860, 0.0029276439822633785324025, 0.0029248330054184994301727,
0.0029219945257961747508486, 0.0029191285700871841705750, 0.0029162351652406703883623, 0.0029133143384638857180205,
0.0029103661172219362530391, 0.0029073905292375236068160, 0.0029043876024906842306667, 0.0029013573652185263120627,
0.0028982998459149642555740, 0.0028952150733304507490135, 0.0028921030764717064173001, 0.0028889638846014470665859,
0.0028857975272381085212091, 0.0028826040341555690560623, 0.0028793834353828694269858, 0.0028761357612039305018167,
0.0028728610421572684947521, 0.0028695593090357078067012, 0.0028662305928860914743281, 0.0028628749250089892305081,
0.0028594923369584031789413, 0.0028560828605414710856927, 0.0028526465278181672904478, 0.0028491833711010012402964,
0.0028456934229547136488796, 0.0028421767161959702837564, 0.0028386332838930533848701, 0.0028350631593655507170153,
0.0028314663761840422592303, 0.0028278429681697845340603, 0.0028241929693943925796601, 0.0028205164141795195677262,
0.0028168133370965340702726, 0.0028130837729661949782821, 0.0028093277568583240752928, 0.0028055453240914762689974,
0.0028017365102326074839556, 0.0027979013510967402185435, 0.0027940398827466267692845, 0.0027901521414924101257281,
0.0027862381638912825390663, 0.0027822979867471417676962, 0.0027783316471102450029635, 0.0027743391822768604783394,
0.0027703206297889167653083, 0.0027662760274336497592617, 0.0027622054132432473587211, 0.0027581088254944918412282,
0.0027539863027083999392661, 0.0027498378836498606195970, 0.0027456636073272705694208, 0.0027414635129921673927833,
0.0027372376401388605206822, 0.0027329860285040598383428, 0.0027287087180665020331547, 0.0027244057490465746667821,
0.0027200771619059379749851, 0.0027157229973471443987056, 0.0027113432963132558499974, 0.0027069380999874587163979,
0.0027025074497926766073634, 0.0026980513873911808464073, 0.0026935699546841987126055, 0.0026890631938115194351518,
0.0026845311471510979446691, 0.0026799738573186563850015, 0.0026753913671672833892344, 0.0026707837197870311237119,
0.0026661509585045101038391, 0.0026614931268824817854798, 0.0026568102687194489357814, 0.0026521024280492437872770,
0.0026473696491406139791397, 0.0026426119764968062894804, 0.0026378294548551481626046, 0.0026330221291866270351630,
0.0026281900446954674651512, 0.0026233332468187060677353, 0.0026184517812257642618999, 0.0026135456938180188319369,
0.0026086150307283703078113, 0.0026036598383208091684657, 0.0025986801631899798721388, 0.0025936760521607427178014,
0.0025886475522877335418257, 0.0025835947108549212540321, 0.0025785175753751632172710, 0.0025734161935897584747222,
0.0025682906134679988291122, 0.0025631408832067177780710, 0.0025579670512298373098703, 0.0025527691661879125638030,
0.0025475472769576743594882, 0.0025423014326415695994010, 0.0025370316825672995489502, 0.0025317380762873559984451,
0.0025264206635785553113127, 0.0025210794944415703629476, 0.0025157146191004603745948, 0.0025103260880021986466869,
0.0025049139518161981960773, 0.0024994782614338353016280, 0.0024940190679679709626349, 0.0024885364227524702745874,
0.0024830303773417197267843, 0.0024775009835101424263432, 0.0024719482932517112531633, 0.0024663723587794599504176,
0.0024607732325249921551741, 0.0024551509671379883737605, 0.0024495056154857109065099, 0.0024438372306525067265426,
0.0024381458659393083172574, 0.0024324315748631324732279, 0.0024266944111565770692147, 0.0024209344287673158020275,
0.0024151516818575909099866, 0.0024093462248037038747545, 0.0024035181121955041103265, 0.0023976673988358756439882,
0.0023917941397402217940673, 0.0023858983901359478493246, 0.0023799802054619417548485, 0.0023740396413680528093376,
0.0023680767537145683786720, 0.0023620915985716886306938, 0.0023560842322189992961374, 0.0023500547111449424606655,
0.0023440030920462853929883, 0.0023379294318275874140606, 0.0023318337876006648123684, 0.0023257162166840538103394,
0.0023195767766024715869239, 0.0023134155250862753614165, 0.0023072325200709195436049, 0.0023010278196964109553481,
0.0022948014823067621287099, 0.0022885535664494426857857, 0.0022822841308748288053830, 0.0022759932345356507817318,
0.0022696809365864386804193, 0.0022633472963829660967620, 0.0022569923734816920218464, 0.0022506162276392008214839,
0.0022442189188116403333494, 0.0022378005071541580875846, 0.0022313610530203356561684, 0.0022249006169616211363732,
0.0022184192597267597736437, 0.0022119170422612227292520, 0.0022053940257066339981005, 0.0021988502714001954820607,
0.0021922858408741102242558, 0.0021857007958550038097087, 0.0021790951982633439377969, 0.0021724691102128581719720,
0.0021658225940099498722195, 0.0021591557121531123157498, 0.0021524685273323410114303, 0.0021457611024285442134846,
0.0021390335005129516400021, 0.0021322857848465214018174, 0.0021255180188793451473363, 0.0021187302662500514289029,
0.0021119225907852072963166, 0.0021050950564987181231273, 0.0020982477275912256713511, 0.0020913806684495044002679,
0.0020844939436458560249764, 0.0020775876179375023304007, 0.0020706617562659762464561, 0.0020637164237565111901030,
0.0020567516857174286800274, 0.0020497676076395242297101, 0.0020427642551954515246552, 0.0020357416942391048895728,
0.0020286999908050000513193, 0.0020216392111076532034194, 0.0020145594215409583780096, 0.0020074606886775631310555,
0.0020003430792682425467160, 0.0019932066602412715667394, 0.0019860514987017956507927, 0.0019788776619311997736447,
0.0019716852173864757651327, 0.0019644742326995879988655, 0.0019572447756768374356240, 0.0019499969142982240274419,
0.0019427307167168074883601, 0.0019354462512580664378677, 0.0019281435864192559230531, 0.0019208227908687633255086,
0.0019134839334454626590447, 0.0019061270831580672642844, 0.0018987523091844809062265, 0.0018913596808711472808775,
0.0018839492677323979370705, 0.0018765211394497986196010, 0.0018690753658714940398285, 0.0018616120170115510799024,
0.0018541311630493004367905, 0.0018466328743286767122991, 0.0018391172213575569552912, 0.0018315842748070976623218,
0.0018240341055110702429247, 0.0018164667844651949558009, 0.0018088823828264733221690, 0.0018012809719125190225581,
0.0017936626232008872833327, 0.0017860274083284027592567, 0.0017783753990904859184165, 0.0017707066674404779358362,
0.0017630212854889641021349, 0.0017553193255030957535871, 0.0017476008599059107299616, 0.0017398659612756523665312,
0.0017321147023450870266539, 0.0017243471560008201813452, 0.0017165633952826110422716, 0.0017087634933826857546100,
0.0017009475236450491562317, 0.0016931155595647951096823, 0.0016852676747874154134422, 0.0016774039431081072989678,
0.0016695244384710795200224, 0.0016616292349688570408253, 0.0016537184068415843295541, 0.0016457920284763272637533,
0.0016378501744063736542136, 0.0016298929193105323938983, 0.0016219203380124312385075, 0.0016139325054798132252838,
0.0016059294968238317366751, 0.0015979113872983442154825, 0.0015898782522992045381361, 0.0015818301673635540527516,
0.0015737672081691112886347, 0.0015656894505334603439125, 0.0015575969704133379579831, 0.0015494898439039192754876,
0.0015413681472381023085203, 0.0015332319567857911038062, 0.0015250813490531776215856, 0.0015169164006820223329593,
0.0015087371884489335424584, 0.0015005437892646454426166, 0.0014923362801732949073323, 0.0014841147383516970308228,
0.0014758792411086194189814, 0.0014676298658840552399621, 0.0014593666902484950408286, 0.0014510897919021973371136,
0.0014427992486744579821480, 0.0014344951385228783230315, 0.0014261775395326321501237, 0.0014178465299157314469528,
0.0014095021880102909474427, 0.0014011445922797915073771, 0.0013927738213123422970256, 0.0013843899538199418218713,
0.0013759930686377377783877, 0.0013675832447232857518263, 0.0013591605611558067629844, 0.0013507250971354436709363,
0.0013422769319825164387192, 0.0013338161451367762689788, 0.0013253428161566586165863, 0.0013168570247185350852537,
0.0013083588506159642151809, 0.0012998483737589411687807, 0.0012913256741731463215379, 0.0012827908319991927650686,
0.0012742439274918727294554, 0.0012656850410194029319476, 0.0012571142530626688591208, 0.0012485316442144679896043,
0.0012399372951787519644928, 0.0012313312867698677125706, 0.0012227136999117975374834, 0.0012140846156373981740056,
0.0012054441150876388205601, 0.0011967922795108381551550, 0.0011881291902619003419159, 0.0011794549288015500353964,
0.0011707695766955663898644, 0.0011620732156140160807669, 0.0011533659273304853455891, 0.0011446477937213110513287,
0.0011359188967648107958214, 0.0011271793185405120501566, 0.0011184291412283803494364, 0.0011096684471080465391373,
0.0011008973185580330843445, 0.0010921158380549794491381, 0.0010833240881728665534171, 0.0010745221515822403144596,
0.0010657101110494342805238, 0.0010568880494357913638046, 0.0010480560496968846800697, 0.0010392141948817375023057,
0.0010303625681320423357186, 0.0010215012526813791214350, 0.0010126303318544325762649, 0.0010037498890662086758941,
0.0009948600078212502888805, 0.0009859607717128519688418, 0.0009770522644222739122264, 0.0009681345697179550890732,
0.0009592077714547255541688, 0.0009502719535730179460261, 0.0009413272000980781811114, 0.0009323735951391753507612,
0.0009234112228888108282347, 0.0009144401676219265933610, 0.0009054605136951127822476, 0.0008964723455458144695262,
0.0008874757476915376906225, 0.0008784708047290547115472, 0.0008694576013336085537138, 0.0008604362222581167813022,
0.0008514067523323745586954, 0.0008423692764622569855308, 0.0008333238796289207169173, 0.0008242706468880048763834,
0.0008152096633688312691343, 0.0008061410142736039032099, 0.0007970647848766078261514, 0.0007879810605234072847989,
0.0007788899266300432158601, 0.0007697914686822300749096, 0.0007606857722345520114971, 0.0007515729229096583980656,
0.0007424530063974587204051, 0.0007333261084543168373926, 0.0007241923149022446178008, 0.0007150517116280949619884,
0.0007059043845827542163241, 0.0006967504197803339882351, 0.0006875899032973623698204, 0.0006784229212719745780188,
0.0006692495599031030193850, 0.0006600699054496667875923, 0.0006508840442297606018626, 0.0006416920626198431946113,
0.0006324940470539251567018, 0.0006232900840227562488244, 0.0006140802600730121876541, 0.0006048646618064809156059,
0.0005956433758792483631993, 0.0005864164890008837132649, 0.0005771840879336241764943, 0.0005679462594915592881427,
0.0005587030905398147360662, 0.0005494546679937357307118, 0.0005402010788180699282026, 0.0005309424100261499182844,
0.0005216787486790752896494, 0.0005124101818848942860548, 0.0005031367967977850677401, 0.0004938586806172365939677,
0.0004845759205872291441124, 0.0004752886039954144966810, 0.0004659968181722957880391, 0.0004567006504904070755681,
0.0004474001883634926336095, 0.0004380955192456860150653, 0.0004287867306306889171352, 0.0004194739100509498966958,
0.0004101571450768429896514, 0.0004008365233158462997325, 0.0003915121324117206363681, 0.0003821840600436882993131,
0.0003728523939256121308821, 0.0003635172218051749865499, 0.0003541786314630598135175, 0.0003448367107121305776064,
0.0003354915473966143456333, 0.0003261432293912849189248, 0.0003167918446006485317858, 0.0003074374809581322877037,
0.0002980802264252762217455, 0.0002887201689909301727620, 0.0002793573966704570567274, 0.0002699919975049447012834,
0.0002606240595604292032823, 0.0002512536709271339139118, 0.0002418809197187298044384, 0.0002325058940716253739001,
0.0002231286821442978268308, 0.0002137493721166826096154, 0.0002043680521896465790359, 0.0001949848105845827899210,
0.0001855997355431850062940, 0.0001762129153274925249194, 0.0001668244382203495280013, 0.0001574343925265138930609,
0.0001480428665748079976500, 0.0001386499487219861751244, 0.0001292557273595155266326, 0.0001198602909254695827354,
0.0001104637279257437565603, 0.0001010661269730276014588, 0.0000916675768613669107254, 0.0000822681667164572752810,
0.0000728679863190274661367, 0.0000634671268598044229933, 0.0000540656828939400071988, 0.0000446637581285753393838,
0.0000352614859871986975067, 0.0000258591246764618586716, 0.0000164577275798968681068, 0.0000070700764101825898713};

///////////////////////////////////////////////////////////////////////
/* The FCJAsymm peakshape is that described in Finger, Cox and Jephcoat (1994)
   J. Appl. Cryst. vol 27, pp 892-900. */

realt FuncFCJAsymm::dfunc_int(realt twopsi, realt twotheta) const {
/* The integral of the FCJ weight function:
0.5(asin((2 cos (twotheta)^2 + 2 sin(twopsi)-2)/|2 sin twopsi - 2|sin(twotheta)) -
    asin((2 cos (twotheta)^2 - 2 sin(twopsi)-2)/|2 sin twopsi + 2|sin(twotheta)) )
Twotheta and twopsi are in radians.  We note that the limit as twopsi -> twotheta
is pi/2.

Note that callers will need to apply the 1/(2_hl) factor found in the FCJ paper.
*/
if(twopsi == 0) return 0.0;
if(fabs(twopsi - twotheta)<1e-8) return M_PI/2.0;
realt stwopsi = sin(twopsi);
realt stwoth  = sin(twotheta);
realt ctwoth = cos(twotheta);
return 0.5 * (asin((2.0*ctwoth*ctwoth + 2*stwopsi -2)/(abs(2*stwopsi-2)*stwoth)) -
       asin((2.0*ctwoth*ctwoth - 2*stwopsi -2)/(abs(2*stwopsi+2)*stwoth)));
}

static double sq(double a) { return a*a; }

void FuncFCJAsymm::more_precomputations()
{
denom=0.0;
radians = M_PI/180.0;
cent_rad = av_[1]*radians;
realt hfunc_neg, hfunc_pos;
// If either of the below give a cosine greater than one, set to 0
// Handle extrema by setting twopsimin to appropriate limit
 twopsimin = 0.0;
 if (cent_rad > M_PI/2) twopsimin = M_PI;
realt cospsimin = cos(cent_rad)*sqrt(sq(av_[4]+av_[5]) + 1.0);
 if(fabs(cospsimin)<1.0) twopsimin = acos(cospsimin);
twopsiinfl = 0.0;
realt cospsiinfl = cos(cent_rad)*sqrt(sq(av_[4]-av_[5]) + 1.0);
 if(fabs(cospsiinfl)<1.0) twopsiinfl = acos(cospsiinfl);
 if(av_[4] == 0 && av_[5] == 0) denom = 1.0;
 else {
/* The denominator for the FCJ expression can be calculated analytically. We define it in terms
of the integral of the weight function, dfunc_int:
 denom = 2* min(h_l,s_l) * (pi/2 - dfunc_int (twopsiinfl,twotheta)) +
         h_l * (v-u) + s_l * (v-u) - (extra_int (twopsiinfl,twotheta) - extra_int (twopsimin,twotheta))

where v = 1/(2h_l) * dfunc_int(twopsiinfl,twotheta) and u = 1/2h_l * dfunc_int(twopsimin,twotheta).
extra_int is the integral of 1/cos(psi).
*/
realt u = 0.5*dfunc_int(twopsimin,cent_rad)/av_[4];
realt v = 0.5*dfunc_int(twopsiinfl,cent_rad)/av_[4];
denom_unscaled = 2.0 * min(av_[5],av_[4]) * (M_PI/(4.0*av_[4]) - v) + (av_[4] + av_[5])* (v - u) -
   (1.0/(2*av_[4]))*0.5*(log(fabs(sin(twopsiinfl) + 1)) - log(fabs(sin(twopsiinfl)-1)) -
                     log(fabs(sin(twopsimin) + 1)) + log(fabs(sin(twopsimin)-1)));
denom = denom_unscaled * 2.0/fabs(cent_rad-twopsimin);     //Scale to [-1,1] interval of G-L integration
// The following two factors are the analytic derivatives of the integral of D with respect to
// h_l and s_l.
realt uu = dfunc_int(twopsiinfl,cent_rad);
realt vv = dfunc_int(twopsimin,cent_rad);
df_dh_factor = (1.0/(2*av_[4]))*(uu - vv) - (1.0/av_[4])*denom_unscaled;
if (av_[4]<av_[5]) {
	df_dh_factor += (1.0/(2*av_[4]))*(M_PI - 2*uu);
        df_ds_factor = (1.0/(2*av_[4]))*(uu - vv);
        } else {
        df_ds_factor = (1.0/(2*av_[4]))*(M_PI - (uu+vv));
}
// Precalculate the x and hfunc coefficients
for(int pt=0;pt<512;pt++) {
    delta_n_neg[pt] = (cent_rad + twopsimin)/2.0 - (cent_rad-twopsimin)*x1024[pt]/2;
    delta_n_pos[pt] = (cent_rad + twopsimin)/2.0 + (cent_rad-twopsimin)*x1024[pt]/2;
    hfunc_neg = sqrt(sq(cos(delta_n_neg[pt]))/sq(cos(cent_rad)) -1);
    hfunc_pos = sqrt(sq(cos(delta_n_pos[pt]))/sq(cos(cent_rad))-1);
    //Weights for negative half of G-L interval
    if(fabs(cos(delta_n_neg[pt])) > fabs(cos(twopsiinfl))) {
            weight_neg[pt] = av_[4] + av_[5] - hfunc_neg;
        } else {
            weight_neg[pt] = 2 * min(av_[4],av_[5]);
        }
    //Weights for positive half of G-L interval
    weight_neg[pt] = weight_neg[pt]/(2.0*av_[4]*hfunc_neg*abs(cos(delta_n_neg[pt])));
    if(fabs(cos(delta_n_pos[pt])) > fabs(cos(twopsiinfl))) {
        weight_pos[pt] = av_[4] + av_[5] - hfunc_pos;
        } else {
        weight_pos[pt] = 2 * min(av_[4],av_[5]);
        }
    weight_pos[pt] = weight_pos[pt]/(2.0*av_[4]*hfunc_pos*abs(cos(delta_n_pos[pt])));
    // Apply Gauss-Legendre weights
    weight_pos[pt]*=w1024[pt];
    weight_neg[pt]*=w1024[pt];
    }
 }
}

bool FuncFCJAsymm::get_nonzero_range(double level,
                                        realt &left, realt &right) const
{
if (level == 0)
        return false;
    else if (fabs(level) >= fabs(av_[0]))
        left = right = 0;
    else {
      // As for Pseudo-Voigt, allow 4 half-widths for Gaussian
      realt pvw = av_[2]*(sqrt(fabs(av_[0]/(level*M_PI*av_[2])-1))+4.); //halfwidths for Lorenzian
      // The first non-zero point occurs when the convoluting PV reaches twopsimin. The
      // last value occurs when the convoluting PV moves completely past the centre
      if(av_[1] < 90) {
        left = twopsimin*180/M_PI - pvw;
        right = av_[1] + pvw;
      } else {
        left = av_[1] - pvw;
        right = twopsimin*180/M_PI + pvw;
      }
    }
    return true;
}

//Pseudo-Voigt with scaling factors as used in crystallography.
realt FuncFCJAsymm::fcj_psv(realt x, realt location, realt fwhm, realt mixing) const {
    realt xa1a2 = (location - x) / fwhm;
    realt ex = exp(- 4.0 * M_LN2 * xa1a2 * xa1a2);
    ex *= sqrt(4.0*M_LN2/M_PI)/fwhm;
    realt lor = 2. / (M_PI * fwhm * (1 + 4* xa1a2 * xa1a2));
    return  (1-mixing) * ex + mixing * lor;
}

CALCULATE_VALUE_BEGIN(FuncFCJAsymm)
    realt numer = 0.0;
    realt fwhm_rad = av_[2]*2*M_PI/180.0;  // Fityk uses hwhm, we use fwhm
if((av_[4]==0 && av_[5]==0) || cent_rad==M_PI/2) {     // Plain PseudoVoigt
  numer = fcj_psv(x*radians,cent_rad,fwhm_rad, av_[3]);
 } else {
    //do the sum over 1024 Gauss-Legendre points
    for(int pt=0; pt < 512; pt++) {
   /* Note that the Pseudo-Voight equation for this calculation is that used
      in powder diffraction, where the coefficients are chosen to give matching
      width parameters, i.e. only one width parameter is necessary and the
      width is expressed in degrees (which means a normalised height in
      degrees */
   // negative and positive sides
      realt psvval = 0.0;
    psvval = fcj_psv(delta_n_neg[pt],x*radians,fwhm_rad,av_[3]);
    numer += weight_neg[pt] * psvval;
    psvval = fcj_psv(delta_n_pos[pt],x*radians,fwhm_rad,av_[3]);
    numer += weight_pos[pt] * psvval;
    }
 }
    //Radians scale below to make up for fwhm_rad in denominator of PV function
CALCULATE_VALUE_END(av_[0]*M_PI/180 * numer/denom)

CALCULATE_DERIV_BEGIN(FuncFCJAsymm)
    realt fwhm_rad = av_[2]*2*M_PI/180.0;
    realt numer = 0.0;
    realt hfunc_neg=0.0,hfunc_pos=0.0;  //FCJ hfunc for neg, pos regions of G-L integration
    realt sumWdGdh = 0.0;  // derivative with respect to H/L
    realt sumWdGds = 0.0;  // derivative with respect to S/L
    realt sumWdRdG = 0.0; // sum w_i * dR/dgamma * W(delta,twotheta)/H
    realt sumWdRde = 0.0 ;// as above with dR/deta
    realt sumWdRdt = 0.0; // as above with dR/d2theta

    //do the sum over 1024 points
    // parameters are height,centre,hwhm,eta (mixing),H/L,S/L
    //                  0      1     2      3          4   5
    for(int pt=0; pt < 512; pt++) {
      for(int side=0;side < 2; side++) {
    realt xa1a2 = 0.0;
    if(side == 0) xa1a2 = (x*radians - delta_n_neg[pt]) / fwhm_rad;
    else          xa1a2 = (x*radians - delta_n_pos[pt]) / fwhm_rad ;
    realt facta = -4.0 * M_LN2 * xa1a2 ;
    realt ex = exp(facta * xa1a2);
    ex *= sqrt(4.0*M_LN2/M_PI)/fwhm_rad;
    realt lor = 2. / (M_PI * fwhm_rad *(1 + 4* xa1a2 * xa1a2));
    realt without_height =  (1-av_[3]) * ex + av_[3] * lor;
    realt psvval = av_[0] * without_height;
    if(side == 0) {
        numer += weight_neg[pt] * psvval;
        hfunc_neg = 1/(2.0*av_[4]*sqrt(sq(cos(delta_n_neg[pt]))/sq(cos(cent_rad))-1));
    } else
    if(side == 1) {           //So hfunc nonzero
        numer += weight_pos[pt] * psvval;
        hfunc_pos = 1.0/(2.0*av_[4]*sqrt(sq(cos(delta_n_pos[pt]))/sq(cos(cent_rad))-1));
    }
    // pseudo-voigt derivatives: first fwhm.  The parameter is expressed in
    // degrees, but our calculations use radians,so we remember to scale at the end.
    realt dRdg = ex/fwhm_rad * (-1.0 + -2.0*facta*xa1a2);  //gaussian part:checked
    dRdg = av_[0] * ((1-av_[3])* dRdg + av_[3]*(-1.0*lor/fwhm_rad +
                                                16*xa1a2*xa1a2/(M_PI*(fwhm_rad*fwhm_rad))*1.0/sq(1.0+4*xa1a2*xa1a2)));
    realt dRde =  av_[0] * (lor - ex);           //with respect to mixing
    realt dRdtt = -1.0* av_[0] * ((1.0-av_[3])*2.0*ex*facta/fwhm_rad - av_[3]*lor*8*xa1a2/(fwhm_rad*(1+4*xa1a2*xa1a2)));
    if(side==0) {
    /* We know that d(FCJ)/d(param) = sum w[i]*dR/d(param) * W/H(delta) */
    sumWdRdG += weight_neg[pt] * dRdg;
    sumWdRde += weight_neg[pt] * dRde;
    sumWdRdt += weight_neg[pt] * dRdtt;
    } else {
      sumWdRdG += weight_pos[pt] * dRdg;
      sumWdRde += weight_pos[pt] * dRde;
      sumWdRdt += weight_pos[pt] * dRdtt;
    }
    /* The derivative for h_l includes the convolution of dfunc with PV only up
       to twopsiinfl when s_l < h_l  as it is zero above this limit, and
       likewise for s_l when h_l < s_l. To save
       program bytes, we keep the same G-L interval and therefore quadrature
       points.  This is defensible as it is equivalent to including the zero
       valued points in the integration.
       The derivative for peak "centre" given here ignores the contribution
       from changes in the weighting function, as it is not possible to
       numerically integrate the derivative of dfunc that appears in the
       full expression.  It seems to work. */
    realt angpt = 0;
    realt dconvol = 0;
    if (side == 1) {
      angpt = cos(delta_n_pos[pt]);
      dconvol =  w1024[pt] * psvval * hfunc_pos / abs(angpt);
      }
    else
      {
      angpt = cos(delta_n_neg[pt]);
      dconvol =  w1024[pt] * psvval * hfunc_neg / abs(angpt);
      }
    if(fabs(angpt) > fabs(cos(twopsiinfl))){   //further from centre than psi_infl
      sumWdGdh += dconvol;
      sumWdGds += dconvol;
    }
    if(fabs(angpt) < fabs(cos(twopsiinfl))) {  //closer to centre than psi_infl
    if (av_[5] < av_[4]) {   // H_L is larger
      sumWdGds += 2.0* dconvol;
    } else {   // S_L is larger
      sumWdGdh += 2.0 * dconvol;
    }
    }
    }
    }
    // Note that we must scale any numerically integrated terms back to the correct interval
    // This scale factor cancels for numer/denom, but not for e.g. numer/denom^2
dy_dv[0] = M_PI/180 * numer/(av_[0]*denom);        // height derivative(note numer contains height)
dy_dv[1] = M_PI*M_PI/(180*180) * sumWdRdt/denom;                // peak position in degrees
dy_dv[2] = 2*M_PI*M_PI/(180*180) * sumWdRdG/denom;     // fwhm/2 is hwhm, in degrees
dy_dv[3] = M_PI/180 * sumWdRde/denom;     // mixing
dy_dv[4] = M_PI/180 * (sumWdGdh/denom - 1.0/av_[4] * numer/denom -
               df_dh_factor*numer/(denom_unscaled*denom));                // h_l
dy_dv[5] = M_PI/180* (sumWdGds/denom - df_ds_factor * numer/(denom*denom_unscaled)); // s_l
dy_dx = -1.0*dy_dv[1];
CALCULATE_DERIV_END(M_PI/180 * numer/denom)

} // namespace fityk
