#ifndef __TGDB_COMMAND_H__
#define __TGDB_COMMAND_H__

/*! 
 * \file
 * tgdb_command.h
 *
 * \brief
 * This interface is intended to allow TGDB to represent a single command.
 */

#include "tgdb_types.h"

/**
 * The current command type. TGDB is capable of having any commands of this
 * type in it's queue.
 *
 * I don't know what this should look like completly.
 */
enum tgdb_command_choice {

    /**
	 * A command from the front end
	 */
    TGDB_COMMAND_FRONT_END,

    /**
	 * A command from the console
	 */
    TGDB_COMMAND_CONSOLE,

    /**
	 * A command from a client of TGDB
	 */
    TGDB_COMMAND_TGDB_CLIENT,

    /**
	 * A priority command is a command that the client context needs
	 * to run before it can finish getting the data necessary for a 
	 * TGDB_COMMAND_TGDB_CLIENT command.
	 */
    TGDB_COMMAND_TGDB_CLIENT_PRIORITY
};

/* This is here to add new client_command/command faster */

/**
 * This is the main tgdb queue command.
 * It will be put into a queue and run.
 * For each TGDB command generated by a client, the client command is
 * with it.
 */
struct tgdb_command {
    /**
	 * The actual command to give.
	 */
    char *tgdb_command_data;

    /**
	 * The type of command this one is.
	 */
    enum tgdb_command_choice command_choice;

    /**
	 * The client command associated with this command.
	 */
    struct tgdb_client_command *client_command;

    /** Private data the client context can use. */
    void *tgdb_client_private_data;
};

/**
 * Creates a new command and initializes it 
 *
 * \param tgdb_command_data
 * The data needed to run the command
 *
 * \param command_choice
 * The type of command to run.
 *
 * \param action_choice
 * The type of action TGDB would like to perform.
 *
 * \param client_data
 * Data that the client can use when prepare_client_for_command is called
 *
 * @return
 * Always is successfull, will call exit on failed malloc
 */
struct tgdb_command *tgdb_command_create(const char *tgdb_command_data,
        enum tgdb_command_choice command_choice, void *client_data);

/** 
 * This will free a TGDB queue command.
 * These are the commands given by TGDB to the debugger.
 * This is a function for debugging. 
 *
 * \param item
 * The command to free
 */
void tgdb_command_destroy(void *item);

/**
 * This will print a TGDB queue command to stderr.
 * These are the commands given by TGDB to the debugger.
 * This is a function for debugging. 
 *
 * \param item
 * The command to print
 */
void tgdb_command_print(void *item);

#endif
