//meta.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2011-2013
 *
 *  This file is part of aroarfw, a RoarAudio framework for
 *  embedded systems (Controlers).
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  aroarfw is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef _AROARFW_META_H_
#define _AROARFW_META_H_

//! Meta data types
typedef enum {
 //! This is used for empty meta data elements
 RMETA_NONE               = 0,
 //! Title of work
 RMETA_TITLE              = 1,
 //! Album work is part of
 RMETA_ALBUM              = 2,
 //! Author of work
 RMETA_AUTHOR             = 3,
 //! Alias for RMETA_AUTHOR.
 //! Should not be used in code but in functions to convert meta data types from strings to IDs.
 RMETA_ARTIST             = RMETA_AUTHOR,
 //! Version of work
 RMETA_VERSION            = 4,
 //! Date of creation. Format: "YYYY-MM-DD" or "YYYY-MM-DD HH:MM:SS UTC".
 //! Time in later form is start time of recording.
 RMETA_DATE               = 5,
 //! License of work
 RMETA_LICENSE            = 6,
 //! Track number of work within the album (or other compilation). (printf: %.2d)
 RMETA_TRACKNUMBER        = 7,
 //! Organization of creation
 RMETA_ORGANIZATION       = 8,
 //! General description
 RMETA_DESCRIPTION        = 9,
 //! Genre.
 //! Freeform string but should match one of the genres defined for the RoarAudio protocol or in form of
 //! %s(0x%x) with readable name and genre ID.
 RMETA_GENRE              = 10,
 //! Location of work creation
 RMETA_LOCATION           = 11,
 //! Contact information for work.
 //! Should be in form: first ["']nick["'] last (comment) <email>/OpenPGPkey/Phone/Room)
 RMETA_CONTACT            = 12,
 //! URL of this stream.
 //! Used for examples in stream dumps as source refrence.
 //! Also used in VCLT Playlists.
 //! See also RMETA_FILEURL.
 RMETA_STREAMURL          = 13,
 //! Homepage of this album, compilation or work
 RMETA_HOMEPAGE           = 14,
 //! URL to thumbnail of this work.
 //! Can be used for display within the media player.
 //! Should point to PNG or JPEG image.
 RMETA_THUMBNAIL          = 15,
 //! Playback length of the work.
 //! (printf: %.2d:%.2d:%.2d, last element may be float with dot as decimal delimeter)
 //! Main use for this is meta data stored outside the data stream. Examples
 //! include VCLT Playlist files.
 RMETA_LENGTH             = 16,
 //! Free form comment about this work
 RMETA_COMMENT            = 17,
 //! The OTHER meta data type can be used to use non-standardized meta data which
 //! have no ID assigned.
 //! Usage of this should be avoided.
 RMETA_OTHER              = 18,
 //! Filename of the given work. Mainly used in VCLT Playlists.
 RMETA_FILENAME           = 19,
 //! URL of the work. Mainly used in VCLT Playlists.
 //! The diffrence between RMETA_STREAMURL and RMETA_FILEURL is that
 //! RMETA_STREAMURL is for (radio) streams and RMETA_FILEURL for files of finit size
 //! (often refrered to as seekable and unseekable data).
 RMETA_FILEURL            = 20,
 //! Server software name and version
 RMETA_SERVER             = 21,
 //! This is used to store the duration of some work.
 //! It's format and usage is undefined.
 //! This should not be used.
 //! See also: RMETA_LENGTH.
 RMETA_DURATION           = 22,
 //! Alias for RMETA_HOMEPAGE.
 //! Should not be used in code but in functions to convert meta data types from strings to IDs.
 RMETA_WWW                = RMETA_HOMEPAGE,
 //! ID3: Official audio file webpage.
 //! Should not be used when creating new files.
 RMETA_WOAF               = 23,
 //! Encoder software name and version
 RMETA_ENCODER            = 24,
 //! Alias to RMETA_ENCODER.
 //! Should not be used when creating new files.
 RMETA_ENCODEDBY          = RMETA_ENCODER,
 //! Year of recording in form YYYY
 RMETA_YEAR               = 25,
 //! CDDB DiskID, (printf: %.8x)
 RMETA_DISCID             = 26,
 //! RePlayGain of track: Peak amplitude.
 RMETA_RPG_TRACK_PEAK     = 27,
 //! RePlayGain of track: correction gain.
 RMETA_RPG_TRACK_GAIN     = 28,
 //! RePlayGain of album: Peak amplitude.
 RMETA_RPG_ALBUM_PEAK     = 29,
 //! RePlayGain of album: correction gain.
 RMETA_RPG_ALBUM_GAIN     = 30,
 //! Used to store hashes for HashTable or Tantalos IDs.
 //! Format: {TYPE}value[,{TYPE}value[,{TYPE}value[,...]]]
 //! For possible types see rht_t from crypto.h.
 //! See also RMETA_GTIN.
 RMETA_HASH               = 31,
 //! Information about the signal.
 //! Used in VCLT Playlist files.
 //! See https://bts.keep-cool.org/wiki/Specs/SignalInfoFormat for format specification.
 RMETA_SIGNALINFO         = 32,
 //! Alias for RMETA_SIGNALINFO. Only valid for Audio only streams.
 //! See https://bts.keep-cool.org/wiki/Specs/SignalInfoFormat for details.
 RMETA_AUDIOINFO          = RMETA_SIGNALINFO,
 //! Offset of the work within the refrenced file or data stream.
 //! This is used by VCLT Playlist files.
 //! The exact format is not yet defined.
 RMETA_OFFSET             = 33,
 //! The performer of the work
 RMETA_PERFORMER          = 34,
 //! Copyright information of the work
 RMETA_COPYRIGHT          = 35,
 //! Likeness of the given work by the user.
 //! Used in VCLT Playlist files.
 //! (printf: %f with dot as decimal delimeter).
 //! Range: 0.0: user dislikes this very much (is never played by player),
 //!        1.0: normal likeness.
 //!        Inf: The user likes this very much (is played by player endlessly).
 RMETA_LIKENESS           = 36,
 //! The composer of the work
 RMETA_COMPOSER           = 37,
 //! XXX
 RMETA_RIGHTS             = 38,
 //! The ISRC code for the work. See Vorbis comments specification.
 RMETA_ISRC               = 39,
 //! The language of this work in POSIX notation or short POSIX notation (two letter code).
 //! Possible values include 'en' for English, 'de' for German.
 //! 'de_DE' for Germany German, 'de_CH' for Switzerland German.
 RMETA_LANGUAGE           = 40,
 //! GTIN of work.
 //! Possible formats are 14 digits GTIN-14, 13 digits GTIN-13 (most common),
 //! 8 digit GTIN-8 (also common) or ISBN-13 without dashes (which is the same as the corresponding
 //! GTIN-13). EAN-8 and EAN-13 are synonyms for GTIN-8 and GTIN-13.
 //! The data is a string of digits without dashes or spaces.
 //! Any dashes or spaces should be ignored by the reading application.
 RMETA_GTIN               = 41,
 //! This is an alias for RMETA_GTIN.
 //! Should not be used in code but in functions to convert meta data types from strings to IDs.
 //! If used in files as string like VCLT Playlist files only ISBN-13 values are valid within the meta date.
 RMETA_ISBN               = RMETA_GTIN,
 //! This is an alias for RMETA_GTIN.
 //! Should not be used in code but in functions to convert meta data types from strings to IDs.
 //! If used in files as string like VCLT Playlist files only EAN-8 or EAN-13 values are valid within the meta date.
 RMETA_EAN                = RMETA_GTIN,
 //! The person or company who published this work
 RMETA_PUBLISHER          = 42,
 //! The number of the disc within the compilation or album
 RMETA_DISCNUMBER         = 43,
 //! The source media this was converted from
 RMETA_SOURCEMEDIA        = 44,
 //! The label which published this work (if any)
 RMETA_LABEL              = 45,
 //! The catalog number of this work, album or compilation as used by the label.
 //! Needs RMETA_LABEL to be present, too.
 RMETA_LABELNO            = 46
} rmeta_type_t;

//! Modes for meta data updates
typedef enum {
 //! Set (or update) a meta date, overwriting old values
 RMETAMODE_SET                = 0,
 //! Add a meta date, not overwriting old values
 RMETAMODE_ADD                = 1,
 //! Delete a meta data for a given type
 RMETAMODE_DELETE             = 2,
 //! Delete all meta data from a stream
 RMETAMODE_CLEAR              = 3,
 //! Mark end of update sequence
 RMETAMODE_FINALIZE           = 4
} rmeta_mode_t;

#endif

//ll
