#!/usr/bin/python
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#
from __future__ import absolute_import, division, print_function


__metaclass__ = type


DOCUMENTATION = """
module: eos_config
author: Peter Sprygada (@privateip)
short_description: Manage Arista EOS configuration sections
description:
- Arista EOS configurations use a simple block indent file syntax for segmenting configuration
  into sections.  This module provides an implementation for working with EOS configuration
  sections in a deterministic way.  This module works with either CLI or eAPI transports.
version_added: 1.0.0
notes:
- Tested against Arista EOS 4.24.6F
- Abbreviated commands are NOT idempotent, see
  L(Network FAQ,../network/user_guide/faq.html#why-do-the-config-modules-always-return-changed-true-with-abbreviated-commands).
- To ensure idempotency and correct diff the configuration lines in the relevant module options should be similar to how they
  appear if present in the running configuration on device including the indentation.
options:
  lines:
    description:
    - The ordered set of commands that should be configured in the section. The commands
      must be the exact same commands as found in the device running-config as found in the
      device running-config to ensure idempotency and correct diff. Be sure
      to note the configuration command syntax as some commands are automatically
      modified by the device config parser.
    aliases:
    - commands
    type: list
    elements: str
  parents:
    description:
    - The ordered set of parents that uniquely identify the section or hierarchy the
      commands should be checked against.  If the parents argument is omitted, the
      commands are checked against the set of top level or global commands.
    type: list
    elements: str
  src:
    description:
    - The I(src) argument provides a path to the configuration file to load into the
      remote system.  The path can either be a full system path to the configuration
      file if the value starts with / or relative to the root of the implemented role
      or playbook. This argument is mutually exclusive with the I(lines) and I(parents)
      arguments. It can be a Jinja2 template as well. The configuration lines in the source
      file should be similar to how it will appear if present in the running-configuration
      (live switch config) of the device i ncluding the indentation to ensure idempotency
      and correct diff. Arista EOS device config has 3 spaces indentation.
    type: path
  before:
    description:
    - The ordered set of commands to push on to the command stack if a change needs
      to be made.  This allows the playbook designer the opportunity to perform configuration
      commands prior to pushing any changes without affecting how the set of commands
      are matched against the system.
    type: list
    elements: str
  after:
    description:
    - The ordered set of commands to append to the end of the command stack if a change
      needs to be made.  Just like with I(before) this allows the playbook designer
      to append a set of commands to be executed after the command set.
    type: list
    elements: str
  match:
    description:
    - Instructs the module on the way to perform the matching of the set of commands
      against the current device config.  If match is set to I(line), commands are
      matched line by line.  If match is set to I(strict), command lines are matched
      with respect to position.  If match is set to I(exact), command lines must be
      an equal match.  Finally, if match is set to I(none), the module will not attempt
      to compare the source configuration with the running configuration on the remote
      device.
    type: str
    default: line
    choices:
    - line
    - strict
    - exact
    - none
  replace:
    description:
    - Instructs the module on the way to perform the configuration on the device.  If
      the replace argument is set to I(line) then the modified lines are pushed to
      the device in configuration mode.  If the replace argument is set to I(block)
      then the entire command block is pushed to the device in configuration mode
      if any line is not correct.
    type: str
    default: line
    choices:
    - line
    - block
    - config
  backup:
    description:
    - This argument will cause the module to create a full backup of the current C(running-config)
      from the remote device before any changes are made. If the C(backup_options)
      value is not given, the backup file is written to the C(backup) folder in the
      playbook root directory or role root directory, if playbook is part of an ansible
      role. If the directory does not exist, it is created.
    type: bool
    default: false
  running_config:
    description:
    - The module, by default, will connect to the remote device and retrieve the current
      running-config to use as a base for comparing against the contents of source.  There
      are times when it is not desirable to have the task get the current running-config
      for every task in a playbook.  The I(running_config) argument allows the implementer
      to pass in the configuration to use as the base config for this module.
      The configuration lines for this option should be similar to how it will appear if present
      in the running-configuration of the device including the indentation to ensure idempotency
      and correct diff.
    type: str
    aliases:
    - config
  defaults:
    description:
    - The I(defaults) argument will influence how the running-config is collected
      from the device.  When the value is set to true, the command used to collect
      the running-config is append with the all keyword.  When the value is set to
      false, the command is issued without the all keyword
    type: bool
    default: false
  save_when:
    description:
    - When changes are made to the device running-configuration, the changes are not
      copied to non-volatile storage by default.  Using this argument will change
      that before.  If the argument is set to I(always), then the running-config will
      always be copied to the startup-config and the I(modified) flag will always
      be set to True.  If the argument is set to I(modified), then the running-config
      will only be copied to the startup-config if it has changed since the last save
      to startup-config.  If the argument is set to I(never), the running-config will
      never be copied to the startup-config. If the argument is set to I(changed),
      then the running-config will only be copied to the startup-config if the task
      has made a change. I(changed) was added in Ansible 2.5.
    default: never
    type: str
    choices:
    - always
    - never
    - modified
    - changed
  diff_against:
    description:
    - When using the C(ansible-playbook --diff) command line argument the module can
      generate diffs against different sources.
    - When this option is configure as I(startup), the module will return the diff
      of the running-config against the startup-config.
    - When this option is configured as I(intended), the module will return the diff
      of the running-config against the configuration provided in the C(intended_config)
      argument.
    - When this option is configured as I(running), the module will return the before
      and after diff of the running-config with respect to any changes made to the
      device configuration.
    - When this option is configured as C(session), the diff returned will be based
      on the configuration session.
    - When this option is configured as C(validate_config), the module will return before
      with the running-config before applying the intended config and after with the session
      config after applying the intended config to the session.
    default: session
    type: str
    choices:
    - startup
    - running
    - intended
    - session
    - validate_config
  diff_ignore_lines:
    description:
    - Use this argument to specify one or more lines that should be ignored during
      the diff.  This is used for lines in the configuration that are automatically
      updated by the system.  This argument takes a list of regular expressions or
      exact line matches.
    type: list
    elements: str
  intended_config:
    description:
    - The C(intended_config) provides the master configuration that the node should
      conform to and is used to check the final running-config against.   This argument
      will not modify any settings on the remote device and is strictly used to check
      the compliance of the current device's configuration against.  When specifying
      this argument, the task should also modify the C(diff_against) value and set
      it to I(intended). The configuration lines for this value should be similar to how it
      will appear if present in the running-configuration of the device including the indentation
      to ensure correct diff.
    type: str
  backup_options:
    description:
    - This is a dict object containing configurable options related to backup file
      path. The value of this option is read only when C(backup) is set to I(true),
      if C(backup) is set to I(no) this option will be silently ignored.
    suboptions:
      filename:
        description:
        - The filename to be used to store the backup configuration. If the filename
          is not given it will be generated based on the hostname, current time and
          date in format defined by <hostname>_config.<current-date>@<current-time>
        type: str
      dir_path:
        description:
        - This option provides the path ending with directory name in which the backup
          configuration file will be stored. If the directory does not exist it will
          be first created and the filename is either the value of C(filename) or
          default filename as described in C(filename) options description. If the
          path value is not given in that case a I(backup) directory will be created
          in the current working directory and backup configuration will be copied
          in C(filename) within I(backup) directory.
        type: path
    type: dict
"""
# noqa: E501

EXAMPLES = """
- name: configure top level settings
  arista.eos.eos_config:
    lines: hostname {{ inventory_hostname }}

- name: load an acl into the device
  arista.eos.eos_config:
    lines:
      - 10 permit ip host 192.0.2.1 any log
      - 20 permit ip host 192.0.2.2 any log
      - 30 permit ip host 192.0.2.3 any log
      - 40 permit ip host 192.0.2.4 any log
    parents: ip access-list test
    before: no ip access-list test
    replace: block

- name: load configuration from file
  arista.eos.eos_config:
    src: eos.cfg

- name: render a Jinja2 template onto an Arista switch
  arista.eos.eos_config:
    backup: true
    src: eos_template.j2

- name: diff the running config against a master config
  arista.eos.eos_config:
    diff_against: intended
    intended_config: "{{ lookup('file', 'master.cfg') }}"

- name: for idempotency, use full-form commands
  arista.eos.eos_config:
    lines:
      - shutdown
    parents: interface Ethernet1

- name: configurable backup path
  arista.eos.eos_config:
    src: eos_template.j2
    backup: true
    backup_options:
      filename: backup.cfg
      dir_path: /home/user
"""

RETURN = """
commands:
  description: The set of commands that will be pushed to the remote device
  returned: always
  type: list
  sample: ['hostname switch01', 'interface Ethernet1', 'no shutdown']
updates:
  description: The set of commands that will be pushed to the remote device
  returned: always
  type: list
  sample: ['hostname switch01', 'interface Ethernet1', 'no shutdown']
backup_path:
  description: The full path to the backup file
  returned: when backup is true
  type: str
  sample: /playbooks/ansible/backup/eos_config.2016-07-16@22:28:34
filename:
  description: The name of the backup file
  returned: when backup is true and filename is not specified in backup options
  type: str
  sample: eos_config.2016-07-16@22:28:34
shortname:
  description: The full path to the backup file excluding the timestamp
  returned: when backup is true and filename is not specified in backup options
  type: str
  sample: /playbooks/ansible/backup/eos_config
date:
  description: The date extracted from the backup file name
  returned: when backup is true
  type: str
  sample: "2016-07-16"
time:
  description: The time extracted from the backup file name
  returned: when backup is true
  type: str
  sample: "22:28:34"
"""
from ansible.module_utils._text import to_text
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import ConnectionError
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.config import (
    NetworkConfig,
    dumps,
)

from ansible_collections.arista.eos.plugins.module_utils.network.eos.eos import (
    get_config,
    get_connection,
    get_session_config,
    load_config,
    run_commands,
)


def get_candidate(module):
    candidate = ""
    if module.params["src"]:
        candidate = module.params["src"]
    elif module.params["lines"]:
        candidate_obj = NetworkConfig(indent=3)
        parents = module.params["parents"] or list()
        candidate_obj.add(module.params["lines"], parents=parents)
        candidate = dumps(candidate_obj, "raw")
    return candidate


def get_running_config(module, config=None, flags=None):
    contents = module.params["running_config"]
    if not contents:
        if config:
            contents = config
        else:
            contents = get_config(module, flags=flags)
    return contents


def save_config(module, result):
    result["changed"] = True
    if not module.check_mode:
        cmd = {
            "command": "copy running-config startup-config",
            "output": "text",
        }
        run_commands(module, [cmd])
    else:
        module.warn(
            "Skipping command `copy running-config startup-config` "
            "due to check_mode.  Configuration not copied to "
            "non-volatile storage",
        )


def main():
    """main entry point for module execution"""
    backup_spec = dict(filename=dict(), dir_path=dict(type="path"))
    argument_spec = dict(
        src=dict(type="path"),
        lines=dict(aliases=["commands"], type="list", elements="str"),
        parents=dict(type="list", elements="str"),
        before=dict(type="list", elements="str"),
        after=dict(type="list", elements="str"),
        match=dict(
            default="line",
            choices=["line", "strict", "exact", "none"],
        ),
        replace=dict(default="line", choices=["line", "block", "config"]),
        defaults=dict(type="bool", default=False),
        backup=dict(type="bool", default=False),
        backup_options=dict(type="dict", options=backup_spec),
        save_when=dict(
            choices=["always", "never", "modified", "changed"],
            default="never",
        ),
        diff_against=dict(
            choices=[
                "startup",
                "session",
                "intended",
                "running",
                "validate_config",
            ],
            default="session",
        ),
        diff_ignore_lines=dict(type="list", elements="str"),
        running_config=dict(aliases=["config"]),
        intended_config=dict(),
    )

    mutually_exclusive = [("lines", "src"), ("parents", "src")]

    required_if = [
        ("match", "strict", ["lines", "src"], True),
        ("match", "exact", ["lines", "src"], True),
        ("replace", "block", ["lines", "src"], True),
        ("replace", "config", ["src"]),
        ("diff_against", "intended", ["intended_config"]),
        ("diff_against", "validate_config", ["intended_config"]),
    ]

    module = AnsibleModule(
        argument_spec=argument_spec,
        mutually_exclusive=mutually_exclusive,
        required_if=required_if,
        supports_check_mode=True,
    )

    warnings = list()

    result = {"changed": False}
    if warnings:
        result["warnings"] = warnings

    diff_ignore_lines = module.params["diff_ignore_lines"]
    config = None
    contents = None
    flags = ["all"] if module.params["defaults"] else []
    connection = get_connection(module)
    # Refuse to diff_against: session if sessions are disabled
    if (
        module.params["diff_against"] in ["session", "validate_config"]
        and not connection.supports_sessions
    ):
        module.fail_json(
            msg="Cannot diff against sessions when sessions are disabled. Please change diff_against to another value",
        )

    if module.params["backup"] or (module._diff and module.params["diff_against"] == "running"):
        contents = get_config(module, flags=flags)
        config = NetworkConfig(indent=1, contents=contents)
        if module.params["backup"]:
            result["__backup__"] = contents

    if any((module.params["src"], module.params["lines"])):
        match = module.params["match"]
        replace = module.params["replace"]
        path = module.params["parents"]

        candidate = get_candidate(module)
        running = get_running_config(module, contents, flags=flags)

        try:
            response = connection.get_diff(
                candidate=candidate,
                running=running,
                diff_match=match,
                diff_ignore_lines=diff_ignore_lines,
                path=path,
                diff_replace=replace,
            )
        except ConnectionError as exc:
            module.fail_json(msg=to_text(exc, errors="surrogate_then_replace"))

        config_diff = response["config_diff"]
        if config_diff:
            commands = config_diff.split("\n")
            if module.params["before"]:
                commands[:0] = module.params["before"]

            if module.params["after"]:
                commands.extend(module.params["after"])

            result["commands"] = commands
            result["updates"] = commands

            replace = module.params["replace"] == "config"
            commit = not module.check_mode

            response = load_config(
                module,
                commands,
                replace=replace,
                commit=commit,
            )

            result["changed"] = True

            if module.params["diff_against"] == "session":
                if "diff" in response:
                    result["diff"] = {"prepared": response["diff"]}
                else:
                    result["changed"] = False

            if "session" in response:
                result["session"] = response["session"]

    running_config = module.params["running_config"]
    startup_config = None
    if module.params["save_when"] == "always":
        save_config(module, result)
    elif module.params["save_when"] == "modified":
        output = run_commands(
            module,
            [
                {"command": "show running-config", "output": "text"},
                {"command": "show startup-config", "output": "text"},
            ],
        )

        running_config = NetworkConfig(
            indent=3,
            contents=output[0],
            ignore_lines=diff_ignore_lines,
        )
        startup_config = NetworkConfig(
            indent=3,
            contents=output[1],
            ignore_lines=diff_ignore_lines,
        )

        if running_config.sha1 != startup_config.sha1:
            save_config(module, result)

    elif module.params["save_when"] == "changed" and result["changed"]:
        save_config(module, result)
    if module._diff:
        if not running_config:
            output = run_commands(
                module,
                {"command": "show running-config", "output": "text"},
            )
            contents = output[0]
        else:
            contents = running_config

        # recreate the object in order to process diff_ignore_lines
        running_config = NetworkConfig(
            indent=3,
            contents=contents,
            ignore_lines=diff_ignore_lines,
        )

        if module.params["diff_against"] == "running":
            if module.check_mode:
                module.warn(
                    "unable to perform diff against running-config due to check mode",
                )
                contents = None
            else:
                contents = config.config_text

        elif module.params["diff_against"] == "startup":
            if not startup_config:
                output = run_commands(
                    module,
                    {"command": "show startup-config", "output": "text"},
                )
                contents = output[0]
            else:
                contents = startup_config.config_text

        elif module.params["diff_against"] in ["intended", "validate_config"]:
            contents = module.params["intended_config"]

        if contents is not None:
            base_config = NetworkConfig(
                indent=3,
                contents=contents,
                ignore_lines=diff_ignore_lines,
            )

            if running_config.sha1 != base_config.sha1:
                before = ""
                after = ""
                if module.params["diff_against"] == "intended":
                    before = running_config
                    after = base_config
                elif module.params["diff_against"] in ("startup", "running"):
                    before = base_config
                    after = running_config
                elif module.params["diff_against"] == "validate_config":
                    before = running = get_running_config(
                        module,
                        None,
                        flags=flags,
                    )
                    replace = module.params["replace"] == "config"
                    after = get_session_config(
                        module,
                        contents.split("\n"),
                        replace=replace,
                        commit=False,
                    )

                result.update(
                    {
                        "changed": False,
                        "diff": {"before": str(before), "after": str(after)},
                    },
                )

    if result.get("changed") and any(
        (module.params["src"], module.params["lines"]),
    ):
        msg = (
            "To ensure idempotency and correct diff the input configuration lines should be"
            " similar to how they appear if present in"
            " the running configuration on device"
        )
        if module.params["src"]:
            msg += " including the indentation"
        if "warnings" in result:
            result["warnings"].append(msg)
        else:
            result["warnings"] = msg

    module.exit_json(**result)


if __name__ == "__main__":
    main()
