# DNS API Structural Info description

For GUI we need to show a list of options and basic description.
So instead of using comments describe a provider info in a special variable that later will be read and parsed.
The variable must be called like `dns_example_info` where the `example` is your provider code as in a file name.

The basic example:
```sh
# shellcheck disable=SC2034
dns_example_info='Example.org
Site: Example.org
Docs: github.com/acmesh-official/acme.sh/wiki/dnsapi2#dns_example
Options:
 Example_Token API Token
Issues: github.com/acmesh-official/acme.sh/issues/9000
Author: Your Name <your.name@example.com>
'
```

The format is both human-readable and easy to parse.

The `# shellcheck disable=SC2034` is needed to ignore an error that the var is not used.
The `dns_example_info` declares a variable with a multi line text.

At the first line is the title of of the API. If this is just a DNS provider then try use it's domain name.
Please write long domains in a CamelCase e.g. `CloudFlare.com`. This will help a user to distinguish providers in a list.

The `Site: Example.org` is a URL (without `https://`) to the provider's site. Sometimes it should be a dedicated DNS page:
```sh
dns_aws_info='Amazon AWS Route53 domain API
Site: docs.aws.amazon.com/route53/
```

The `Docs: github.com/acmesh-official/acme.sh/wiki/dnsapi2#dns_example` is a link to Wiki page with instructions. Some providers may have own wiki page e.g. `Lexicon`. The `https://` at beginning is stripped.

The `Options:` is a section with list of parameters (environment variables) to configure the provider. Each option should be declared on a separate line.
The line ` Example_Token API Token` starts with one space and declares a variable `Example_Token` with a title `API Token`. 
You may have multiple options and you can specify default value and show if the option is required:

    Options:
     VARIABLE1 Title for the option1.
     VARIABLE2 Title for the option2. Default "default value".
     VARIABLE3 Title for the option3. A long description to show on UI. Optional.

By default all the variables are mandatory. The `Optional.` is a special mark that the variable is not required.
The `Default "default value".` is a special mark and the value between the double quotes will be used as a default. Such variable are optional.
Only the first sentence will be a title so the the `A long description to show on UI` will be an extended description to show separately and it can be long and contain links.
The HTML is not allowed in a title or description.

A DNS provider may have alternative options like CloudFlare may use API KEY or API Token.
You can use a second section `OptionsAlt:` section. See [dns_cf.sh](https://github.com/acmesh-official/acme.sh/blob/master/dnsapi/dns_cf.sh).


The `Issues: github.com/acmesh-official/acme.sh/issues/9000` is a link to an support issue or a forum topic where you can reach an author to report a bug. At beginning you may omit it and we'll add ourselves once a PR is merged and the issue is created. The `https://` at beginning is stripped.

The `Author: Your Name <your.name@example.com>` is an optional field with a developer name and email (is not required). The author can be a GitHub username `@yourname`. You may use a link e.g. `Author: Alex Loban <https://github.com/LAV45>`. Multiple authors should be separated with a comma `,` e.g. `Author: Wolfgang Ebner, Sven Neubuaer`.

## Domain aliases

If a provider has multiple domains e.g. `AlibabaCloud.com` has an additional `Aliyun.com` then you can declare them in a dedicated field `Domains:`:
```sh
dns_1984hosting_info='1984.hosting
Domains: 1984.is
Site: 1984.hosting
'
```
So here the https://1984.hosting is the main page but the https://1984.is is also used.
So if user looking for the `1984.is` it may be confused it it see only the `1984.hosting`.
The `Domains:` may be also useful to search a provider in a drop-down list with autocomplete.


## Extended description
If the API is not for a specific provider but for a software (e.g. PowerDNS) or a protocol (e.g. nsupdate) then the title will be not a domain but a text.
Also you may add a description on the next line(s) staring with a space:
  
```sh
dns_acmedns_info='acme-dns Server API
 The acme-dns is a limited DNS server with RESTful API to handle ACME DNS challenges.
Site: github.com/joohoi/acme-dns
'
```

See the [initial commit](https://github.com/acmesh-official/acme.sh/commit/6b7b5caf54ea0b45508e158db3748d00f48672f2#diff-defdf80606e9123d8383965fa2bd6281a2567dc76c7d246a5244b41ec43429feR3) for mare samples. 

## Parsing

Here a script (dash and ash complaint) to generate a list of all infos:
```sh
#!/bin/ash
for f in ./dnsapi/dns_*.sh
do
  filename=$(basename -- "$f")
  dns_api="${filename%.*}"
  echo "$dns_api"
  dns_api_info_var="${dns_api}_info"
  # shellcheck source=./dnsapi/dns_*.sh
  . "$f"
  info=""
  eval info=\$$dns_api_info_var
  echo "$info"
done
```

Now execute it and stored result to info.txt file:

    sh ./dns_info.sh > info.txt

The resulted file has size of 40Kb bytes and gzipped it's about 10Kb.

See example of how to parse it in JavaScript: https://github.com/yurt-page/acmesh-parse-dnsapi-info