"""Place HLA calls into group for validation and presentation.

Uses p-groups with identical protein sequences in the antigen binding domains:

http://hla.alleles.org/alleles/p_groups.html

HLA allele nomenclature:

https://www.ebi.ac.uk/ipd/imgt/hla/
https://github.com/jrob119/IMGTHLA

HLA sequences are from the 1000 genomes build 38 reference:

ftp://ftp.1000genomes.ebi.ac.uk/vol1/ftp/technical/reference/GRCh38_reference_genome/

based on IMGT/HLA-3.18.0
"""
import toolz as tz

def hla_protein(name, data):
    group = tz.get_in([data["genome_build"], name], pgroups)
    if group:
        base = name.split("*")[0]
        group = "%s*%s" % (base, group)
    else:
        group = _from_name(name)
    return group

def _from_name(name):
    """Parse the HLA base name (group + protein) from a full name.

    Separates out synonymous and non-coding indicators.

    http://hla.alleles.org/nomenclature/naming.html
    """
    return ":".join(name.split(":")[:2])

pgroups = {}
pgroups["hg38"] = \
{'HLA-A*01:01:01:01': '01:01P',
 'HLA-A*01:01:01:02N': '',
 'HLA-A*01:01:38L': '01:01P',
 'HLA-A*01:02': '',
 'HLA-A*01:03': '',
 'HLA-A*01:04N': '',
 'HLA-A*01:09': '',
 'HLA-A*01:11N': '',
 'HLA-A*01:14': '',
 'HLA-A*01:16N': '',
 'HLA-A*01:20': '',
 'HLA-A*02:01:01:01': '02:01P',
 'HLA-A*02:01:01:02L': '02:01P',
 'HLA-A*02:01:01:03': '02:01P',
 'HLA-A*02:01:01:04': '02:01P',
 'HLA-A*02:02:01': '',
 'HLA-A*02:03:01': '02:03P',
 'HLA-A*02:03:03': '02:03P',
 'HLA-A*02:05:01': '02:05P',
 'HLA-A*02:06:01': '',
 'HLA-A*02:07:01': '02:07P',
 'HLA-A*02:10': '02:10P',
 'HLA-A*02:251': '',
 'HLA-A*02:259': '',
 'HLA-A*02:264': '02:03P',
 'HLA-A*02:265': '02:07P',
 'HLA-A*02:266': '02:01P',
 'HLA-A*02:269': '',
 'HLA-A*02:279': '',
 'HLA-A*02:32N': '',
 'HLA-A*02:376': '',
 'HLA-A*02:43N': '',
 'HLA-A*02:455': '02:01P',
 'HLA-A*02:48': '',
 'HLA-A*02:51': '',
 'HLA-A*02:533': '',
 'HLA-A*02:53N': '',
 'HLA-A*02:57': '',
 'HLA-A*02:60:01': '02:60P',
 'HLA-A*02:65': '02:65P',
 'HLA-A*02:68': '',
 'HLA-A*02:77': '',
 'HLA-A*02:81': '02:81P',
 'HLA-A*02:89': '02:01P',
 'HLA-A*02:95': '',
 'HLA-A*03:01:01:01': '03:01P',
 'HLA-A*03:01:01:02N': '',
 'HLA-A*03:01:01:03': '03:01P',
 'HLA-A*03:02:01': '03:02P',
 'HLA-A*03:11N': '',
 'HLA-A*03:21N': '',
 'HLA-A*03:36N': '',
 'HLA-A*11:01:01': '',
 'HLA-A*11:01:18': '11:01P',
 'HLA-A*11:02:01': '11:02P',
 'HLA-A*11:05': '',
 'HLA-A*11:110': '11:02P',
 'HLA-A*11:25': '',
 'HLA-A*11:50Q': '',
 'HLA-A*11:60': '',
 'HLA-A*11:69N': '',
 'HLA-A*11:74': '',
 'HLA-A*11:75': '',
 'HLA-A*11:77': '11:02P',
 'HLA-A*23:01:01': '23:01P',
 'HLA-A*23:09': '',
 'HLA-A*23:38N': '',
 'HLA-A*24:02:01:01': '24:02P',
 'HLA-A*24:02:01:02L': '24:02P',
 'HLA-A*24:02:01:03': '24:02P',
 'HLA-A*24:02:03Q': '24:02P',
 'HLA-A*24:02:10': '24:02P',
 'HLA-A*24:03:01': '24:03P',
 'HLA-A*24:07:01': '24:07P',
 'HLA-A*24:08': '',
 'HLA-A*24:09N': '',
 'HLA-A*24:10:01': '24:10P',
 'HLA-A*24:11N': '',
 'HLA-A*24:152': '',
 'HLA-A*24:20': '',
 'HLA-A*24:215': '',
 'HLA-A*24:61': '',
 'HLA-A*24:86N': '',
 'HLA-A*25:01:01': '25:01P',
 'HLA-A*26:01:01': '',
 'HLA-A*26:11N': '',
 'HLA-A*26:15': '',
 'HLA-A*26:50': '',
 'HLA-A*29:01:01:01': '29:01P',
 'HLA-A*29:01:01:02N': '',
 'HLA-A*29:02:01:01': '29:02P',
 'HLA-A*29:02:01:02': '29:02P',
 'HLA-A*29:46': '29:02P',
 'HLA-A*30:01:01': '30:01P',
 'HLA-A*30:02:01:01': '30:02P',
 'HLA-A*30:02:01:02': '30:02P',
 'HLA-A*30:04:01': '30:04P',
 'HLA-A*30:89': '',
 'HLA-A*31:01:02': '',
 'HLA-A*31:01:23': '31:01P',
 'HLA-A*31:04': '',
 'HLA-A*31:14N': '',
 'HLA-A*31:46': '31:01P',
 'HLA-A*32:01:01': '32:01P',
 'HLA-A*32:06': '',
 'HLA-A*33:01:01': '33:01P',
 'HLA-A*33:03:01': '33:03P',
 'HLA-A*33:07': '',
 'HLA-A*34:01:01': '34:01P',
 'HLA-A*34:02:01': '34:02P',
 'HLA-A*36:01': '',
 'HLA-A*43:01': '',
 'HLA-A*66:01:01': '66:01P',
 'HLA-A*66:17': '66:01P',
 'HLA-A*68:01:01:01': '68:01P',
 'HLA-A*68:01:01:02': '68:01P',
 'HLA-A*68:01:02:01': '68:01P',
 'HLA-A*68:01:02:02': '68:01P',
 'HLA-A*68:02:01:01': '68:02P',
 'HLA-A*68:02:01:02': '68:02P',
 'HLA-A*68:02:01:03': '68:02P',
 'HLA-A*68:02:02': '68:02P',
 'HLA-A*68:03:01': '68:03P',
 'HLA-A*68:08:01': '68:08P',
 'HLA-A*68:113': '',
 'HLA-A*68:17': '',
 'HLA-A*68:18N': '',
 'HLA-A*68:22': '',
 'HLA-A*68:71': '',
 'HLA-A*69:01': '',
 'HLA-A*74:01': '',
 'HLA-A*74:02:01:01': '74:01P',
 'HLA-A*74:02:01:02': '74:01P',
 'HLA-A*80:01:01:01': '80:01P',
 'HLA-A*80:01:01:02': '80:01P',
 'HLA-B*07:02:01': '07:02P',
 'HLA-B*07:05:01': '07:05P',
 'HLA-B*07:06': '07:05P',
 'HLA-B*07:156': '07:02P',
 'HLA-B*07:33:01': '07:33P',
 'HLA-B*07:41': '',
 'HLA-B*07:44': '07:02P',
 'HLA-B*07:50': '',
 'HLA-B*08:01:01': '08:01P',
 'HLA-B*08:08N': '',
 'HLA-B*08:132': '',
 'HLA-B*08:134': '',
 'HLA-B*08:19N': '',
 'HLA-B*08:20': '',
 'HLA-B*08:33': '',
 'HLA-B*08:79': '',
 'HLA-B*13:01:01': '13:01P',
 'HLA-B*13:02:01': '13:02P',
 'HLA-B*13:02:03': '13:02P',
 'HLA-B*13:02:09': '13:02P',
 'HLA-B*13:08': '',
 'HLA-B*13:15': '',
 'HLA-B*13:25': '',
 'HLA-B*14:01:01': '14:01P',
 'HLA-B*14:02:01': '',
 'HLA-B*14:07N': '',
 'HLA-B*15:01:01:01': '15:01P',
 'HLA-B*15:01:01:02N': '',
 'HLA-B*15:01:01:03': '15:01P',
 'HLA-B*15:02:01': '15:02P',
 'HLA-B*15:03:01': '',
 'HLA-B*15:04:01': '15:04P',
 'HLA-B*15:07:01': '15:07P',
 'HLA-B*15:108': '',
 'HLA-B*15:10:01': '15:10P',
 'HLA-B*15:11:01': '15:11P',
 'HLA-B*15:13:01': '15:13P',
 'HLA-B*15:16:01': '15:16P',
 'HLA-B*15:17:01:01': '15:17P',
 'HLA-B*15:17:01:02': '15:17P',
 'HLA-B*15:18:01': '15:18P',
 'HLA-B*15:220': '15:03P',
 'HLA-B*15:25:01': '15:25P',
 'HLA-B*15:27:01': '15:27P',
 'HLA-B*15:32:01': '15:32P',
 'HLA-B*15:42': '',
 'HLA-B*15:58': '',
 'HLA-B*15:66': '',
 'HLA-B*15:77': '',
 'HLA-B*15:83': '',
 'HLA-B*18:01:01:01': '18:01P',
 'HLA-B*18:01:01:02': '18:01P',
 'HLA-B*18:02': '',
 'HLA-B*18:03': '',
 'HLA-B*18:17N': '',
 'HLA-B*18:26': '',
 'HLA-B*18:94N': '',
 'HLA-B*27:04:01': '27:04P',
 'HLA-B*27:05:02': '27:05P',
 'HLA-B*27:05:18': '27:05P',
 'HLA-B*27:06': '',
 'HLA-B*27:07:01': '27:07P',
 'HLA-B*27:131': '',
 'HLA-B*27:24': '',
 'HLA-B*27:25': '',
 'HLA-B*27:32': '',
 'HLA-B*35:01:01:01': '35:01P',
 'HLA-B*35:01:01:02': '35:01P',
 'HLA-B*35:01:22': '35:01P',
 'HLA-B*35:02:01': '35:02P',
 'HLA-B*35:03:01': '35:03P',
 'HLA-B*35:05:01': '35:05P',
 'HLA-B*35:08:01': '35:08P',
 'HLA-B*35:14:02': '35:14P',
 'HLA-B*35:241': '35:01P',
 'HLA-B*35:41': '',
 'HLA-B*37:01:01': '37:01P',
 'HLA-B*37:01:05': '37:01P',
 'HLA-B*38:01:01': '38:01P',
 'HLA-B*38:02:01': '38:02P',
 'HLA-B*38:14': '',
 'HLA-B*39:01:01:01': '39:01P',
 'HLA-B*39:01:01:02L': '39:01P',
 'HLA-B*39:01:01:03': '39:01P',
 'HLA-B*39:01:03': '39:01P',
 'HLA-B*39:01:16': '39:01P',
 'HLA-B*39:01:21': '39:01P',
 'HLA-B*39:05:01': '39:05P',
 'HLA-B*39:06:02': '39:06P',
 'HLA-B*39:10:01': '39:10P',
 'HLA-B*39:13:02': '39:13P',
 'HLA-B*39:14': '',
 'HLA-B*39:34': '',
 'HLA-B*39:38Q': '',
 'HLA-B*40:01:01': '40:01P',
 'HLA-B*40:01:02': '40:01P',
 'HLA-B*40:02:01': '40:02P',
 'HLA-B*40:03': '40:03P',
 'HLA-B*40:06:01:01': '40:06P',
 'HLA-B*40:06:01:02': '40:06P',
 'HLA-B*40:10:01': '',
 'HLA-B*40:150': '40:01P',
 'HLA-B*40:40': '40:40P',
 'HLA-B*40:72:01': '40:72P',
 'HLA-B*40:79': '',
 'HLA-B*41:01:01': '41:01P',
 'HLA-B*41:02:01': '41:02P',
 'HLA-B*42:01:01': '42:01P',
 'HLA-B*42:02': '',
 'HLA-B*42:08': '',
 'HLA-B*44:02:01:01': '44:02P',
 'HLA-B*44:02:01:02S': '44:02P',
 'HLA-B*44:02:01:03': '44:02P',
 'HLA-B*44:02:17': '44:02P',
 'HLA-B*44:02:27': '44:02P',
 'HLA-B*44:03:01': '',
 'HLA-B*44:03:02': '44:03P',
 'HLA-B*44:04': '',
 'HLA-B*44:09': '',
 'HLA-B*44:138Q': '',
 'HLA-B*44:150': '',
 'HLA-B*44:23N': '',
 'HLA-B*44:26': '',
 'HLA-B*44:46': '',
 'HLA-B*44:49': '',
 'HLA-B*44:56N': '',
 'HLA-B*45:01:01': '45:01P',
 'HLA-B*45:04': '',
 'HLA-B*46:01:01': '46:01P',
 'HLA-B*46:01:05': '46:01P',
 'HLA-B*47:01:01:01': '47:01P',
 'HLA-B*47:01:01:02': '47:01P',
 'HLA-B*48:01:01': '48:01P',
 'HLA-B*48:03:01': '48:03P',
 'HLA-B*48:04': '',
 'HLA-B*48:08': '',
 'HLA-B*49:01:01': '49:01P',
 'HLA-B*49:32': '',
 'HLA-B*50:01:01': '50:01P',
 'HLA-B*51:01:01': '',
 'HLA-B*51:01:02': '51:01P',
 'HLA-B*51:02:01': '51:02P',
 'HLA-B*51:07:01': '51:07P',
 'HLA-B*51:42': '',
 'HLA-B*52:01:01:01': '52:01P',
 'HLA-B*52:01:01:02': '52:01P',
 'HLA-B*52:01:01:03': '52:01P',
 'HLA-B*52:01:02': '52:01P',
 'HLA-B*53:01:01': '53:01P',
 'HLA-B*53:11': '',
 'HLA-B*54:01:01': '54:01P',
 'HLA-B*54:18': '',
 'HLA-B*55:01:01': '55:01P',
 'HLA-B*55:01:03': '55:01P',
 'HLA-B*55:02:01': '',
 'HLA-B*55:12': '',
 'HLA-B*55:24': '',
 'HLA-B*55:48': '',
 'HLA-B*56:01:01': '',
 'HLA-B*56:03': '',
 'HLA-B*56:04': '',
 'HLA-B*57:01:01': '57:01P',
 'HLA-B*57:03:01': '57:03P',
 'HLA-B*57:06': '',
 'HLA-B*57:11': '',
 'HLA-B*57:29': '57:01P',
 'HLA-B*58:01:01': '',
 'HLA-B*58:31N': '',
 'HLA-B*59:01:01:01': '59:01P',
 'HLA-B*59:01:01:02': '59:01P',
 'HLA-B*67:01:01': '67:01P',
 'HLA-B*67:01:02': '67:01P',
 'HLA-B*67:02': '',
 'HLA-B*73:01': '',
 'HLA-B*78:01:01': '78:01P',
 'HLA-B*81:01': '81:01P',
 'HLA-B*82:02:01': '82:02P',
 'HLA-C*01:02:01': '01:02P',
 'HLA-C*01:02:11': '01:02P',
 'HLA-C*01:02:29': '01:02P',
 'HLA-C*01:02:30': '01:02P',
 'HLA-C*01:03': '01:03P',
 'HLA-C*01:06': '',
 'HLA-C*01:08': '',
 'HLA-C*01:14': '',
 'HLA-C*01:21': '',
 'HLA-C*01:30': '',
 'HLA-C*01:40': '01:02P',
 'HLA-C*02:02:02:01': '02:02P',
 'HLA-C*02:02:02:02': '02:02P',
 'HLA-C*02:10': '02:02P',
 'HLA-C*02:11': '',
 'HLA-C*02:16:02': '02:16P',
 'HLA-C*02:69': '02:02P',
 'HLA-C*02:85': '',
 'HLA-C*02:86': '',
 'HLA-C*02:87': '',
 'HLA-C*03:02:01': '03:02P',
 'HLA-C*03:02:02:01': '03:02P',
 'HLA-C*03:02:02:02': '03:02P',
 'HLA-C*03:02:02:03': '03:02P',
 'HLA-C*03:03:01': '03:03P',
 'HLA-C*03:04:01:01': '03:04P',
 'HLA-C*03:04:01:02': '03:04P',
 'HLA-C*03:04:02': '03:04P',
 'HLA-C*03:04:04': '03:04P',
 'HLA-C*03:05': '',
 'HLA-C*03:06': '',
 'HLA-C*03:100': '03:04P',
 'HLA-C*03:13:01': '03:13P',
 'HLA-C*03:20N': '',
 'HLA-C*03:219': '03:04P',
 'HLA-C*03:261': '',
 'HLA-C*03:40:01': '03:40P',
 'HLA-C*03:41:02': '03:41P',
 'HLA-C*03:46': '',
 'HLA-C*03:61': '',
 'HLA-C*04:01:01:01': '04:01P',
 'HLA-C*04:01:01:02': '04:01P',
 'HLA-C*04:01:01:03': '04:01P',
 'HLA-C*04:01:01:04': '04:01P',
 'HLA-C*04:01:01:05': '04:01P',
 'HLA-C*04:01:62': '04:01P',
 'HLA-C*04:03:01': '04:03P',
 'HLA-C*04:06': '',
 'HLA-C*04:09N': '',
 'HLA-C*04:128': '',
 'HLA-C*04:161': '04:01P',
 'HLA-C*04:177': '',
 'HLA-C*04:70': '',
 'HLA-C*04:71': '',
 'HLA-C*05:01:01:01': '05:01P',
 'HLA-C*05:01:01:02': '05:01P',
 'HLA-C*05:08': '',
 'HLA-C*05:09:01': '05:09P',
 'HLA-C*05:93': '05:01P',
 'HLA-C*06:02:01:01': '06:02P',
 'HLA-C*06:02:01:02': '06:02P',
 'HLA-C*06:02:01:03': '06:02P',
 'HLA-C*06:23': '',
 'HLA-C*06:24': '',
 'HLA-C*06:46N': '',
 'HLA-C*07:01:01:01': '07:01P',
 'HLA-C*07:01:01:02': '07:01P',
 'HLA-C*07:01:02': '07:01P',
 'HLA-C*07:01:19': '07:01P',
 'HLA-C*07:01:27': '07:01P',
 'HLA-C*07:01:45': '07:01P',
 'HLA-C*07:02:01:01': '07:02P',
 'HLA-C*07:02:01:02': '07:02P',
 'HLA-C*07:02:01:03': '07:02P',
 'HLA-C*07:02:01:04': '07:02P',
 'HLA-C*07:02:01:05': '07:02P',
 'HLA-C*07:02:05': '07:02P',
 'HLA-C*07:02:06': '07:02P',
 'HLA-C*07:02:64': '07:02P',
 'HLA-C*07:04:01': '07:04P',
 'HLA-C*07:04:02': '07:04P',
 'HLA-C*07:06': '07:01P',
 'HLA-C*07:149': '',
 'HLA-C*07:18': '07:01P',
 'HLA-C*07:19': '',
 'HLA-C*07:26': '',
 'HLA-C*07:30': '',
 'HLA-C*07:32N': '',
 'HLA-C*07:384': '',
 'HLA-C*07:385': '',
 'HLA-C*07:386': '',
 'HLA-C*07:391': '',
 'HLA-C*07:392': '',
 'HLA-C*07:49': '',
 'HLA-C*07:56:02': '07:56P',
 'HLA-C*07:66': '07:02P',
 'HLA-C*07:67': '',
 'HLA-C*08:01:01': '08:01P',
 'HLA-C*08:01:03': '08:01P',
 'HLA-C*08:02:01:01': '08:02P',
 'HLA-C*08:02:01:02': '08:02P',
 'HLA-C*08:03:01': '08:03P',
 'HLA-C*08:04:01': '08:04P',
 'HLA-C*08:112': '',
 'HLA-C*08:20': '08:01P',
 'HLA-C*08:21': '',
 'HLA-C*08:22': '08:01P',
 'HLA-C*08:24': '08:01P',
 'HLA-C*08:27': '',
 'HLA-C*08:36N': '',
 'HLA-C*08:40': '08:03P',
 'HLA-C*08:41': '',
 'HLA-C*08:62': '',
 'HLA-C*12:02:02': '12:02P',
 'HLA-C*12:03:01:01': '12:03P',
 'HLA-C*12:03:01:02': '12:03P',
 'HLA-C*12:08': '',
 'HLA-C*12:13': '',
 'HLA-C*12:19': '',
 'HLA-C*12:22': '',
 'HLA-C*12:99': '',
 'HLA-C*14:02:01': '14:02P',
 'HLA-C*14:03': '',
 'HLA-C*14:21N': '',
 'HLA-C*14:23': '14:02P',
 'HLA-C*15:02:01': '',
 'HLA-C*15:05:01': '15:05P',
 'HLA-C*15:05:02': '15:05P',
 'HLA-C*15:13': '15:02P',
 'HLA-C*15:16': '',
 'HLA-C*15:17': '',
 'HLA-C*15:96Q': '',
 'HLA-C*16:01:01': '',
 'HLA-C*16:02:01': '16:02P',
 'HLA-C*16:04:01': '16:04P',
 'HLA-C*17:01:01:01': '17:01P',
 'HLA-C*17:01:01:02': '17:01P',
 'HLA-C*17:01:01:03': '17:01P',
 'HLA-C*17:03': '17:01P',
 'HLA-C*18:01': '18:01P',
 'HLA-DQA1*01:01:02': '01:01P',
 'HLA-DQA1*01:02:01:01': '01:02P',
 'HLA-DQA1*01:02:01:02': '01:02P',
 'HLA-DQA1*01:02:01:03': '01:02P',
 'HLA-DQA1*01:02:01:04': '01:02P',
 'HLA-DQA1*01:03:01:01': '01:03P',
 'HLA-DQA1*01:03:01:02': '01:03P',
 'HLA-DQA1*01:04:01:01': '01:01P',
 'HLA-DQA1*01:04:01:02': '01:01P',
 'HLA-DQA1*01:05:01': '01:01P',
 'HLA-DQA1*01:07': '',
 'HLA-DQA1*01:10': '',
 'HLA-DQA1*01:11': '01:02P',
 'HLA-DQA1*02:01': '',
 'HLA-DQA1*03:01:01': '03:01P',
 'HLA-DQA1*03:02': '03:01P',
 'HLA-DQA1*03:03:01': '03:01P',
 'HLA-DQA1*04:01:02:01': '04:01P',
 'HLA-DQA1*04:01:02:02': '04:01P',
 'HLA-DQA1*04:02': '04:01P',
 'HLA-DQA1*05:01:01:01': '05:01P',
 'HLA-DQA1*05:01:01:02': '05:01P',
 'HLA-DQA1*05:03': '05:01P',
 'HLA-DQA1*05:05:01:01': '05:01P',
 'HLA-DQA1*05:05:01:02': '05:01P',
 'HLA-DQA1*05:05:01:03': '05:01P',
 'HLA-DQA1*05:11': '05:01P',
 'HLA-DQA1*06:01:01': '06:01P',
 'HLA-DQB1*02:01:01': '02:01P',
 'HLA-DQB1*02:02:01': '',
 'HLA-DQB1*03:01:01:01': '03:01P',
 'HLA-DQB1*03:01:01:02': '03:01P',
 'HLA-DQB1*03:01:01:03': '03:01P',
 'HLA-DQB1*03:02:01': '03:02P',
 'HLA-DQB1*03:03:02:01': '03:03P',
 'HLA-DQB1*03:03:02:02': '03:03P',
 'HLA-DQB1*03:03:02:03': '03:03P',
 'HLA-DQB1*03:05:01': '03:05P',
 'HLA-DQB1*05:01:01:01': '05:01P',
 'HLA-DQB1*05:01:01:02': '05:01P',
 'HLA-DQB1*05:03:01:01': '05:03P',
 'HLA-DQB1*05:03:01:02': '05:03P',
 'HLA-DQB1*06:01:01': '06:01P',
 'HLA-DQB1*06:02:01': '06:02P',
 'HLA-DQB1*06:03:01': '06:03P',
 'HLA-DQB1*06:09:01': '06:09P',
 'HLA-DRB1*01:01:01': '01:01P',
 'HLA-DRB1*01:02:01': '01:02P',
 'HLA-DRB1*03:01:01:01': '03:01P',
 'HLA-DRB1*03:01:01:02': '03:01P',
 'HLA-DRB1*04:03:01': '04:03P',
 'HLA-DRB1*07:01:01:01': '07:01P',
 'HLA-DRB1*07:01:01:02': '07:01P',
 'HLA-DRB1*08:03:02': '08:03P',
 'HLA-DRB1*09:21': '09:01P',
 'HLA-DRB1*10:01:01': '10:01P',
 'HLA-DRB1*11:01:01': '11:01P',
 'HLA-DRB1*11:01:02': '11:01P',
 'HLA-DRB1*11:04:01': '11:04P',
 'HLA-DRB1*12:01:01': '12:01P',
 'HLA-DRB1*12:17': '12:01P',
 'HLA-DRB1*13:01:01': '13:01P',
 'HLA-DRB1*13:02:01': '13:02P',
 'HLA-DRB1*14:05:01': '14:05P',
 'HLA-DRB1*14:54:01': '14:01P',
 'HLA-DRB1*15:01:01:01': '15:01P',
 'HLA-DRB1*15:01:01:02': '15:01P',
 'HLA-DRB1*15:01:01:03': '15:01P',
 'HLA-DRB1*15:01:01:04': '15:01P',
 'HLA-DRB1*15:02:01': '15:02P',
 'HLA-DRB1*15:03:01:01': '15:03P',
 'HLA-DRB1*15:03:01:02': '15:03P',
 'HLA-DRB1*16:02:01': '16:02P'}
